(module lshift/jexplorer
    (jexplore)
  (import s2j)
  (import type-system)
  (import srfi-1)
  (import lshift/error)

  (define-generic-java-methods
    set-accessible)

  (define *back* '())
  (define *forward* '())
  (define *current* #f)
  (define *field-list* #f)

  (define (set-current! new-current)
    (set! *current* new-current)
    (set! *field-list* #f))

  (define (reset)
    (set! *back* '())
    (set! *forward* '())
    (set-current! #f))

  (define (visit o)
    (when *current*
      (set! *back* (cons *current* *back*)))
    (set-current! o)
    (set! *forward* '())
    (dump))

  (define (back)
    (when (pair? *back*)
      (set! *forward* (cons *current* *forward*))
      (set-current! (car *back*))
      (set! *back* (cdr *back*))
      (dump)))

  (define (forward)
    (when (pair? *forward*)
      (set! *back* (cons *current* *back*))
      (set-current! (car *forward*))
      (set! *forward* (cdr *forward*))
      (dump)))

  (define (dump-java-object x include-statics)
    (write x)
    (newline)
    (for-each (lambda (class-and-fields)
		(write (java-class-name (car class-and-fields)))
		(newline)
		(for-each (lambda (entry)
			    (if (or include-statics
				    (not (memq 'static (java-field-flags (fourth entry)))))
				(begin
				  (display "  ")
				  (display (first entry))
				  (display " => ")
				  (if (java-null? x)
				      (display "#<nothing, since this is a null object>")
				      (write ((second entry) x)))
				  (newline))))
			  (cdr class-and-fields)))
	      (all-fields-of x)))

  (define (dump)
    (if (java-object? *current*)
	(dump-java-object *current* #f)
	(pretty-print *current*))
    (newline))

  (define (dump-statics)
    (if (java-object? *current*)
	(dump-java-object *current* #t)
	(pretty-print *current*))
    (newline))

  (define (print)
    (pretty-print *current*)
    (newline))

  (define (all-fields-of/nocache x)
    (let ((classes (java-class-precedence-list (type-of x))))
      (map (lambda (class)
	     (cons class (filter-map (lambda (field)
				       (set-accessible field (->jboolean #t))
				       (list (java-field-name field)
					     (java-field-accessor-procedure field)
					     (java-field-modifier-procedure field)
					     field))
				     (java-class-declared-fields class))))
	   (reverse classes))))

  (define (all-fields-of x)
    (if (eq? x *current*)
	(begin (when (not *field-list*) (set! *field-list* (all-fields-of/nocache *current*)))
	       *field-list*)
	(all-fields-of/nocache x)))

  (define (all-fields)
    (all-fields-of *current*))

  (define (fields)
    (pretty-print (all-fields))
    (newline))

  (define (find-map f l)
    (let search ((l l))
      (cond
       ((null? l) #f)
       ((f (car l)))
       (else (search (cdr l))))))

  (define (eval-new-value expr)
    (let* ((v0 (eval expr)))
      (cond
       ((string? v0) (->jstring v0))
       ((number? v0) (->jint v0))
       ((boolean? v0) (->jboolean v0))
       (else v0))))

  (define (field n . maybe-new-value)
    (cond
     ((find-map (lambda (class-and-fields)
		  (find-map (lambda (entry)
			      (and (eq? (first entry) n) entry))
			    (cdr class-and-fields)))
		(all-fields))
      => (lambda (entry)
	   (if (null? maybe-new-value)
	       (visit ((second entry) *current*))
	       ((third entry) *current* (eval-new-value (car maybe-new-value))))))
     (else (display "\nUnknown field name\n"))))

  (define (abort)
    (display "\nBye! (Aborting with error)\n")
    (error 'jexplorer-abort))

  (define (help)
    (display "\nJExplore, a trivial Java object explorer\nCommands:\n")
    (for-each (lambda (command) (display "  - ")(display command)(newline))
	      (append (map car (force commands))
		      '(quit q)))
    (newline))

  (define (scheme-repl . maybe-exprs)
    (call-with-current-continuation
     (lambda (escape)
       (let* ((e (sisc-initial-environment))
	      (ep (lambda (expr)
		    (with/fc
			(lambda (m k) (debug-dump-exception m k))
		      (lambda () (pretty-print (eval expr e))))
		    (newline))))
	 (for-each (lambda (binding) (putprop (car binding) e (cadr binding)))
		   `((current ,(lambda () *current*))
		     (reset ,reset)
		     (visit ,visit)
		     (back ,back)
		     (forward ,forward)
		     (dump ,dump)
		     (exit ,(lambda args (escape args)))))
	 (if (null? maybe-exprs)
	     (let loop ()
	       (pretty-print `(JEXPLORE-SCHEME))
	       (newline)
	       (display "jexplore-scheme> ")
	       (let ((expr (read)))
		 (cond ((eof-object? expr))
		       (else (ep expr)
			     (loop)))))
	     (for-each ep maybe-exprs))))))

  (define (length-cmd)
    (pretty-print (java-array-length *current*))
    (newline))

  (define (at-cmd n . maybe-new-value)
    (if (null? maybe-new-value)
	(visit (java-array-ref *current* n))
	(java-array-set! *current* n (eval-new-value (car maybe-new-value)))))

  (define commands
    (delay
      `((dump ,dump) (d ,dump)
	(dump-statics ,dump-statics) (ds ,dump-statics)
	(print ,print) (p ,print)
	(back ,back) (b ,back)
	(fields ,fields)
	(field ,field)
	(forward ,forward) (f ,forward)
	(reset ,reset)
	(abort ,abort)
	(help ,help) (? ,help)
	(length ,length-cmd) (@ ,at-cmd)
	(! ,scheme-repl)
	)))

  (define (prompt-and-read)
    (pretty-print `(JEXPLORE (back ,(length *back*))
			     (forward ,(length *forward*))))
    (newline)
    (display "jexplore> ")
    (read))

  (define (jexplore . maybe-new-object)
    (when (pair? maybe-new-object)
      (reset)
      (visit (car maybe-new-object)))
    (repl))

  (define (run-command command args)
    (cond
     ((memq command '(quit q)) (display "\nBye!\n"))
     ((assq command (force commands)) => (lambda (cell)
					   (apply (cadr cell) args)
					   (repl)))
     (else (display "\nUnknown jexplore command\n")
	   (repl))))

  (define (repl)
    (let loop ((command (prompt-and-read)))
      (cond
       ((eof-object? command))
       ((symbol? command) (run-command command '()))
       ((pair? command) (run-command (car command) (cdr command)))
       (else (display "\nBad jexplore command\n")
	     (repl)))))
  )
