(module lshift/struct-validation
    ((define-checked-struct report-checked-struct-error)
     optional
     all-of
     string-is-length
     string-less-than-length
     string-of-digits?
     string-from)

  (import srfi-13)
  (import srfi-26)

  ;; Some procedures for checking the validity of values and structs,
  ;; and syntax for defining record types at the same time as the validation predicate.

  (define (report-checked-struct-error struct-name field-name field-description field-value)
    (error (string-append "Checked structure failure: Field "(symbol->string field-name)
			  " of structure "(symbol->string struct-name)
			  " has validation predicate \""
			  (call-with-output-string
			   (lambda (port) (write field-description port)))"\", but value "
			  (call-with-output-string
			   (lambda (port) (write field-value port))))))

  (define-syntax define-checked-struct
    (lambda (x)
      (syntax-case x ()
	((_ name ((field field-predicate?) ...))
	 (let* ((name-syntax (syntax name))
		(name-string (symbol->string (syntax-object->datum
					      name-syntax)))
		(wrap-symbol (lambda (pre stx post)
			       (datum->syntax-object stx
						     (string->symbol
						      (string-append pre
								     (symbol->string
								      (syntax-object->datum stx))
								     post))))))
	   (with-syntax ((make-name (wrap-symbol "make-" name-syntax "*"))
			 (checked-make-name (wrap-symbol "make-" name-syntax ""))
			 (name? (wrap-symbol "" name-syntax "?"))
			 ((field-getter ...)
			  (map (lambda (f)
				 (wrap-symbol
				  (string-append name-string "-")
				  (datum->syntax-object name-syntax f)
				  ""))
			       (syntax-object->datum (syntax (field ...)))))
			 ((field-setter ...)
			  (map (lambda (f)
				 (wrap-symbol
				  (string-append "set-" name-string "-")
				  (datum->syntax-object name-syntax f)
				  "!"))
			       (syntax-object->datum (syntax (field ...))))))
			(syntax
			 (begin
			   (define-record-type name (make-name field ...)
			     name?
			     (field field-getter field-setter) ...)
			   (define checked-make-name
			     (lambda (field ...)
			       (if (not (field-predicate? field))
				   (report-checked-struct-error 'name
								'field
								(quote field-predicate?)
								field))
			       ...
			       (make-name field ...)))))))))))

  ;; Validation predicates and predicate-makers.

  ;; Either an empty string or something fulfilling the predicate given
  (define (optional pred)
    (lambda (val) (or (equal? "" val) (pred val))))

  (define-syntax all-of
    (syntax-rules ()
      ((_ pred ...)
       (lambda (val) (and (pred val) ...)))))

  ;; The following require srfi-13 for string procedures

  ;; Assumes it's given a string: usual pattern is (all-of string? (string-is-length 10))
  (define (string-is-length len)
    (lambda (val) (equal? (string-length val) len)))

  (define (string-less-than-length len)
    (lambda (val) (< (string-length val) len)))

  (define (string-of-digits? val)
    (and (string? val) (string-every char-numeric? val)))

  ;; cut from srfi-26
  (define (string-from strings)
    (cut member <> strings))

  )