(module unit-test
    (test-case
     test-suite
     assert-equivalence
     assert-true
     assert-false
     assert-error
     assert-error-equivalence
     error-record-message-=?
     error-record-message-contains?
     call-with-new-test-run
     )

  (import srfi-13)
  (import string-io)
  (import debugging)

  ;; Lightweight procedures to aid unit tests.
  ;; TODO: Port Noel's SchemeUnit over to SISC

  (define (assert-equivalence operator expected actual)
    (if (operator expected actual)
	#t
	(error (format "Assert failed: arguments are not equivalent. Got ~a  Expected ~a"
		       actual
		       expected))))

  (define (assert-true thunk)
    (if (thunk)
	#t
	(error "Assert failed: thunk returned false")))


  (define (assert-false thunk)
    (if (thunk)
	(error "Assert failed: thunk returned true"))
    #t)

  (define (assert-error thunk)
    (let ((threw-error #f))
      (with/fc 
	  (lambda (error-record error-k)
	    (set! threw-error #t))
	(lambda ()
	  (thunk)
	  (set! threw-error #f)))
      (if (not threw-error)
	  (error "Assert failed: thunk did not raise an error"))))

  (define (assert-error-equivalence operator expected thunk)
    (let ((thrown-error #f))
      (with/fc
	  (lambda (error-record error-k)
	    (set! thrown-error (list error-record)))
	thunk)
      (cond
       ((not thrown-error) (error "Assert failed: thunk did not raise an error"))
       ((not (operator expected (car thrown-error)))
	(error (format "Assert failed: errors are not equivalent. Got ~a  Expected ~a"
		       (car thrown-error)
		       expected)))
       (else #t))))

  (define (error-record-message-=? expected error-record)
    (cond
     ((assq 'message error-record) => (lambda (actual) (equal? expected (cdr actual))))
     (else #f)))

  (define (error-record-message-contains? expected error-record)
    (cond
     ((assq 'message error-record) => (lambda (actual) (string-contains (cdr actual) expected)))
     (else #f)))

  (define current-test-run (make-parameter #f))

  (define (call-with-current-test-run proc)
    (let ((test-run (current-test-run)))
      (if (not test-run)
	  (error "Not inside a test-run")
	  (proc test-run))))

  (define (record-failure! test-name error-record error-k)
    (call-with-current-test-run
     (lambda (test-run)
       (set-car! test-run (cons (list test-name error-record error-k) (car test-run))))))

  (define (record-success! test-name time-taken)
    (call-with-current-test-run
     (lambda (test-run)
       (set-car! (cdr test-run) (cons (list test-name time-taken) (cadr test-run))))))

  (define (test-case test-name test-thunk)
    (lambda ()
      (display (format "Running ~a... " test-name))
      (with/fc
	  (lambda (error-record error-k)
	    (display "failed\n")
	    (record-failure! test-name error-record error-k))
	(lambda ()
	  (let* ((t (time (test-thunk)))
		 (time-taken (cadr t)))
	    (display (format "success ~a\n" time-taken))
	    (record-success! test-name time-taken))))))

  (define (test-suite suite-name test-cases)
    (display (format "Running test suite ~a...\n" suite-name))
    (for-each
     (lambda (test-case)
       (apply test-case '()))
     test-cases)
    (display (format "Completed test suite ~a.\n" suite-name)))

  (define (report-on-test-run test-run)
    (let ((failures (car test-run))
	  (successes (cadr test-run)))
      (display "----------------------------------------") (newline)
      (display "Test report:") (newline)
      (if (not (null? failures))
	  (begin
	    (newline)
	    (display "Failed tests:") (newline)
	    (for-each
	     (lambda (failure)
	       (let ((test-name (car failure))
		     (error-record (cadr failure))
		     (error-k (caddr failure)))
		 (display "----------------------------------------") (newline)
		 (for-each display (list "Test: "test-name)) (newline)
		 (newline)
		 (for-each (lambda (pair)
			     (for-each display (list (car pair)": "(cdr pair)))
			     (newline))
			   error-record)
		 (newline)
		 (display "STACK TRACE") (newline)
		 (newline)
		 (display (with-output-to-string (lambda () (print-stack-trace error-k))))))
	     (reverse failures))))
      (display "----------------------------------------") (newline)
      (for-each display (list (length successes)" tests passed.")) (newline)
      (for-each display (list (length failures)" tests failed.")) (newline)
      (display "----------------------------------------") (newline)
      )
    test-run)

  (define (call-with-new-test-run thunk)
    (parameterize ((current-test-run (list '() '())))
      (thunk)
      (report-on-test-run (current-test-run))))

  )