;; Using XSLT to generate UI

(module lshift/xslt-ui
    (
     ;; Core XSLT transformation-and-transmission API
     send-xslt/back
     send-xslt/finish
     send-xslt/suspend
     send-xslt*

     ;; Validated field record type
     <validated-field>
     make-validated-field
     validated-field?

     validated-field-name
     validated-field-content
     validated-field-content1
     validated-field-error
     validated-field-content-source
     validated-field-on-submit
     validated-field-metadata

     set-validated-field-content!
     set-validated-field-error!

     xslt-ui-dump-request-parameters

     ;; Validated field transformation-and-I/O API
     send-xslt-ui
     send-xslt-ui/back
     send-xslt-ui/finish
     template->sxml
     template->plain-sxml

     ;; Premade validated-field implementations
     readonly-vf
     make-xslt-ui-template
     get-template-vf
     )

  (import srfi-9)
  (import srfi-13)
  (import srfi-26)
  (import utils/servlet-tools)
  (import utils/sxml-tools)
  (import siscweb/contcentric)
  (import siscweb/response)
  (import siscweb/request)
  (import s2j)
  (import lshift/xslt)
  (import lshift/common)
  (import lshift/error)

  (define-generic-java-methods
    set-content-type
    get-parameter-names
    get-parameter-values
    get-writer)

  (define-record-type <validated-field>
    (make-validated-field* name content error content-source on-submit metadata)
    validated-field?
    (name		validated-field-name)
    (content		validated-field-content		set-validated-field-content!)
    (error		validated-field-error		set-validated-field-error!)
    (content-source	validated-field-content-source)
    (on-submit		validated-field-on-submit)
    (metadata		validated-field-metadata))

  (define (validated-field-content1 vf)
    (let ((content (validated-field-content vf)))
      (if (null? content)
	  ""
	  (car content))))

  (define (make-validated-field name
				content
				content-source-or-false
				on-submit-or-false
				. maybe-metadata)
    (make-validated-field* name
			   content
			   ""
			   content-source-or-false
			   on-submit-or-false
			   (if (pair? maybe-metadata)
			       (car maybe-metadata)
			       '())))

  (define (base-uri-parameter req)
    `("base-uri"
      ,(->string (absolute-context-path req))))

  (define (send-xslt-action action stylesheets value)
    (action
     (lambda ()
       (send-xslt* (current-response)
		   (list (base-uri-parameter (current-request)))
		   stylesheets
		   value))))

  (define (send-xslt/back stylesheets value)
    (send-xslt-action send/back stylesheets value))
  
  (define (send-xslt/finish stylesheets value)
    (send-xslt-action send/finish stylesheets value))
  
  (define (send-xslt/suspend stylesheets value)
    (send/suspend
     (lambda (k-url k-resume)
       (send-xslt* (current-response)
		   `(,(base-uri-parameter (current-request))
		     ("k-url" ,k-url))
		   stylesheets
		   value))))

  (define (send-xslt* res extra-parameters stylesheets value)
    (if (null? stylesheets)
	(error+ "No stylesheets specified")
	(let loop ((stylesheets stylesheets)
		   (value (->dom value)))
	  (let* ((sheet (first (car stylesheets)))
		 (parameters (append extra-parameters
				     (second (car stylesheets))))
		 (transform-current (lambda () (xslt-transform/dom sheet value parameters)))
		 (rest (cdr stylesheets)))
	    (if (pair? rest)
		;; We want all but the last one to be intermediate
		(loop rest (transform-current))
		(case rest
		  (else
		   ;;(set-content-type res (->jstring "text/html"))
		   ;;(set-content-type res (->jstring "text/xml"))
		   ;;(set-content-type res (->jstring "application/xhtml+xml"))
		   (do-xslt-transform-to-writer sheet
						value
						(get-writer res)
						parameters))))))))

  (define (update-validated-field-content! vf)
    (let ((content-source (validated-field-content-source vf)))
      (when content-source
	(set-validated-field-content! vf (content-source)))))

  (define (validated-field->sxml namer vf)
    (let ((name (namer vf)))
      `(,(validated-field-name vf) (field (name ,name)
					  ,@(map (lambda (value) `(value ,value))
						 (validated-field-content vf))
					  (errormessage ,(validated-field-error vf))
					  ,@(validated-field-metadata vf)))))

  (define (walk-template visit-vf template-value)
    (let walk ((o template-value))
      (cond
	((pair? o)		(cons (walk (car o)) (walk (cdr o))))
	((validated-field? o)	(visit-vf o))
	(else			o))))

  (define (update-template-vf-content! o)
    (cond
     ((pair? o)			(begin (update-template-vf-content! (car o))
				       (update-template-vf-content! (cdr o))))
     ((validated-field? o)	(update-validated-field-content! o))
     (else			'we-do-not-care)))

  (define (friendly-namer)
    (let ((counter 0))
      (lambda (vf)
	(set! counter (+ counter 1))
	(string-append "X" (number->string counter) "_"
		       (symbol->string (validated-field-name vf))))))

  (define (template->sxml template)
    (update-template-vf-content! template)
    (let ((new-field-name (friendly-namer)))
      (walk-template (cut validated-field->sxml new-field-name <>)
		     template)))

  (define (template->plain-sxml template)
    (walk-template (lambda (vf) `(,(validated-field-name vf)
				  ,@(begin (update-validated-field-content! vf)
					   (validated-field-content vf))))
		   template))

  ;; If we are not returning, the only thing we need to do is transform the template given;
  ;; no validation will be performed, so we don't need to keep track of the fields.
  (define (send-xslt-ui/final-action action stylesheets template)
    (action stylesheets (template->sxml template)))
  
  (define (send-xslt-ui/back stylesheets template)
    (send-xslt-ui/final-action send-xslt/back stylesheets template))

  (define (send-xslt-ui/finish stylesheets template)
    (send-xslt-ui/final-action send-xslt/finish stylesheets template))

  (define (request-parameters->alist req)
    (let ((parameter-names (enumeration->list (get-parameter-names req) ->string)))
      (map (lambda (name)
	     (cons name (get-all-parameter-values req name)))
	   parameter-names)))

  (define (get-all-parameter-values req name)
    (let ((jvalues (get-parameter-values req (->jstring name))))
      (if (java-null? jvalues)
	  '()
	  (map ->string (->list jvalues)))))

  (define (process-submitted-vf! req name vf action)
    (if (null? (get-all-parameter-values req (string-append "_" name)))
	#t ;; no magic "present" field for this name, so ignore and assume OK
	(let* ((parameter-values (get-all-parameter-values req name)))
	  (call-with-current-continuation
	   (lambda (escape)
	     (define (die error-message)
	       (set-validated-field-content! vf parameter-values)
	       (set-validated-field-error! vf error-message)
	       (escape #f))
	     (let* ((validator! (or (validated-field-on-submit vf)
				    (lambda (_1 vals _2) vals)))
		    (new-values (validator! action parameter-values die)))
	       (when new-values
		 (set-validated-field-content! vf new-values)
		 (set-validated-field-error! vf ""))
	       #t))))))

  (define xslt-ui-dump-request-parameters (make-parameter #f))

  ;; There's an implied /suspend
  (define (send-xslt-ui stylesheets template-value . extra-validation-procedures)
    (update-template-vf-content! template-value)
    (let loop ()
      (let* ((vf-stack '())
	     (push-vf! (lambda (name vf) (set! vf-stack (cons (cons name vf) vf-stack))))
	     (finish-vf-stack (lambda () (reverse vf-stack)))
	     (new-field-name* (friendly-namer))
	     (new-field-name (lambda (vf)
			       (let ((name (new-field-name* vf)))
				 (push-vf! name vf)
				 name))))
	(let* ((value (walk-template (cut validated-field->sxml new-field-name <>)
				     template-value))
	       (req (send-xslt/suspend stylesheets value))
	       (_ (when (xslt-ui-dump-request-parameters)
		    (pretty-print (request-parameters->alist req))
		    (newline)))
	       (action (let ((actions (get-all-parameter-values req "X_action")))
			 (cond
			  ((null? actions) #f)
			  ((pair? (cdr actions)) (error+ "send-xslt-ui: too many actions supplied"
							 actions))
			  (else (string->symbol
				 (string-map (lambda (c)
					       (if (char-whitespace? c)
						   #\-
						   c))
					     (string-downcase (car actions)))))))))
	  (for-each (lambda (entry)
		      (if (not (process-submitted-vf! req (car entry) (cdr entry) action))
			  (set! action #f)))
		    (finish-vf-stack))
	  (for-each (lambda (extra-validation-procedure)
		      (call-with-current-continuation
		       (lambda (escape)
			 (define (die)
			   (set! action #f)
			   (escape 'dummy))
			 (extra-validation-procedure action die))))
		    extra-validation-procedures)
	  (or action
	      (loop))))))

  (define (readonly-vf name val . maybe-metadata)
    (apply make-validated-field name #f (lambda () (list val)) #f maybe-metadata))

  (define (make-xslt-ui-template . items)
    (cons 'data items))

  (define (get-template-vf template path field-name)
    (with-sxml template
      (let ((nodes (select (string-append "/data/" path "/node()"))))
	(or (find (lambda (node) (and (validated-field? node)
				      (eq? (validated-field-name node) field-name)))
		  nodes)
	    (error+ "get-template-vf: no nodes matching path" template path)))))
  )

(module lshift/field-parameters
    (get-field-parameters
     field-updater-from-parameters
     apply-field-converters ;; necessary to export? the module below uses it too
     build-field-parameters
     validated-field/parameters)

  (import pattern-matching)
  (import srfi-26)
  (import lshift/error)
  (import lshift/xslt-ui)
    
  (define (parameter-present? name field-parameters)
    (cond
      ((assq name field-parameters) #t)
      (else #f)))
  
  (define (get-field-parameters name field-parameters)
    (cond
      ((assq name field-parameters) => cdr)
      (else '())))

  (define (apply-field-converters pre-name mid post-name field-parameters)
    (lambda (action vals0 die)
      (fold (lambda (checker-converter vals) (checker-converter action vals die))
	    vals0
	    (append (get-field-parameters pre-name field-parameters)
		    (if mid (list mid) '())
		    (get-field-parameters post-name field-parameters)))))

  ;; ('a -> undefined) x (list string -> list 'a) x parameters -> on-submit-procedure
  (define (field-updater-from-parameters setter inbound-converter field-parameters)
    (lambda (action supplied-vals die)
      (let ((new-vals
	     ((apply-field-converters 'input-pre
				      inbound-converter
				      'input-post
				      field-parameters)
	      action supplied-vals die)))
	(setter new-vals)
	supplied-vals)))

  (define (build-options-metadata opts)
    `(options ,@(map (lambda (entry)
		       (if (string? entry)
			   `(option (value ,entry) (label ,entry))
			   `(option (value ,(second entry)) (label ,(first entry)))))
		     opts)))

  (define build-field-parameters
    (let ()
      (define (trim-p action vals die)
	(map string-trim-both vals))

      (define (strip-whitespace action vals die)
	(map (lambda (str) (string-filter (lambda (ch) (not (char-whitespace? ch))) str))
	     vals))

      (define (required-nonempty action vals die)
	(if (any (lambda (p) (string=? p "")) vals)
	    (die "This field is required")
	    vals))

      (define (required1 action vals die)
	(if (or (null? vals)
		(not (null? (cdr vals))))
	    (die "Please choose one of the available options")
	    vals))

      (define (requiredn action vals die)
	(if (null? vals)
	    (die "Please choose one or more of the available options")
	    vals))

      (define (digits-only action vals die)
	(if (any (lambda (str) (string-any (lambda (ch) (not (char-numeric? ch))) str)) vals)
	    (die "This field may only contain digits 0-9")
	    vals))

      (define (valid-email action vals die)
	(map (lambda (p)
	       (cond
		 ((string=? p "") p)
		 ;; TODO: Add more email-address validation
		 ((string-index p #\@) p)
		 (else (die "This field must either be empty, or contain a valid email address"))))
	     vals))

      (define (string-length-compares op desc n)
	(lambda (action vals die)
	  (map (lambda (p)
		 (if (op (string-length p) n)
		     (die (string-append "This field may not be "desc" than "(number->string n)" characters"))
		     p))
	       vals)))

      (define (parse-boolean-ui-param action vals die)
	(if (null? vals)
	    (list "0") ;; off.
	    (list "1"))) ;; on.

      (define (render-boolean-ui-param action vals die)
	(if (string=? (car vals) "0")
	    '()	;; off.
	    '("on"))) ;; on.

      (define (numeric-ineq operator desc n)
	(lambda (action vals die)
	  (map (lambda (p)
		 (if (not (or (string=? p "")
			      (and-let* ((pnum (string->number p)))
				(operator pnum n))))
		     (die (string-append "This must be a number "desc" "(number->string n)))
		     p))
	       vals)))
    
      (lambda (field-entry)
	(define metadata '())
	(define input-pre '())
	(define input-post '())
	(define output-pre '())
	(define output-post '())
	(let-syntax ((push! (syntax-rules () ((_ l v) (set! l (cons v l)))))
		     (finish (syntax-rules () ((_ l) (if (null? l)
							 '()
							 `((l ,@(reverse l))))))))
	  (define (process-clause clause)
	    (match clause
	      (trim		(push! input-pre trim-p))
	      (strip-whitespace	(push! input-pre strip-whitespace))
	      (required		(begin (push! metadata '(required))
				       (push! input-pre required-nonempty)))
	      (required1	(begin (push! metadata '(required))
				       (push! input-pre required1)))
	      (requiredn	(begin (push! metadata '(required))
				       (push! input-pre requiredn)))
	      (valid-email	(push! input-pre valid-email))
	      (boolean		(begin (push! input-pre parse-boolean-ui-param)
				       (push! output-post render-boolean-ui-param)))
	      (digits-only	(push! input-pre digits-only))
	      ((custom input-pre ,what)		(push! input-pre what))
	      ((custom input-post ,what)	(push! input-post what))
	      ((custom output-pre ,what)	(push! output-pre what))
	      ((custom output-post ,what)	(push! output-post what))
	      ((>= ,n)		(push! input-pre (numeric-ineq >=
							       "greater than or equal to"
							       n)))
	      ((<= ,n)		(push! input-pre (numeric-ineq <=
							       "less than or equal to"
							       n)))
	      ((length= ,n)	(begin (push! input-pre (string-length-compares
							 (lambda args (not (apply = args)))
							 "other" n))
				       (push! metadata `(maxlength ,n))))
	      ((maxlength ,n)	(begin (push! input-pre (string-length-compares > "longer" n))
				       (push! metadata `(maxlength ,n))))
	      ((minlength ,n)   (begin (push! input-pre (string-length-compares < "shorter" n))
				       (push! metadata `(minlength ,n))))
	      ((options ,o ...)	(push! metadata (build-options-metadata o)))
	      (,_			(error+ "build-parameters: bad clause"
						`((clause ,clause))))))
	  (for-each process-clause field-entry)
	  `(,@(finish metadata)
	    ,@(finish input-pre)
	    ,@(finish input-post)
	    ,@(finish output-pre)
	    ,@(finish output-post))))))

  (define (validated-field/parameters name initial-content parameters-expr)
    (let ((parameters (build-field-parameters parameters-expr)))
      (letrec ((field (make-validated-field
		       name
		       initial-content
		       #f
		       (field-updater-from-parameters
			(cut set-validated-field-content! field <>)
			values
			parameters)
		       (get-field-parameters 'metadata parameters))))
	field)))
  
  )

(module lshift/db-validated-fields
    (ui-db-result
     ui-db-result*
     ui-db-result/ro
     ui-db-result/ro*
     ui-db-result/parameters
     build-all-parameters)

  (import srfi-1)
  (import srfi-26)
  (import lshift/db)
  (import lshift/error)
  (import lshift/xslt-ui)
  (import lshift/field-parameters)
  
  (define (field-parameters-for field-name parameters)
    (cond
     ((assq field-name parameters) => cdr)
     (else '())))

  ;; Get a single value for a parameter
  (define (get-parameter name field-parameters)
    (and-let* ((entry (assq name field-parameters)))
      (cadr entry)))


  
  ;; This is the point where we switch from a single value to a list of values.
  (define (outbound-converter-for-field db-field-type)
    (let ((main-converter (db-field-type-scheme->sxml db-field-type)))
      (lambda (action p die)
	(list (main-converter p)))))

  ;; This is the point where we switch from a list of values to a single value.
  (define (inbound-converter-for-field field-name db-field-type)
    (let ((main-converter (db-field-type-sxml->scheme db-field-type)))
      (lambda (action vals die)
	(cond
	 ((null? vals) (error+ "db-result-field-updater: too few values" field-name vals))
	 ((pair? (cdr vals)) (error+ "db-result-field-updater: too many values" field-name vals))
	 (else (main-converter (car vals) die))))))

  (define (db-result-field-updater r field-name db-field-type field-parameters)
    (field-updater-from-parameters
     (cut db-result-set! r field-name <>)
     (inbound-converter-for-field field-name db-field-type)
     field-parameters))

  (define (db-result-field-content-source r field-name db-field-type field-parameters)
    (let ((if-absent-values (get-parameter 'if-absent-values field-parameters)))
      (lambda ()
	(let* ((scheme-value (db-result-get r field-name (lambda () 'absent)))
	       (value-present-in-result (not (eq? scheme-value 'absent))))
	  (cond
	   (value-present-in-result
	    ((apply-field-converters 'output-pre
			       (outbound-converter-for-field db-field-type)
			       'output-post
			       field-parameters)
	     #t
	     scheme-value
	     (lambda (message)
	       (error+ (string-append "db-result-field-content-source: "
				      message)
		       `((field-name ,field-name)
			 (scheme-value ,scheme-value)
			 (db-result-class ,(db-result-class-print-string
					    (db-result-class r))))))))
	   (if-absent-values)
	   (else '()))))))

  (define (db-result->template-value r readonly parameters)
    (let ((class (db-result-class r)))
      (define (build-field field-name)
	(let* ((type (db-result-class-field-type class field-name))
	       (field-parameters (field-parameters-for field-name parameters))
	       (content-source (db-result-field-content-source r field-name type field-parameters))
	       (on-submit (if readonly
			      #f
			      (db-result-field-updater r field-name type field-parameters)))
	       (vf (make-validated-field field-name
					 #f
					 content-source
					 on-submit
					 (get-field-parameters 'metadata field-parameters))))
	  (and-let* ((error-message (get-parameter 'errormessage field-parameters)))
	    (set-validated-field-error! vf error-message))
	  vf))
      (cons (db-result-class-name class)
	    (filter-map build-field (db-result-class-field-names class)))))

  (define (ui-db-result db-result . p) (ui-db-result/parameters
					#f db-result #f
					(if (pair? p) (car p) '())))
  (define (ui-db-result* newname db-result . p) (ui-db-result/parameters
						 newname db-result #f
						 (if (pair? p) (car p) '())))

  (define (ui-db-result/ro db-result . p) (ui-db-result/parameters
					   #f db-result #t
					   (if (pair? p) (car p) '())))
  (define (ui-db-result/ro* newname db-result . p) (ui-db-result/parameters
						    newname db-result #t
						    (if (pair? p) (car p) '())))

  ;; parameters: pair-alist mapping field-name to field-parameters
  ;; field-parameters: pair-alist mapping parameter name to parameter values
  ;;
  ;; Defined parameters for each field:
  ;;
  ;; output-pre: multiple values expected
  ;; output-post: multiple values expected
  ;; input-pre: multiple values expected
  ;; input-post: multiple values expected
  ;;
  ;;	Each of these parameters is to map to a list of functions,
  ;;	each list of which is run through in left-to-right order.
  ;;
  ;;	Each function in each list has signature:
  ;;		action * val-or-vals * die --> val-or-vals
  ;;	where:
  ;;		action is a symbol, or false if any fault has been
  ;;		found with the values as submitted by the user thus
  ;;		far through the entire validation-and-checking chain
  ;;
  ;;		val-or-vals is either a single value or a list of
  ;;		values, depending on whether you're an
  ;;		output-pre/input-post or output-post/input-pre
  ;;		function, respectively; see below for details
  ;;
  ;;		die is a function taking an error-message string
  ;;		indicating the problem that this validator has
  ;;		discovered with the val-or-vals. If it is called, it
  ;;		aborts the current input/output-chain and causes the
  ;;		action parameter supplied to input-chains for other
  ;;		fields to be set to #f
  ;;
  ;;	Each function should return val-or-vals appropriately
  ;;	transformed.
  ;;
  ;;
  ;;	The output-pre functions are expected to take a single scheme
  ;;	value, directly from the db-result, and convert it somehow to
  ;;	another single scheme value. After the output-pre functions
  ;;	have run, the db-result's db-field-type-scheme->sxml function
  ;;	is run on the value, converting it to an SXML value, and then
  ;;	this SXML value is placed into a list, since HTML/HTTP
  ;;	supports *multiple* values for each control. This list of SXML
  ;;	values (containing a single value) is then run through each of
  ;;	the output-post functions, which must map the list of SXML
  ;;	values to a new list of values. At the final stage of the
  ;;	chain, the values must be strings. They are put into a
  ;;	validated-field structure and sent off to the user agent using
  ;;	send-xslt-ui.
  ;;
  ;;	On the way back in, the multiple (possibly zero) string values
  ;;	for each control are fed to each input-pre function in
  ;;	turn. The input-pre functions must map lists of values to
  ;;	lists of values, just as the output-post functions. After the
  ;;	input-pre functions have each had their turn, the list of
  ;;	values is checked. If it is empty, an error is raised; if it
  ;;	has more than one element, an error is raised. The
  ;;	db-field-type-sxml->scheme function is then run on the single
  ;;	value in the list of values (which must be an SXML value
  ;;	appropriate for the type of db-field-type-sxml->scheme
  ;;	converter being used) converting it into a scheme value. The
  ;;	scheme value is passed through the chain of input-post
  ;;	functions, which like the output-pre functions map a single
  ;;	scheme value to another single scheme value. Finally, the
  ;;	resulting scheme value is stored into the db-result using
  ;;	db-result-set!.
  ;;
  ;; if-absent-values: single value expected
  ;;
  ;;	Usually, if a particular field named in a db-result-class is
  ;;	absent from the db-result being processed, no validated-field
  ;;	SXML fragment is generated at all. Supplying the
  ;;	if-absent-values parameter, containing a single list of
  ;;	strings, causes the output-chain to be skipped, the list of
  ;;	strings being supplied to the user agent as the current values
  ;;	for the field, and the input-chain being processed as usual.
  ;;
  ;; metadata: multiple values expected
  ;;
  ;;	The values of this parameter are taken together and passed
  ;;	unchanged to the make-validated-field constructor.
  ;;
  ;; errormessage: single value expected
  ;;
  ;;	If this parameter is supplied for a field, it will be used as
  ;;	the original errormessage text for the validated-field instead
  ;;	of the empty string.

  (define (build-all-parameters parameters)
    (map (lambda (field-parameters)
	   (cons (car field-parameters) (build-field-parameters (cdr field-parameters))))
	 parameters))
  
  (define (ui-db-result/parameters newname db-result readonly parameters)
    (let ((sxml (db-result->template-value db-result readonly parameters)))
      (if newname
	  (cons newname (cdr sxml))
	  sxml)))

  )
