;; Utilities for dealing with servlet-y IO

(module utils/servlet-tools

    (copy-stream request-xml forward-response real-path
                 real-uri get-authenticated-user user-in-role?
                 redirect absolute-context-path)

  (import srfi-13)

  (define (jlength/null-check x)
    (if (java-null? x)
	0
	(->number (jlength x))))

  (define (absolute-context-path request)
    (let* ((full (->string (get-request-url request)))
	   (context-path (->string (get-context-path request)))
	   (slashes-index (string-contains full "//"))
	   (root-index (if slashes-index
			   (string-index full #\/ (+ slashes-index 2))
			   0))
	   (root (substring full 0 root-index))
	   (base (string-append root context-path "/")))
      (->jstring base)))
  
  (import s2j)
  
  (define-java-classes
    (<java.file> |java.io.File|)
    (<java.hash-map> |java.util.HashMap|)
    (<commons.io-utils> |org.apache.commons.io.IOUtils|)
    (<commons.copy-utils> |org.apache.commons.io.CopyUtils|))
  
  (define-generic-java-methods
    (to-uri |toURI|)
    (io-copy copy)
    next
    has-next
    to-string
    equals-ignore-case
    key-set
    get
    iterator
    (jsubstring |substring|)
    (jlength |length|)
    set-header ;; HttpServletResponse
    get-output-stream
    flush ;; BufferedOutputStream
    get-method
    get-session
    get-servlet-context
    (get-request-url |getRequestURL|)
    get-context-path
    get-servlet-path
    get-path-info
    get-real-path
    get-query-string
    get-input-stream
    get-user-principal
    is-user-in-role
    get-name
    get-character-encoding
    send-redirect
    (encode-redirect-url |encodeRedirectURL|))

  (define (get-authenticated-user request)
    (let ((user (get-user-principal request)))
      (if (java-null? user)
          #f
          (->string (get-name user)))))

  (define (user-in-role? request role)
    (->boolean (is-user-in-role request (->jstring role))))
  
  (define (copy-stream in out)
    (io-copy (java-null <commons.copy-utils>) in out)
    (flush out))

  (define (forward-response headers res stream)
    (iterator-for-each (lambda (key)
                         (if (not (->boolean (equals-ignore-case key (->jstring "transfer-encoding"))))
                             (set-header res key (get headers key))))
                       (iterator (key-set headers)))
    (copy-stream stream (get-output-stream res)))

  (define (redirect response uri)
    (send-redirect response (encode-redirect-url response uri)))
  
  ;; Get XML posted in a request
  (define (request-xml request)
    (case (->symbol (get-method request))
      ((|GET|)
       (->string (get-query-string request)))
      ((|POST|)
       (->string
        (to-string (java-null <commons.io-utils>)
                   (get-input-stream request)
                   (get-character-encoding request))))))
  
  (define (iterator-for-each f iterator)
    (if (->boolean (has-next iterator))
        (begin (f (next iterator))
               (iterator-for-each f iterator))))

  (define (real-path rel-to-context)
    (let ((result (get-real-path *SERVLET-CONTEXT* (->jstring rel-to-context))))
      (and (not (java-null? result))
	   (->string result))))

  (define (real-uri rel-to-context)
    (to-uri (java-new <java.file> (->jstring (real-path rel-to-context)))))

  )