; Debugging code

(define-method debug-dump <continuation>
  (let ((partial-cont
	 (compute-slot-getter <continuation> 'partial-continuation))
	(stack (compute-slot-getter <continuation> 'stack)))
    (letrec ((backtrace
	      (lambda (frame-number ctxt)
		(if (not (null? (func ctxt)))
		    (begin
		      (print "    ---- Frame ")
		      (print frame-number)
		      (print ":\n    ")
		      (print (func ctxt))
		      (if (not (null? (curr-gf ctxt)))
			  (begin
			    (print " (in gf ")
			    (print (curr-gf ctxt))
			    (print ")")))
		      (print " at ip ")
		      (print (ip ctxt))
		      (print ".\n    env: ")
		      (print (env ctxt))
		      (newline)
		      (if (not (null? (next ctxt)))
			  (backtrace (+ frame-number 1) (next ctxt)))))))
	     (next (lambda (ctxt) (indexed-ref ctxt 0)))
	     (ip (lambda (ctxt) (indexed-ref ctxt 1)))
	     (env (lambda (ctxt) (indexed-ref ctxt 2)))
	     (func (lambda (ctxt) (indexed-ref ctxt 3)))
	     (curr-gf (lambda (ctxt) (indexed-ref ctxt 4))))
      (lambda (cont)
	(print "Stack:\t\t")
	(print (stack cont))
	(print "\nBacktrace:\t\n")
	(backtrace 0 (partial-cont cont))))))

(require 'disassemble)

(define-method debug-dump <function>
  (lambda (self . rest)
    (let ((assembly (disassemble self)))
      (define dump
	(lambda (code n depth)
	  (cond
	   ((= n 0))
	   ((null? code))
	   (else
	    (print (car (car code)))
	    (print "\t")
	    (set! code (cons (cdr (car code)) (cdr code)))
	    (let ((x (* depth 4)))
	      (while (> x 0)
		     (print " ")
		     (set! x (- x 1))))
	    (print (car code))
	    (newline)
	    (if (eq? (car (car code)) 'lambda)
		(dump (disassemble (car (cdr (car code))))
		      (+ depth 1)))
	    (dump (cdr code) (if (> n 0) (- n 1) n) depth)))))
      (define offset (if (null? rest)
			 assembly
			 (let ((n (car rest)))
			   (while (and (> n 0) (instance? assembly <pair>))
				  (set! assembly (cdr assembly))
				  (set! n (- n 1)))
			   assembly)))
      (define length (if (or (null? rest)
			     (null? (cdr rest)))
			 -1
			 (car (cdr rest))))
      (dump offset length 0))))

(define-class <debug-state> <object> ((exception #t #f)
				      (argument #t #f)
				      (continuation #t #f)
				      frame-number
				      last-pos))

(define-method initialize <debug-state>
  (let ((set-x! (compute-slot-setter <debug-state> 'exception))
	(set-a! (compute-slot-setter <debug-state> 'argument))
	(set-c! (compute-slot-setter <debug-state> 'continuation)))
    (lambda (self x a c)
      (set-x! self x)
      (set-a! self a)
      (set-c! self c)
      (set-frame-number! self 0)
      (set-last-pos! self 0)
      self)))

(define-method get-debug-frame <debug-state>
  (lambda (ds)
    (let ((frame-num (frame-number ds))
	  (frame (partial-continuation (continuation ds))))
      (while (> frame-num 0)
	     (set! frame (indexed-ref frame 0))
	     (set! frame-num (- frame-num 1)))
      frame)))

(define-method get-call-depth <debug-state>
  (lambda (ds)
    (let loop ((depth 0) (cont (partial-continuation (continuation ds))))
      (if (or (null? cont)
	      (null? (indexed-ref cont 0))
	      (null? (indexed-ref (indexed-ref cont 0) 3)))
	  depth
	  (loop (+ depth 1) (indexed-ref cont 0))))))

(define debug-command-table '())

; Format of a debug-command-procedure:
; (lambda (debug-state arguments)
;  ...)

(define (add-debug-command name-list help-text default-args-list command-procedure)
  (set! debug-command-table
	(cons (list name-list help-text default-args-list command-procedure)
	      debug-command-table)))

(define (lookup-debug-command name)
  (let loop ((commands debug-command-table))
    (cond
     ((null? commands) #f)
     ((memq name (car (car commands)))
      (cdr (car commands)))
     (else
      (loop (cdr commands))))))

(define (run-debug-command name debug-state args)
  (let ((command (lookup-debug-command name)))
    (if (not command)
	(print "No command matches what you entered. Try again.\n")
	(begin
	  (if args
	      (set! args
		    (let loop ((args args)
			       (defaults (car (cdr command))))
		      (cond
		       ((null? defaults) args)
		       ((null? args) defaults)
		       (else
			(cons (car args)
			      (loop (cdr args)
				    (cdr defaults))))))))
	  ((car (cdr (cdr command))) debug-state args)))))

(define (debug-main-repl debug-state)
  (while #t	; Never stops looping this loop. Use (abort), or
		; return to a continuation, to escape.
	 (print ">>>>debug>>>> (? for help): ")
	 (if (stream-at-eof? standard-input)
	     (abort))
	 (let ((command (read)))
	   (if (eof-object? command)
	       (abort))
	   (newline)
	   (if (instance? command <pair>)
	       (run-debug-command (car command) debug-state (cdr command))
	       (run-debug-command command debug-state #f))
	   (newline))))

(add-debug-command '(? h help)
"\
    ?, h, help			Display this message\n\
    (help), (help index)	Display index of all commands available\n\
    (help <topic>)		Display help on <topic>\n\
"
        '(index)
	(let ((help-for (lambda (topic)
			  (print "HELP FOR ")
			  (print topic)
			  (print "\n\n")
			  (print
			   (let ((command (lookup-debug-command topic)))
			     (if command
				 (car command)
				 "    Not found\n"))))))
	  (lambda (ds args)
	    (if args
		(while (not (null? args))
		       (if (eq? (car args) 'index)
			   (begin
			     (print "HELP INDEX:\n\n")
			     (for-each
			      (lambda (command)
				(print "    ")
				(print (car command))
				(newline))
			      debug-command-table))
			   (help-for (car args)))
		       (set! args (cdr args)))
		(help-for 'help)))))

(add-debug-command '(q quit a abort)
"\
    q, quit, a, abort\n\
\n\
    Quit the debugger, returning to the interactive top-level.\n\
"
        '()
	(lambda (ds args)
	  (abort)))

(add-debug-command '(re recap)
"\
    re, recap\n\
\n\
    Print the reason the debugger was activated, and the arguments passed\n\
    to the debugger.\n\
"
        '()
	(lambda (ds args)
	  (print "= = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = \n")
	  (print "    Exception raised:\t\t")
	  (print (exception ds))
	  (newline)
	  (print "    Arguments:\t\t\t")
	  (print (argument ds))
	  (newline)
	  (print "= = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = \n")
	))

(add-debug-command '(bt backtrace)
"\
    bt, backtrace\n\
\n\
    Display the call stack (continuation) which was active at the time of\n\
    the exception.\n\
"
        '()
	(lambda (ds args)
	  (debug-dump (continuation ds))))

(add-debug-command '(f dump-frame)
"\
    f, dump-frame\n\
\n\
    Print a short summary of the currently selected frame\n\
"
        '()
	(lambda (ds args)
	  (let ((frame (get-debug-frame ds)))
	    (print "Current frame:\n")
	    (print "    Function:\t\t")
	    (print (indexed-ref frame 3))
	    (newline)
	    (print "    Generic Function:\t")
	    (print (indexed-ref frame 4))
	    (newline)
	    (print "    IP:\t\t\t")
	    (print (indexed-ref frame 1))
	    (newline))))

(add-debug-command '(s select-frame)
"\
    s, select-frame\n\
    (select-frame <framenum>)\n\
\n\
    Selects a specific frame from the current call stack. Defaults to\n\
    selecting the topmost (innermost) frame.\n\
"
        '(0)
	(lambda (ds args)
	  (set-last-pos! ds 0)
	  (if args
	      (set-frame-number! ds (car args))
	      (set-frame-number! ds 0))
	  (let ((maxdepth (get-call-depth ds)))
	    (if (> (frame-number ds) maxdepth)
		(set-frame-number! ds maxdepth))
	    (if (< (frame-number ds) 0)
		(set-frame-number! ds 0)))
	  (run-debug-command 'dump-frame ds #f)))

(add-debug-command '(d down)
"\
    d, down\n\
\n\
    Selects the calling stack frame of the currently selected frame.\n\
"
        '()
	(lambda (ds args)
	  (let ((frame (get-debug-frame ds)))
	    (if (or (null? frame)
		    (null? (indexed-ref frame 0))
		    (null? (indexed-ref (indexed-ref frame 0) 3)))
		(print "Currently selected frame has no caller.\n")
		(begin
		  (set-frame-number! ds (+ (frame-number ds) 1))
		  (run-debug-command 'dump-frame ds #f))))))

(add-debug-command '(u up)
"\
    u, up\n\
\n\
    Selects the called stack frame of the currently selected frame.\n\
"
        '()
	(lambda (ds args)
	  (if (= (frame-number ds) 0)
	      (print "Currently selected frame has no callee.\n")
	      (begin
		(set-frame-number! ds (- (frame-number ds) 1))
		(run-debug-command 'dump-frame ds #f)))))

(add-debug-command '(l list)
"\
    l, list     Disassemble from ip-10 to ip+10\n\
    (list <s> <l>)  Disassemble from position <s> for <l> instructions\n\
\n\
    <s> can be an instruction number, a \"+\" sign representing the last\n\
        instruction disassembled, or \"ip\", representing the instruction\n\
        pointer associated with this stack frame. This value defaults to\n\
        \"+\".\n\
    <l> can be omitted, in which case the value 20 will be assumed.\n\
\n\
    The list command displays the disassembly of the function active in\n\
    the currently selected stack-frame (use u and d to select stack\n\
    frames).\n\
"
        '(+ 20)
	(let ((dump-function
	       (lambda (func ip ofs minus length)
		 (let* ((asm (disassemble func))
			(curr asm))
		   (while (and (not (null? curr))
			       (> minus 0)
			       (< (car (car curr)) ofs))
			  (set! curr (cdr curr))
			  (set! minus (- minus 1)))
		   (while (and (not (null? curr))
			       (< (car (car curr)) ofs))
			  (set! curr (cdr curr))
			  (set! asm (cdr asm)))
		   (print "\tRelOfs\tOffset\tInstruction\n")
		   (print "\t--------------------------------\n")
		   (while (and (not (null? asm))
			       (> length 0))
			  (print "\t")
			  (print (- (car (car asm)) ip))
			  (print "\t")
			  (print (car (car asm)))
			  (print "\t")
			  (print (cdr (car asm)))
			  (newline)
			  (set! minus (- minus 1))
			  (set! asm (cdr asm))
			  (set! length (- length 1)))
		   (if (null? asm)
		       (binary-length func)
		       (car (car asm)))))))
	  (lambda (ds args)
	    (let* ((frame (get-debug-frame ds))
		   (func (indexed-ref frame 3))
		   (ip (indexed-ref frame 1)))
	      (cond
	       ((not (instance? func <function>))
		(print "The currently selected frame does not contain a disassemblable\n")
		(print "function.\n"))
	       (args
		(set-last-pos! ds
			       (dump-function func
					      ip
					      (case (car args)
						((+) (last-pos ds))
						((ip) ip)
						(else (car args)))
					      0
					      (car (cdr args)))))
	       (else
		(set-last-pos! ds (dump-function func ip ip 10 20))))))))

(add-debug-command '(e env)
"\
    e, env          Print the currently selected frame's environment\n\
    (env <n>)       Print the nth scope only\n\
    (env <s> <o>)   Print the s'th scope, o'th offset\n\
\n\
    <n>, <s> and <o> can all be either numbers or the value \"all\".\n\
"
        '(all all)
	(letrec ((print-offset
		  (lambda (env offset)
		    (print "\t")
		    (print offset)
		    (print ":\t")
		    (print (indexed-ref env offset))
		    (newline)))
		 (print-scope
		  (lambda (env scope offset)
		    (print "    Scope ")
		    (print scope)
		    (print ":\n")
		    (while (> scope 0)
			   (set! env (cdr env))
			   (set! scope (- scope 1)))
		    (set! env (car env))
		    (if (eq? offset 'all)
			(let ((n 0)
			      (max (indexed-length env)))
			  (while (< n max)
				 (print-offset env n)
				 (set! n (+ n 1))))
			(print-offset env offset)))))
	  (lambda (ds args)
	    (if args
		(let* ((frame (get-debug-frame ds))
		       (env (indexed-ref frame 2)))
		  (case (car args)
		    ((all)
		     (let ((n 0)
			   (max (list-length env)))
		       (while (< n max)
			      (print-scope env n (car (cdr args)))
			      (set! n (+ n 1)))))
		    (else
		     (print-scope env (car args) (car (cdr args))))))
		(run-debug-command 'env ds '(all all))))))

(add-debug-command '(st stack)
"\
    st, stack\n\
\n\
    Prints the contents of the stack at the time the exception occurred.\n\
"
        '()
	(lambda (ds args)
	  (print "    Stack contents:\n")
	  (let* ((n 0)
		 (stack (continuation-stack (continuation ds)))
		 (max (indexed-length stack)))
	    (while (< n max)
		   (print "\t")
		   (print (- max (+ n 1)))
		   (print ":\t")
		   (print (indexed-ref stack n))
		   (newline)
		   (set! n (+ n 1))))))

(add-debug-command '(r return)
"\
    (r <expr>)\n\
    (return <expr>)\n\
\n\
    Returns the value of <expr> (which is executed in the toplevel\n\
    environment) to the point at which the exception occurred, and tries\n\
    to continue execution.\n\
"
        '()
	(lambda (ds args)
	  (if (or (not args)
		  (null? args))
	      (begin
		(print "You _must_ enter in an expression to give a result to the\n")
		(print "continuation after the exception.\n"))
	      (let ((thunk (compile (car args))))
		(if (null? thunk)
		    (begin
		      (print "Your expression didn't compile properly.\n")
		      (print "Please try again.\n"))
		    ((continuation ds) (thunk)))))))

(add-debug-command '(r! return-here)
"\
    (r! <expr>)\n\
    (return-here <expr>)\n\
\n\
    Returns the value of <expr> (which is executed in the toplevel\n\
    environment) to the CURRENTLY SELECTED STACK FRAME, and continue\n\
    execution from where THAT STACK FRAME left off.\n\
"
        '()
	(lambda (ds args)
	  (if (or (not args)
		  (null? args))
	      (begin
		(print "You _must_ enter in an expression to give a result to the\n")
		(print "continuation at this stack frame.\n"))
	      (let ((thunk (compile (car args))))
		(if (null? thunk)
		    (begin
		      (print "Your expression didn't compile properly.\n")
		      (print "Please try again.\n"))
		    ((rebuild-continuation (continuation ds)
					   (get-debug-frame ds))
		     (thunk)))))))

(add-debug-command '(v eval)
"\
    v, eval         Descend into a read-eval-print loop\n\
    (eval <expr>)   Print the result of expression <expr>\n\
\n\
    To exit the read-eval-print loop and return to the debugger,\n\
    hit your interrupt key (usually ^C)\n\
"
        '()
	(lambda (ds args)
	  (if (or (not args)
		  (null? args))
	      (call/cc
	       (lambda (return-to-debugger)
		 (try
		  (lambda (ex args cont)
		    (if (and (eq? ex 'signal-raised)
			     (= args 2))	; signal 2 is SIGINT
			(begin
			  (remove-handler)
			  (print "\n;Returning to debugger...\n")
			  (return-to-debugger #t))))
		  (begin
		    (define repl
		      (lambda ()
			(print "debug-repl> ")
			(if (stream-at-eof? standard-input)
			    (return-to-debugger #f))
			(let ((thunk (compile (read))))
			  (if (not (null? thunk))
			      (let ((result (thunk)))
				(print "; Returned:\n")
				(print result)
				(newline)))
			  (repl))))
		    repl))))
	      (let ((thunk (compile (car args))))
		(if (null? thunk)
		    (print "Your expression did not compile correctly.\n")
		    (let ((result (thunk)))
		      (print "Your expression returned:\n")
		      (print result)
		      (newline)))))))

(while (remove-handler))	; Remove all currently installed exception handlers

(install-handler
   (lambda (exception argument continuation)
     (let ((ds (make <debug-state> exception argument continuation)))
       (newline)
       (run-debug-command 'recap ds #f)
       (newline)
       (run-debug-command 'stack ds #f)
       (newline)
       (run-debug-command 'select-frame ds '(0))
       (newline)
       (debug-main-repl ds))))
