(require 'windowing-toolkit)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Class definitions

(define-class <button> <control> (default?		; boolean
				  selecting		; boolean
				  (title get-title)))	; <string>

(define-class <checkbox> <control> ((value #t wt:set-value!)	; boolean
				    (title get-title)))		; <string>

(define-class <key-listener> <wt> (action))	; lambda (self event key-make?
						;	  key-sym key-string state) -> boolean

(define-class <panel> <wt-visible> (color))	; <integer>

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; (pseudo-) Configuration section

(define wt-widgets:control-font #f)

(define (set-widget-control-font f)
  (set! wt-widgets:control-font f))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Utility functions and macros

(define-macro (incr name by)
  `(set! ,name (+ ,name ,by)))

(define-macro (decr name by)
  `(set! ,name (- ,name ,by)))

(define-method draw-raised-rect <gc>
  (lambda (self x y w h inverted?)
    (let ((white (lookup-color self 65535 65535 65535))
	  (lgray (lookup-color self 49152 49152 49152))
	  (dgray (lookup-color self 32768 32768 32768))
	  (black (lookup-color self 0 0 0)))
      (set-fg-color self lgray)
      (fill-rectangle self x y w h)
      (set-fg-color self black)
      (draw-rectangle self x y w h)
      (do ((count 0 (+ count 1)))
	  ((= count 2))
	(incr x 1)
	(incr y 1)
	(decr w 2)
	(decr h 2)
	(set-fg-color self (if inverted? dgray white))
	(draw-line self x y (+ x w) y)
	(draw-line self x y x (+ y h))
	(set-fg-color self (if inverted? white dgray))
	(draw-line self (+ x w) y (+ x w) (+ y h))
	(draw-line self x (+ y h) (+ x w) (+ y h))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <button>

(define-method initialize <button>
  (lambda (self p title x y w h a)
    (set-default?! self #f)
    (set-title! self title)
    (set-selecting! self #f)
    (next-method self p x y w h a)))

(define-method paint <button>
  (lambda (self gc)
    (if (intersects? self gc)
	(wt-widgets:paint-button self gc (eq? (selecting self) 'in)))))

(define (wt-widgets:paint-button self gc inverted?)
  (let ((black (lookup-color gc 0 0 0))
	(x (get-x self))
	(y (get-y self))
	(width (get-width self))
	(height (get-height self))
	(title (get-title self)))
    (if (default? self)
	(begin
	  (set-fg-color gc black)
	  (draw-rectangle gc x y width height)
	  (incr x 1)
	  (incr y 1)
	  (decr width 2)
	  (decr height 2)))
    (draw-raised-rect gc x y width height inverted?)
    (if wt-widgets:control-font
	(begin
	  (set-font gc wt-widgets:control-font)
	  (set-fg-color gc black)
	  (draw-string gc
		       (+ (+ x (if inverted? 2 0))
			  (- (/ width 2)
			     (/ (text-width wt-widgets:control-font title) 2)))
		       (+ (+ y (if inverted? 2 0))
			  (/ (* height 2) 3))
		       title)))))

(define-method handle-event <button>
  (wt-event-handler (self event-type event)
    ((button-press)
     (if (not (selecting self))
	 (begin
	   (set-selecting! self 'in)
	   (capture-mouse self)
	   (paint self (get-gc self))))
     (exit #t))
    ((button-release)
     (let ((old-value (selecting self)))
       (if old-value
	   (begin
	     (set-selecting! self #f)
	     (release-mouse self)
	     (if (eq? old-value 'in)
		 (begin
		   (action self event)
		   (paint self (get-gc self))))))
       (exit #t)))
    ((pointer-motion)
     (let ((old-value (selecting self)))
       (if old-value
	   (begin
	     (set-selecting! self (if (contains? self (at event 'x) (at event 'y))
				      'in
				      'out))
	     (if (not (eq? (selecting self) old-value))
		 (paint self (get-gc self)))
	     (exit #t)))))
    ((key-press)
     (if (and (equal? (at event 'string) "\r")
	      (or (eq? (at event 'window) self)
		  (default? self))
	      (action self event))
	 (exit #t)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <checkbox>

(define-method initialize <checkbox>
  (lambda (self p title value x y w h a)
    (wt:set-value! self value)
    (set-title! self title)
    (next-method self p x y w h a)))

(define-method set-value! <checkbox>
  (lambda (self new)
    (wt:set-value! self new)
    (paint self (get-gc self))))

(define-method paint <checkbox>
  (lambda (self gc)
    (let ((x (get-x self))
	  (y (get-y self)))
      (draw-raised-rect gc x y 16 16 (value self))
      (if wt-widgets:control-font
	  (begin
	    (set-font gc wt-widgets:control-font)
	    (set-fg-color gc (lookup-color gc 0 0 0))
	    (draw-string gc (+ x 22) (+ y 14) (get-title self)))))))

(define-method handle-event <checkbox>
  (wt-event-handler (self event-type event)
    ((button-press)
     (set-value! self (not (value self)))
     (action self event)
     (exit #t))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <key-listener>

(define-method initialize <key-listener>
  (lambda (self parent action)
    (set-action! self action)
    (next-method self parent)))

(define-method handle-event <key-listener>
  (wt-event-handler (self event-type event)
    ((key-press key-release)
     (if (and (action self)
	      ((action self) self event (eq? event-type 'key-press)
			     (at event 'keysym) (at event 'string)
			     (at event 'state)))
	 (exit #t)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <panel>

(define-method initialize <panel>
  (lambda (self parent color)
    (set-color! self color)
    (next-method self parent)))

(define-method get-x <panel> (lambda (self) (get-x (parent self))))
(define-method get-y <panel> (lambda (self) (get-y (parent self))))
(define-method get-width <panel> (lambda (self) (get-width (parent self))))
(define-method get-height <panel> (lambda (self) (get-height (parent self))))

(define-method paint <panel>
  (lambda (self gc)
    (set-fg-color gc (color self))
    (fill-rectangle gc (get-x self) (get-y self) (get-width self) (get-height self))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Module initialisation and registration

(provide 'wt-widgets)
