(require 'avl-tree)
(require 'queue)
(require 'x-windows)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Class definitions

(define-class <display> <object> (peer			; <x-pointer>
				  root-window-id))	; <integer>

(define-class <font> <object> (peer			; <x-pointer>
			       display))		; <display>

(define-class <gc> <object> (peer			; <integer>
			     (cliprect get-cliprect #f)	; #(x y w h)
			     window))			; <window>

(define-class <event> <object> (raw-event		; alist
				display))		; <display>

(define-class <wt> <object> (parent))			; <wt-container> or #f

(define-class <wt-visible> <wt> ())

(define-class <wt-container> <wt-visible> (children))	; <queue <wt>>

(define-class <window> <wt-container> (display		; <display>
				       peer		; <integer>
				       focus		; <wt>
				       mouse-focus	; <wt>
				       on-close		; (lambda (self) -> boolean) or null
				       gc))		; <gc>

(define-class <control> <wt-visible> ((x get-x)		; <integer>
				      (y get-y)		; <integer>
				      (width get-width)	; <integer>
				      (height get-height);<integer>
				      (action get-action)))
							; lambda (<control> <event>) -> boolean

(define-method print-string <display> (lambda (self w) "#<display>"))
(define-method print-string <gc> (lambda (self w) "#<gc>"))
(define-method print-string <event> (lambda (self w) "#<event>"))
(define-method print-string <wt> (lambda (self w) "#<wt>"))
(define-method print-string <wt-container> (lambda (self w) "#<wt-container>"))
(define-method print-string <window> (lambda (self w) "#<window>"))
(define-method print-string <control> (lambda (self w) "#<control>"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Utility functions and macros

(if (not (global-variable-bound? 'assoc))
    (define-global-variable 'assoc
      (lambda (item l)
	(if (null? l)
	    #f
	    (if (equal? (caar l) item)
		(car l)
		(assoc item (cdr l)))))))

(define-macro (define-abstract-method gfname class signature)
  `(define-method ,gfname ,class
     (lambda ,signature
       (raise-exception 'abstract-method-called ',gfname))))

(define-macro (wt-event-handler names . cases)
  `(lambda ,names
     (case (call/cc (lambda (exit)
		      (case ,(cadr names)
			,@cases)
		      'do-next))
       ((#t) #t)
       ((#f) #f)
       (else (next-method ,@names)))))

(define (wt:field n e)
  (let ((b (assq n e)))
    (if b
	(cdr b)
	(raise-exception 'x-field-not-found (list n e)))))

(define (wt:set-field k v e)
  (let ((b (assq k e)))
    (if b
	(begin
	  (set-cdr! b v)
	  e)
	(cons (cons k v) e))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; The peer object

(define wt:peer (make <x-peer>))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <display>

(define-method initialize <display>
  (lambda (self display-path)
    (if (or (not display-path)
	    (equal? display-path ""))
	(set! display-path '()))
    (set-peer! self (open-display wt:peer display-path))
    (set-root-window-id! self (root-window wt:peer (peer self)))
    self))

(define-method destroy <display>
  (lambda (self)
    (if (not (null? (peer self)))
	(close-display wt:peer (peer self)))
    (set-peer! self '())))

(define-method lookup-color <display>
  (lambda (self r g b)
    (wt:find-color (peer self) (list r g b))))

(define wt:colors '())

(define (wt:find-color disp info)
  (let ((record (assoc info wt:colors)))
    (if record
	(cdr record)
	(let ((pix (apply allocate-color wt:peer disp info)))
	  (and pix
	       (begin
		 (set! wt:colors (cons (cons info pix) wt:colors))
		 pix))))))

(define wt:event-queue (make <queue>))

(define-method post-event <display>
  (lambda (self type dest-win . args)
    (insert wt:event-queue (make <event> `((type . ,type)
					   (window . ,dest-win)
					   ,@args) self))))

(define-method next-event <display>
  (lambda (self)
    (let ((node (dequeue wt:event-queue)))
      (if node
	  (queue-node-value node)
	  (make <event> (next-event wt:peer (peer self)) self)))))

(define-method event-loop <display>
  (lambda (self)
    (let next ()
      (let* ((event (next-event self))
	     (type (at event 'type))
	     (dest-win (at event 'window)))
	(case type
	  ((shutdown-wt) #t)
	  (else
	   (if dest-win
	       (if (and (not (handle-event dest-win type event))
			(instance? type <symbol>))
;		   (for-each print `("Unhandled event: " ,type ": " ,(raw-event event) "\n"))))
		   #t))
	   (next)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <font>

(define-method initialize <font>
  (lambda (self display name weight face size)
    (set-display! self display)
    (set-peer! self (obtain-font-handle wt:peer (peer display) name weight face size))
    (if (null? (peer self))
	#f
	self)))

(define-method destroy <font>
  (lambda (self)
    (if (not (null? (peer self)))
	(release-font-handle wt:peer (peer (display self)) (peer self)))
    (set-peer! self '())))

(define-method text-width <font>
  (lambda (self text)
    (text-width wt:peer (peer self) text)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <gc>

(define-method initialize <gc>
  (lambda (self win)
    (set-window! self win)
    (set-peer! self (obtain-graphics-context wt:peer (peer (display win)) (peer win)))
    (set-cliprect! self (vector 0 0 (get-width win) (get-height win)))
    self))

(define-method set-cliprect! <gc>
  (let ((setter (compute-slot-setter <gc> 'cliprect)))
    (lambda (self rect)
      (setter self rect)
      (set-clip-rectangle wt:peer (peer (display (window self))) (peer self)
			  0 0 rect))))

(define-method destroy <gc>
  (lambda (self)
    (if (not (null? (peer self)))
	(release-graphics-context wt:peer (peer (display (window self))) (peer self)))
    (set-peer! self '())))

(define-method lookup-color <gc>
  (lambda (self r g b)
    (lookup-color (display (window self)) r g b)))

(define-method draw-rectangle <gc>
  (lambda (self x y w h)
    (draw-rectangle wt:peer (peer (display (window self))) (peer (window self)) (peer self)
		    x y w h)))

(define-method draw-line <gc>
  (lambda (self x y x1 y1)
    (draw-line wt:peer (peer (display (window self))) (peer (window self)) (peer self)
	       x y x1 y1)))

(define-method fill-rectangle <gc>
  (lambda (self x y w h)
    (fill-rectangle wt:peer (peer (display (window self))) (peer (window self)) (peer self)
		    x y w h)))

(define-method fill-arc <gc>
  ; Arcs are specified by their bounding rectangle and two angles,
  ; a1 the angle to start at (in degrees * 64) and
  ; a2 the radial length of the arc (in degrees * 64).
  ; Positive values are counterclockwise; the origin is at 3 o'clock.
  (lambda (self x y w h a1 a2)
    (fill-arc wt:peer (peer (display (window self))) (peer (window self)) (peer self)
	      x y w h a1 a2)))

(define-method draw-arc <gc>
  ; Arcs are specified by their bounding rectangle and two angles,
  ; a1 the angle to start at (in degrees * 64) and
  ; a2 the radial length of the arc (in degrees * 64).
  ; Positive values are counterclockwise; the origin is at 3 o'clock.
  (lambda (self x y w h a1 a2)
    (draw-arc wt:peer (peer (display (window self))) (peer (window self)) (peer self)
	      x y w h a1 a2)))

(define-method draw-string <gc>
  (lambda (self x y str)
    (draw-string wt:peer (peer (display (window self))) (peer (window self)) (peer self)
		 x y str)))

(define-method set-fg-color <gc>
  (lambda (self color)
    (set-fg-color wt:peer (peer (display (window self))) (peer self) color)))

(define-method set-bg-color <gc>
  (lambda (self color)
    (set-bg-color wt:peer (peer (display (window self))) (peer self) color)))

(define-method set-font <gc>
  (lambda (self font)
    (set-font wt:peer (peer (display (window self))) (peer self) (peer font))))

(define-method get-x <gc> (lambda (self) (indexed-ref (get-cliprect self) 0)))
(define-method get-y <gc> (lambda (self) (indexed-ref (get-cliprect self) 1)))
(define-method get-width <gc> (lambda (self) (indexed-ref (get-cliprect self) 2)))
(define-method get-height <gc> (lambda (self) (indexed-ref (get-cliprect self) 3)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <event>

(define-method initialize <event>
  (lambda (self raw-event display)
    (set-raw-event! self raw-event)
    (let ((binding (assq 'window raw-event)))
      (set-cdr! binding (wt:find-window-peer (cdr binding))))
    (set-display! self display)
    self))

(define-method at <event>
  (lambda (self key)
    (wt:field key (raw-event self))))

(define-method put <event>
  (lambda (self key value)
    (set-raw-event! self (wt:set-field key value (raw-event self)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <wt>

(define-method initialize <wt>
  (lambda (self parent)
    (set-parent! self #f)
    (if parent
	(register-child parent self))
    self))

(define-method handle-event <wt>
  (lambda (self event-type event)
    #f))

(define-method get-gc <wt>
  (lambda (self)
    (if (parent self)
	(let ((gc (get-gc (parent self))))
	  (set-cliprect! gc (vector (get-x self) (get-y self)
				    (get-width self) (get-height self)))
	  gc)
	(raise-exception 'unrooted-wt-component (cons 'get-gc self)))))

(define-method request-focus <wt>
  (lambda (self)
    (and (parent self)
	 (make-focused (parent self) self))))

(define-method release-focus <wt>
  (lambda (self)
    (and (parent self)
	 (make-focused (parent self) #f))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <wt-visible>

(define-abstract-method get-x <wt-visible> (self))
(define-abstract-method get-y <wt-visible> (self))
(define-abstract-method get-width <wt-visible> (self))
(define-abstract-method get-height <wt-visible> (self))

(define-method contains? <wt-visible>
  (lambda (self x y)
    (and (>= x (get-x self))
	 (>= y (get-y self))
	 (< (- x (get-x self)) (get-width self))
	 (< (- y (get-y self)) (get-height self)))))

(define-method intersects? <wt-visible> 
  (lambda (self gc)
    #t))	;;; %%% change me later %%%

(define-abstract-method paint <wt-visible> (self gc))

(define-method handle-event <wt-visible>
  (wt-event-handler (self event-type event)
    ((expose)
     (paint self (get-gc self))
     (exit #t))))

(define-method capture-mouse <wt>
  (lambda (self)
    (and (parent self)
	 (make-mouse-focused (parent self) self))))

(define-method release-mouse <wt>
  (lambda (self)
    (and (parent self)
	 (make-mouse-focused (parent self) #f))))

(define-method gain-focus <wt>
  (lambda (self)
    #t))

(define-method release-focus <wt>
  (lambda (self)
    #t))

(define-method gain-mouse-focus <wt>
  (lambda (self)
    #t))

(define-method release-mouse-focus <wt>
  (lambda (self)
    #t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <wt-container>

(define-method initialize <wt-container>
  (lambda (self parent)
    (set-children! self (make <queue>))
    (next-method self parent)))

(define-method register-child <wt-container>
  (lambda (self child)
    (if (contains? (children self) child)
	#t
	(begin
	  (insert (children self) child)
	  (set-parent! child self)))))

(define-method make-focused <wt-container>
  (lambda (self to-focus)
    (and (parent self)
	 (make-focused (parent self) to-focus))))

(define-method make-mouse-focused <wt-container>
  (lambda (self to-focus)
    (and (parent self)
	 (make-mouse-focused (parent self) to-focus))))

(define (wt:intersection r1 r2)
  (let* ((x1 (indexed-ref r1 0))
	 (x2 (indexed-ref r2 0))
	 (y1 (indexed-ref r1 1))
	 (y2 (indexed-ref r2 1))
	 (x (max (indexed-ref r1 0) (indexed-ref r2 0)))
	 (y (max (indexed-ref r1 1) (indexed-ref r2 1))))
    (vector x
	    y
	    (- (min (+ x1 (indexed-ref r1 2)) (+ x2 (indexed-ref r2 2))) x)
	    (- (min (+ y1 (indexed-ref r1 3)) (+ y2 (indexed-ref r2 3))) y))))

(define (wt:rect-empty? r)
  (or (<= (indexed-ref r 2) 0)
      (<= (indexed-ref r 3) 0)))

(define (wt:regions out in)
  (let ((inter (wt:intersection out in)))
    (if (wt:rect-empty? inter)
	(list inter out)
	(let* ((result '())
	       (xi (indexed-ref inter 0))
	       (yi (indexed-ref inter 1))
	       (wi (indexed-ref inter 2))
	       (hi (indexed-ref inter 3))
	       (xo (indexed-ref out 0))
	       (yo (indexed-ref out 1))
	       (wo (indexed-ref out 2))
	       (ho (indexed-ref out 3))
	       (yi-yo (- yi yo))
	       (xi-xo (- xi xo))
	       (top (vector xo yo wo yi-yo))
	       (bot (vector xo (+ yi hi) wo (- (- ho hi) yi-yo)))
	       (lt (vector xo yi xi-xo hi))
	       (rt (vector (+ xi wi) yi (- (- wo wi) xi-xo))))
	  (if (not (wt:rect-empty? top))
	      (set! result (cons top result)))
	  (if (not (wt:rect-empty? bot))
	      (set! result (cons bot result)))
	  (if (not (wt:rect-empty? lt))
	      (set! result (cons lt result)))
	  (if (not (wt:rect-empty? rt))
	      (set! result (cons rt result)))
	  (cons inter result)))))

(define-method paint <wt-container>
  (lambda (self gc)
    (let ((out (get-cliprect gc)))
      (iterate-over (children self)
		    (lambda (child)
		      (if (instance? child <wt-visible>)
			  (let ((inter (wt:intersection out (vector (get-x child)
								    (get-y child)
								    (get-width child)
								    (get-height child)))))
			    (if (not (wt:rect-empty? inter))
				(begin
				  (set-cliprect! gc inter)
				  (paint child gc))))))))))

(define-method handle-event <wt-container>
  (wt-event-handler (self event-type event)
    ((button-press button-release pointer-motion)
     (let ((x (at event 'x))
	   (y (at event 'y)))
       (iterate-over-backwards (children self)
			       (lambda (child)
				 (if (and (instance? child <wt-visible>)
					  (contains? child x y))
				     (begin
				       (put event 'window child)
				       (exit (handle-event child event-type event))))))))
    ((expose))	; pass it on to the next-method
    (else
     (iterate-over-backwards (children self)
			     (lambda (child)
			       (if (handle-event child event-type event)
				   (exit #t)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Routines taking care of the window-peer registry.

(define wt:registry (make <avl-tree>
			  (lambda (x y)
			    (< (car x) (car y)))
			  (lambda (x y)
			    (= (car x) (car y)))))

(define (wt:register-window-peer peer win)
  (let* ((new-binding (cons peer win))
	 (old-binding (lookup wt:registry new-binding)))
    (if old-binding
	(set-cdr! (node-value old-binding) win)
	(insert wt:registry new-binding))))

(define (wt:find-window-peer peer)
  (cond
   ((instance? peer <integer>)
    (if (= peer 0)
	#f
	(let ((binding (lookup wt:registry (cons peer '()))))
	  (if (not binding)
	      (raise-exception 'window-peer-not-found peer))
	  (cdr (node-value binding)))))
   (else
    peer)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <window>

(define-method initialize <window>
  (lambda (self display title x y w h)
    (set-display! self display)
    (set-peer! self (make-window wt:peer (peer display) (root-window-id display)
				 title x y w h))
    (wt:register-window-peer (peer self) self)
    (set-focus! self #f)
    (set-mouse-focus! self #f)
    (set-gc! self (make <gc> self))
    (next-method self #f)))

(define-method destroy <window>
  (lambda (self)
    (if (not (null? (peer self)))
	(free-window wt:peer (peer (display self)) (peer self)))
    (set-peer! self '())))

(define-method open <window>
  (lambda (self)
    (show-window wt:peer (peer (display self)) (peer self))))

(define-method close <window>
  (lambda (self)
    (if (or (null? (on-close self))
	    ((on-close self) self))
	(begin
	  (hide-window wt:peer (peer (display self)) (peer self))))))

(define-method wt:get-geom <window>
  (lambda (self)
    (get-window-geometry wt:peer (peer (display self)) (peer self))))

(define (wt:geom-getter n)
  (lambda (self)
    (wt:field n (wt:get-geom self))))

(define-method get-x <window> (lambda (self) 0))
(define-method get-y <window> (lambda (self) 0))
(define-method get-width <window> (wt:geom-getter 'width))
(define-method get-height <window> (wt:geom-getter 'height))

(define-method make-focused <window>
  (lambda (self to-focus)
    (and (or (not (focus self))
	     (release-focus (focus self)))
	 (begin
	   (set-focus! self to-focus)
	   (if to-focus
	       (gain-focus to-focus))
	   #t))))

(define-method make-mouse-focused <window>
  (lambda (self to-focus)
    (and (or (not (mouse-focus self))
	     (release-mouse-focus (mouse-focus self)))
	 (begin
	   (set-mouse-focus! self to-focus)
	   (if to-focus
	       (gain-mouse-focus to-focus))
	   #t))))

(define-method get-gc <window>
  (let ((getter (compute-slot-getter <window> 'gc)))
    (lambda (self)
      (let ((gc (getter self)))
	(set-cliprect! gc (vector 0 0 (get-width self) (get-height self)))
	gc))))

(define-method paint <window>
  (lambda (self gc)
    (set-fg-color gc (lookup-color gc 65535 65535 65535))
    (fill-rectangle gc 0 0 (get-width self) (get-height self))
    (next-method self gc)))

(define-method handle-event <window>
  (wt-event-handler (self event-type event)
    ((key-press key-release)
     (if (focus self)
	 (begin
	   (put event 'window (focus self))
	   (if (handle-event (focus self) event-type event)
	       (exit #t)))))
    ((button-press button-release pointer-motion)
     (if (mouse-focus self)
	 (begin
	   (put event 'window (mouse-focus self))
	   (if (handle-event (mouse-focus self) event-type event)
	       (exit #t)))))
    ((expose)
     (let ((gc (get-gc self)))
       (set-cliprect! gc (vector (at event 'x) (at event 'y)
				 (at event 'width) (at event 'height)))
       (paint self gc))
     (exit #t))
    ((window-delete)
     (if (eq? (at event 'window) self)
	 (begin
	   (close self)
	   (exit #t))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Methods on <control>

(define-method initialize <control>
  (lambda (self parent x y w h action)
    (set-x! self x)
    (set-y! self y)
    (set-width! self w)
    (set-height! self h)
    (set-action! self action)
    (next-method self parent)))

(define-method action <control>
  (lambda (self event)
    (let ((a (get-action self)))
      (and a
	   (a self event)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Module initialisation and registration

(provide 'wt)
(provide 'windowing-toolkit)
