/* $Id: compile.c,v 1.3 1998/04/05 10:33:48 tonyg Exp $ */

#include "memory.h"
#include "class.h"
#include "buffer.h"
#include "scan.h"
#include "parse.h"
#include "bytecode.h"

#include "function.h"
#include "vector.h"
#include "pair.h"
#include "symbol.h"

#include "compile.h"

#include <string.h>

typedef struct Code {
    BUFFER buf;

    OBJECT head, tail;      /* of litlist */
    OBJECT scope;           /* local environment, in other words */
} Code, *CODE;

PRIVATE CODE newcode(void) {
    CODE c = getmem(sizeof(Code));

    c->buf = newbuf(0);
    c->head = c->tail = c->scope = NULL;

    temp_register(&c->head, 1);
    temp_register(&c->tail, 1);
    temp_register(&c->scope, 1);

    return c;
}

PRIVATE void killcode(CODE code) {
    deregister_root(3);

    freemem(code);
}

PRIVATE OBJECT make_template(CODE code, char argc) {
    return newfunc(NULL, code->head, argc,
                   code->buf->pos, code->buf->buf, NULL);
}

/*--------------------------------------------------------------------------*/
/*  Code buffer handling                                                    */

#define gen(___c, ___x)             buf_append((___c)->buf, (___x))
#define patch(___c, ___p, ___x)     (___c)->buf->buf[(___p)] = (___x)

PRIVATE void patch_word(CODE code, word pos, unsigned short datum) {
    patch(code, pos, (char) (datum & 0xFF));
    patch(code, pos+1, (char) (datum >> 8));
}

/*--------------------------------------------------------------------------*/
/*  Literal table handling code                                             */

PRIVATE char get_lit(CODE code, OBJECT obj) {
    OBJECT curr = code->head;
    char num = 0;

    while (curr != NULL) {
        if (CAR(curr) == obj)
            return num;

        curr = CDR(curr);
        num++;
    }

    temp_register(&obj, 1);
    curr = cons(obj, NULL);
    deregister_root(1);

    if (code->tail != NULL)
        SETCDR(code->tail, curr);
    else
        code->head = curr;

    code->tail = curr;

    return num;
}

/*--------------------------------------------------------------------------*/
/*  Scope name-handling code                                                */

PRIVATE OBJECT enter_scope(CODE code, char size) {
  OBJECT v = newvector(size, 0);
  temp_register(&v, 1);
  v = cons(v, code->scope);
  deregister_root(1);
  return v;
}

PRIVATE OBJECT exit_scope(CODE code) {
    return CDR(code->scope);
}

PRIVATE int scope_lookup(CODE code, OBJECT sym, char *snum, char *ofs) {
    OBJECT scope = code->scope;
    OBJECT vec;

    *snum = 0;

    while (scope != NULL) {
        vec = CAR(scope);

        for (*ofs = 0; *ofs < NUMIDX(vec); (*ofs)++)
            if (IGET(vec, *ofs) == sym)
                return 1;

        scope = CDR(scope);
        (*snum)++;
    }

    return 0;
}

/*--------------------------------------------------------------------------*/
/*  Warning of syntax errors                                                */

PRIVATE void warning(char *msg) {
    fprintf(stderr, "%s\n", msg);
}

/*--------------------------------------------------------------------------*/
/*  Compilation proper                                                      */

PRIVATE int i_compile(CODE code, OBJECT expr, int tailpos);
    /* Forward reference */

PRIVATE int compile_lambda(CODE code, OBJECT expr) {
    CODE fcode;
    char argc = 0;
    OBJECT org, prev;
    OBJECT curr;

    if (list_length(expr) < 3) {
        warning("lambda: syntax is (lambda (arg ...) body ...)");
        return 0;
    }

    org = prev = NULL;
    curr = CAR(CDR(expr));

    temp_register(&org, 1);

    while (curr != NULL) {
        argc++;

        if (!instance(curr, pair_class)) {
            argc = -argc;

            if (prev == NULL)
                org = prev = cons(curr, NULL);
            else {
                SETCDR(prev, cons(curr, NULL));
                prev = CDR(prev);
            }

            break;
        }

        if (prev == NULL)
            org = prev = cons(CAR(curr), NULL);
        else {
            SETCDR(prev, cons(CAR(curr), NULL));
            prev = CDR(prev);
        }

        curr = CDR(curr);
    }

    org = listtovect(org);

    fcode = newcode();
    fcode->scope = cons(org, code->scope);

    org = newsym("begin");
    org = cons(org, CDR(CDR(expr)));

    if (!i_compile(fcode, org, 1)) {
        deregister_root(1);
        return 0;
    }

    gen(fcode, OP_RETURN);

    gen(code, OP_MAKE_LAMBDA);
    gen(code, get_lit(code, make_template(fcode, argc)));

    killcode(fcode);

    deregister_root(1); /* Removes OBJECT org */
    return 1;
}

PRIVATE int compile_cond(CODE code, OBJECT exprs, int tailpos) {
    OBJECT expr;
    word test_pos, end_true_pos;

    if (exprs == NULL) {
        gen(code, OP_LITREF);
        gen(code, get_lit(code, false));
        return 1;
    }

    expr = CAR(exprs);

    if (!instance(expr, pair_class)) {
        warning("cond syntax: (cond (expr body ...) (expr body ...) ...)");
        return 0;
    }

    if (CAR(expr) == newsym("else"))
        return i_compile(code, cons(newsym("begin"), CDR(expr)), tailpos);
    else {
        if (!i_compile(code, CAR(expr), 0))
            return 0;

        gen(code, OP_JUMP_FALSE);
        test_pos = code->buf->pos;
        gen(code, 0);
        gen(code, 0);

	{
	  OBJECT tmp = newsym("begin");
	  int result;
	  temp_register(&tmp, 1);
	  tmp = cons(tmp, CDR(expr));
	  result = i_compile(code, tmp, tailpos);
	  deregister_root(1);
	  if (!result)
	    return 0;
	}

        gen(code, OP_JUMP);
        end_true_pos = code->buf->pos;
        gen(code, 0);
        gen(code, 0);

        patch_word(code, test_pos, code->buf->pos - test_pos);

        if (!compile_cond(code, CDR(exprs), tailpos))
            return 0;

        patch_word(code, end_true_pos, code->buf->pos - end_true_pos);

        return 1;
    }
}

/********************************************
 *                                          *
 * compile_case: This code is very yucky.   *
 *               It's far, far too long.    *
 *                                          *
 ********************************************/

PRIVATE int compile_case(CODE code, OBJECT exprs, int tailpos) {
    OBJECT final_jumps = NULL;

    if (exprs == NULL) {
        warning("case: syntax is (case testobj ((item ...) body ...) ...)");
        return 0;
    }

    gen(code, OP_ENTERSCOPE);
    gen(code, 1);

    code->scope = enter_scope(code, 1);

    if (!i_compile(code, CAR(exprs), 0))
        return 0;

    exprs = CDR(exprs);

    gen(code, OP_LOCALSET);
    gen(code, 0);
    gen(code, 0);

    temp_register(&final_jumps, 1);

    while (1) {
        OBJECT expr;

        if (exprs == NULL) {
            gen(code, OP_LITREF);
            gen(code, get_lit(code, undefined));
            break;
        }

        expr = CAR(exprs);
        exprs = CDR(exprs);

        if (!instance(expr, pair_class)) {
            warning("case: syntax is (case testobj ((item ...) body ...) ...)");
            deregister_root(1); /* final_jumps */
            return 0;
        }

        if (CAR(expr) == NULL)
            continue;

        temp_register(&expr, 1);

        expr = cons(CAR(expr), CDR(expr));  /* So that we don't change the */
                                            /* original */

        SETCDR(expr, cons(NULL, CDR(expr)));
	SETCAR(CDR(expr), newsym("begin"));

        if (CAR(expr) == newsym("else")) {
            if (!i_compile(code, CDR(expr), tailpos)) {
                deregister_root(2); /* expr, final_jumps */
                return 0;
            }

            deregister_root(1); /* expr */
            break;
        }

        {
            OBJECT tocode_jumps = NULL;
            word stepover_pos;

            temp_register(&tocode_jumps, 1);

            while (CAR(expr) != NULL) {
                gen(code, OP_LITREF);
                gen(code, get_lit(code, CAR(CAR(expr))));
                gen(code, OP_PUSH);
                gen(code, OP_LOCALREF);
                gen(code, 0);
                gen(code, 0);

                gen(code, OP_JUMP_EQ);
                tocode_jumps = cons(MKNUM(code->buf->pos), tocode_jumps);
                gen(code, 0);
                gen(code, 0);

                SETCAR(expr, CDR(CAR(expr)));
            }

            gen(code, OP_JUMP);
            stepover_pos = code->buf->pos;
            gen(code, 0);
            gen(code, 0);

            while (tocode_jumps != NULL) {
                word jump_pos = (word) NUM(CAR(tocode_jumps));

                patch_word(code, jump_pos, code->buf->pos - jump_pos);

                tocode_jumps = CDR(tocode_jumps);
            }

            deregister_root(1); /* tocode_jumps */

            if (!i_compile(code, CDR(expr), tailpos)) {
                deregister_root(2); /* expr, final_jumps */
                return 0;
            }

            gen(code, OP_JUMP);
            final_jumps = cons(MKNUM(code->buf->pos), final_jumps);
            gen(code, 0);
            gen(code, 0);

            patch_word(code, stepover_pos, code->buf->pos - stepover_pos);
        }

        deregister_root(1); /* expr */
    }

    while (final_jumps != NULL) {
        word jump_pos = (word) NUM(CAR(final_jumps));

        patch_word(code, jump_pos, code->buf->pos - jump_pos);

        final_jumps = CDR(final_jumps);
    }

    deregister_root(1); /* final_jumps */

    code->scope = exit_scope(code);
    gen(code, OP_EXITSCOPE);

    return 1;
}

PRIVATE int compile_and(CODE code, OBJECT expr, int tailpos) {
    word test_pos;

    if (!i_compile(code, CAR(expr), CDR(expr) == NULL ? tailpos : 0))
        return 0;

    if (CDR(expr) != NULL) {
        gen(code, OP_JUMP_FALSE);
        test_pos = code->buf->pos;
        gen(code, 0);
        gen(code, 0);

        if (!compile_and(code, CDR(expr), tailpos))
            return 0;

        patch_word(code, test_pos, code->buf->pos - test_pos);
    }

    return 1;
}

PRIVATE int compile_or(CODE code, OBJECT expr, int tailpos) {
    word test_pos;

    if (!i_compile(code, CAR(expr), CDR(expr) == NULL ? tailpos : 0))
        return 0;

    if (CDR(expr) != NULL) {
        gen(code, OP_JUMP_TRUE);
        test_pos = code->buf->pos;
        gen(code, 0);
        gen(code, 0);

        if (!compile_or(code, CDR(expr), tailpos))
            return 0;

        patch_word(code, test_pos, code->buf->pos - test_pos);
    }

    return 1;
}

PRIVATE int i_compile(CODE code, OBJECT expr, int tailpos) {
    OBJECT class = GETCLASS(expr);

    if (subclass(class, symbol_class)) {
        char snum, ofs;

        if (scope_lookup(code, expr, &snum, &ofs)) {
            gen(code, OP_LOCALREF);
            gen(code, snum);
            gen(code, ofs);
            return 1;
        }

        gen(code, OP_GLOBALREF);
        gen(code, get_lit(code, expr));

        return 1;
    }

    if (subclass(class, pair_class)) {
        if (CAR(expr) == newsym("define")) {
            if (code->scope != NULL) {
                warning("define: only valid at toplevel and internally");
                return 0;
            }

            if (GETCLASS(CAR(CDR(expr))) != symbol_class) {
                warning("define: needs a symbol as first argument");
                return 0;
            }

            gen(code, OP_LITREF);
            gen(code, get_lit(code, CAR(CDR(expr))));
            gen(code, OP_PUSH);
            if (!i_compile(code, CAR(CDR(CDR(expr))), 0))
                return 0;
            gen(code, OP_PUSH);
            gen(code, OP_GLOBALREF);
            gen(code, get_lit(code, newsym("define-global-variable")));
            gen(code, tailpos ? OP_TAIL_CALL : OP_CALL);
            gen(code, 2);

            return 1;
        }

        if (CAR(expr) == newsym("quote")) {
            if (CDR(expr) == NULL) {
                warning("Quote: nothing to quote (\"(quote)\")");
                return 0;
            }

            gen(code, OP_LITREF);
            gen(code, get_lit(code, CAR(CDR(expr))));
            return 1;
        }

        if (CAR(expr) == newsym("lambda"))
            return compile_lambda(code, expr);

        if (CAR(expr) == newsym("if")) {
            word test_pos, end_true_pos;

            if (list_length(expr) < 3) {
                warning("if: syntax is (if <bool-expr> "
                        "<true-expr> [<false-expr>])");
                return 0;
            }

            if (!i_compile(code, CAR(CDR(expr)), 0))
                return 0;

            gen(code, OP_JUMP_FALSE);
            test_pos = code->buf->pos;
            gen(code, 0);
            gen(code, 0);

            if (!i_compile(code, CAR(CDR(CDR(expr))), tailpos))
                return 0;

            gen(code, OP_JUMP);
            end_true_pos = code->buf->pos;
            gen(code, 0);
            gen(code, 0);

            patch_word(code, test_pos, code->buf->pos - test_pos);

            if (CDR(CDR(CDR(expr))) != NULL) {
                if (!i_compile(code, CAR(CDR(CDR(CDR(expr)))), tailpos))
                    return 0;
            } else {
                gen(code, OP_LITREF);
                gen(code, get_lit(code, NULL));
            }

            patch_word(code, end_true_pos, code->buf->pos - end_true_pos);

            return 1;
        }

        if (CAR(expr) == newsym("set!")) {
            char snum, ofs;

            if (list_length(expr) < 3) {
                warning("set! has syntax (set! symbol value)");
                return 0;
            }

            if (!instance(CAR(CDR(expr)), symbol_class)) {
                warning("set! expects a symbol as first arg");
                return 0;
            }

            if (!i_compile(code, CAR(CDR(CDR(expr))), 0))
                return 0;

            if (scope_lookup(code, CAR(CDR(expr)), &snum, &ofs)) {
                gen(code, OP_LOCALSET);
                gen(code, snum);
                gen(code, ofs);
                return 1;
            }

            gen(code, OP_GLOBALSET);
            gen(code, get_lit(code, CAR(CDR(expr))));

            return 1;
        }

        if (CAR(expr) == newsym("begin")) {
            OBJECT org, prev;
	    OBJECT curr = CDR(expr);
            int numdefs = 0;
	    int numcmds = 0;

            org = prev = NULL;
            temp_register(&org, 1);

            while (curr != NULL) {
                if (!instance(CAR(curr), pair_class) ||
                    CAR(CAR(curr)) != newsym("define")) {
		  curr = CDR(curr);
		  continue;
		}

                if (!instance(CAR(CDR(CAR(curr))), symbol_class)) {
                    warning("internal define: needs symbol as first arg");
                    deregister_root(1);
                    return 0;
                }

                if (prev == NULL)
                    org = prev = cons(CDR(CAR(curr)), NULL);
                else {
                    SETCDR(prev, cons(CDR(CAR(curr)), NULL));
                    prev = CDR(prev);
                }

                numdefs++;
                curr = CDR(curr);
            }

            if (numdefs > 0) {
                int i;

		curr = org;

                gen(code, OP_ENTERSCOPE);
                gen(code, (char) numdefs);

                code->scope = enter_scope(code, numdefs);

                for (i=0; i<numdefs; i++) {
                    ISET(CAR(code->scope), i, CAR(CAR(curr)));

                    curr = CDR(curr);
                }

                curr = org;

                for (i=0; i<numdefs; i++) {
                    if (!i_compile(code, CAR(CDR(CAR(curr))), 0)) {
                        deregister_root(1);
                        return 0;
                    }

                    gen(code, OP_LOCALSET_NAMED);
                    gen(code, 0);
                    gen(code, i);
                    gen(code, get_lit(code, CAR(CAR(curr))));

                    curr = CDR(curr);
                }
            }

	    curr = CDR(expr);
	    org = prev = NULL;

	    while (curr != NULL) {
	      if (instance(CAR(curr), pair_class) &&
		  CAR(CAR(curr)) == newsym("define")) {
		curr = CDR(curr);
		continue;
	      }

	      if (prev == NULL)
		org = prev = cons(CAR(curr), NULL);
	      else {
		SETCDR(prev, cons(CAR(curr), NULL));
		prev = CDR(prev);
	      }

	      numcmds++;
	      curr = CDR(curr);
	    }

            if (numcmds == 0) {
	      gen(code, OP_LITREF);
	      gen(code, get_lit(code, undefined));
            } else {
	      curr = org;

	      while (CDR(curr) != NULL) {
		if (!i_compile(code, CAR(curr), 0)) {
		  deregister_root(1);
		  return 0;
		}
		
		curr = CDR(curr);
	      }

	      if (!i_compile(code, CAR(curr), tailpos)) {
		deregister_root(1);
		return 0;
	      }
            }

	    deregister_root(1);

            if (numdefs > 0) {
	      code->scope = exit_scope(code);
	      gen(code, OP_EXITSCOPE);
            }

            return 1;
        }

        if (CAR(expr) == newsym("cond"))
            return compile_cond(code, CDR(expr), tailpos);

        if (CAR(expr) == newsym("and"))
            if (CDR(expr) == NULL) {
                gen(code, OP_LITREF);
                gen(code, get_lit(code, true));
                return 1;
            } else
                return compile_and(code, CDR(expr), tailpos);

        if (CAR(expr) == newsym("or"))
            if (CDR(expr) == NULL) {
                gen(code, OP_LITREF);
                gen(code, get_lit(code, false));
                return 1;
            } else
                return compile_or(code, CDR(expr), tailpos);

        if (CAR(expr) == newsym("case"))
            return compile_case(code, CDR(expr), tailpos);

        if (CAR(expr) == newsym("let")) {
            if (list_length(expr) < 3) {
                warning("let: syntax is (let ((var val) ...) body ...)");
                return 0;
            }

            if (instance(CAR(CDR(expr)), symbol_class)) {
                OBJECT org, prev, curr;
                char argc = 0;

                gen(code, OP_ENTERSCOPE);
                gen(code, 1);

                code->scope = enter_scope(code, 1);
                ISET(CAR(code->scope), 0, CAR(CDR(expr)));

                org = prev = NULL;
                temp_register(&org, 1);

                curr = CAR(CDR(CDR(expr)));

                while (curr != NULL) {
                    if (prev == NULL)
                        org = prev = cons(CAR(CAR(curr)), NULL);
                    else {
                        SETCDR(prev, cons(CAR(CAR(curr)), NULL));
                        prev = CDR(prev);
                    }

                    curr = CDR(curr);
                    argc++;
                }

                org = cons(org, CDR(CDR(CDR(expr))));
                org = cons(newsym("lambda"), org);

                if (!compile_lambda(code, org)) {
                    deregister_root(1);
                    return 0;
                }

                deregister_root(1);

                gen(code, OP_LOCALSET_NAMED);
                gen(code, 0);
                gen(code, 0);
                gen(code, get_lit(code, CAR(CDR(expr))));

                curr = CAR(CDR(CDR(expr)));

                while (curr != NULL) {
                    if (!i_compile(code, CAR(CDR(CAR(curr))), 0))
                        return 0;

                    gen(code, OP_PUSH);
                    curr = CDR(curr);
                }

                gen(code, OP_LOCALREF);
                gen(code, 0);
                gen(code, 0);

                gen(code, OP_CALL);
                gen(code, argc);

                code->scope = exit_scope(code);
                gen(code, OP_EXITSCOPE);
            } else {
                char numbindings;
                OBJECT newscope;

                numbindings = (char) list_length(CAR(CDR(expr)));

                if (numbindings > 0) {
                    OBJECT curr;
                    char i;

                    gen(code, OP_ENTERSCOPE);
                    gen(code, numbindings);

                    temp_register(&newscope, 1);

                    newscope = enter_scope(code, numbindings);
                    code->scope = enter_scope(code, numbindings);

                    curr = CAR(CDR(expr));

                    for (i=0; i<numbindings; i++) {
                        if (!i_compile(code, CAR(CDR(CAR(curr))), 0)) {
                            deregister_root(1);
                            return 0;
                        }

                        gen(code, OP_LOCALSET);
                        gen(code, 0);
                        gen(code, i);

                        ISET(CAR(newscope), i, CAR(CAR(curr)));

                        curr = CDR(curr);
                    }

                    code->scope = newscope;

                    deregister_root(1);
                }

		{
		  OBJECT tmp = newsym("begin");
		  int result;
		  temp_register(&tmp, 1);
		  tmp = cons(tmp, CDR(CDR(expr)));
		  result = i_compile(code, tmp, tailpos);
		  deregister_root(1);
		  if (!result)
		    return 0;
		}

                if (numbindings > 0) {
                    code->scope = exit_scope(code);

                    gen(code, OP_EXITSCOPE);
                }
            }

            return 1;
        }

        if (CAR(expr) == newsym("let*")) {
            char numbindings;

            if (list_length(expr) < 3) {
                warning("let*: syntax is (let* ((var val) ...) body ...)");
                return 0;
            }

            numbindings = (char) list_length(CAR(CDR(expr)));

            if (numbindings > 0) {
                OBJECT curr;
                char i;

                gen(code, OP_ENTERSCOPE);
                gen(code, numbindings);

                code->scope = enter_scope(code, numbindings);

                curr = CAR(CDR(expr));

                for (i=0; i<numbindings; i++) {
                    if (!i_compile(code, CAR(CDR(CAR(curr))), 0))
                        return 0;

                    gen(code, OP_LOCALSET);
                    gen(code, 0);
                    gen(code, i);

                    ISET(CAR(code->scope), i, CAR(CAR(curr)));

                    curr = CDR(curr);
                }
            }
	    
	    {
	      OBJECT tmp = newsym("begin");
	      int result;
	      temp_register(&tmp, 1);
	      tmp = cons(tmp, CDR(CDR(expr)));
	      result = i_compile(code, tmp, tailpos);
	      deregister_root(1);
	      if (!result)
		return 0;
	    }

            if (numbindings > 0) {
                code->scope = exit_scope(code);

                gen(code, OP_EXITSCOPE);
            }

            return 1;
        }

        if (CAR(expr) == newsym("letrec")) {
            char numbindings;

            if (list_length(expr) < 3) {
                warning("let: syntax is (let ((var val) ...) body ...)");
                return 0;
            }

            numbindings = (char) list_length(CAR(CDR(expr)));

            if (numbindings > 0) {
                OBJECT curr;
                char i;

                gen(code, OP_ENTERSCOPE);
                gen(code, numbindings);

                code->scope = enter_scope(code, numbindings);

                curr = CAR(CDR(expr));

                for (i = 0; i < numbindings; i++) {
                    ISET(CAR(code->scope), i, CAR(CAR(curr)));
                    curr = CDR(curr);
                }

                curr = CAR(CDR(expr));

                for (i = 0; i < numbindings; i++) {
                    if (!i_compile(code, CAR(CDR(CAR(curr))), 0))
                        return 0;

                    gen(code, OP_LOCALSET_NAMED);
                    gen(code, 0);
                    gen(code, i);
                    gen(code, get_lit(code, CAR(CAR(curr))));

                    curr = CDR(curr);
                }
            }

	    {
	      OBJECT tmp = newsym("begin");
	      int result;
	      temp_register(&tmp, 1);
	      tmp = cons(tmp, CDR(CDR(expr)));
	      result = i_compile(code, tmp, tailpos);
	      deregister_root(1);
	      if (!result)
		return 0;
	    }

            if (numbindings > 0) {
                code->scope = exit_scope(code);

                gen(code, OP_EXITSCOPE);
            }

            return 1;
        }

        if (CAR(expr) == newsym("do")) {
            warning("do is not implemented by the native compiler.");
            return 0;
        }

        if (CAR(expr) == newsym("while")) {
            word test_pos, end_pos;

            if (list_length(expr) < 2) {
                warning("while: syntax is (while <expr> [<expr> ...])");
                return 0;
            }

            test_pos = code->buf->pos;

            if (!i_compile(code, CAR(CDR(expr)), 0))
                return 0;

            gen(code, OP_JUMP_FALSE);
            end_pos = code->buf->pos;
            gen(code, 0);
            gen(code, 0);

	    {
	      OBJECT tmp = newsym("begin");
	      int result;
	      temp_register(&tmp, 1);
	      tmp = cons(tmp, CDR(CDR(expr)));
	      result = i_compile(code, tmp, 0);
	      deregister_root(1);
	      if (!result)
		return 0;
	    }

            gen(code, OP_JUMP);
            {
                word ofs = test_pos - code->buf->pos;
                gen(code, (char) (ofs & 0xFF));
                gen(code, (char) (ofs >> 8));
            }

            patch_word(code, end_pos, code->buf->pos - end_pos);

            return 1;
        }

        if (CAR(expr) == newsym("quasiquote")) {
            warning("quasiquote is not implemented by the native compiler.");
            return 0;
        }

        /* After all those special forms, the only thing a cons can be */
        /* is a function application! */

        {
            OBJECT curr = CDR(expr);
            char argc = (char) list_length(curr);

            while (curr != NULL) {
                if (!i_compile(code, CAR(curr), 0))
                    return 0;
                gen(code, OP_PUSH);

                curr = CDR(curr);
            }

            if (CAR(expr) == newsym("next-method"))
                gen(code, tailpos ? OP_TAIL_CALL_NEXT : OP_CALL_NEXTMETHOD);
            else {
                if (!i_compile(code, CAR(expr), 0))
                    return 0;
                gen(code, tailpos ? OP_TAIL_CALL : OP_CALL);
            }

            gen(code, argc);
        }

        return 1;
    }

    gen(code, OP_LITREF);
    gen(code, get_lit(code, expr));

    return 1;
}

OBJECT compile(OBJECT expr) {
    CODE code;
    OBJECT template;

    if (expr == undefined)
        return NULL;            /* End of file signaled */

    code = newcode();

    temp_register(&expr, 1);

    if (i_compile(code, expr, 1)) {
        gen(code, OP_RETURN);

        template = make_template(code, 0);
    } else
        template = NULL;

    deregister_root(1);

    killcode(code);

    return template;
}

