/* $Id: memory.c,v 1.8 1998/04/05 10:33:48 tonyg Exp $ */

#include "memory.h"
#include "class.h"

#include "object.h"
#include "null.h"
#include "integer.h"
#include "symbol.h"
#include "string.h"
#include "pair.h"
#include "vector.h"
#include "function.h"
#include "prim.h"
#include "gf.h"
#include "thread.h"
#include "misc.h"
#include "float.h"

#include "stream.h"
#include "file.h"
#include "socket.h"
#include "xwindows.h"

#include <string.h>
#include <stdarg.h>
#include <setjmp.h>

#ifdef unix
#  include <sys/time.h>
#  include <sys/resource.h>
#endif

#if WANT_FIXED_HEAP
PRIVATE long max_bytes_alloced;
PRIVATE long curr_bytes_alloced;
#endif

int debug_flag;
volatile OBJECT signal_raised;

OBJECT null_class, int_class, object_class;

OBJECT true, false, undefined;
OBJECT currthr;

/* Stuff to do with memory allocation and GC: */

#define ROOTPAGE_SIZE   32

typedef struct Rootpage *ROOTPAGE;
typedef struct Rootpage {
    ROOTPAGE prev;
    int top;
    struct {
        OBJECT *loc;
        int size;
    } root[ROOTPAGE_SIZE];
} Rootpage;

PRIVATE ROOTPAGE all_roots;
PRIVATE ROOTPAGE temp_roots;

/* Error handling */

void fatal(char *format, ...) {
    va_list vl;

    fprintf(stderr, "FATAL ERROR: ");

    va_start(vl, format);
    vfprintf(stderr, format, vl);
    va_end(vl);

    fputc('\n', stderr);

    exit(1);
}

void error(char *format, ...) {
    va_list vl;

    fprintf(stderr, "\nERROR: ");

    va_start(vl, format);
    vfprintf(stderr, format, vl);
    va_end(vl);

    fputc('\n', stderr);

    dump_context_on(stdout, NULL);

    exit(2);
}

/* Memory allocation and deallocation */

void init_memory(long maxbytes, int dflag) {
    debug_flag = dflag;
    signal_raised = NULL;

#if WANT_FIXED_HEAP
    max_bytes_alloced = maxbytes;
    curr_bytes_alloced = 0;
#endif

#ifdef unix
    {
        struct rlimit rlp;

        getrlimit(RLIMIT_DATA, &rlp);

        if (maxbytes > rlp.rlim_max)
            rlp.rlim_cur = rlp.rlim_max;
        else
            rlp.rlim_cur = maxbytes;

        setrlimit(RLIMIT_DATA, &rlp);
    }
#endif

    all_objects = NULL;
    all_roots = NULL;
    temp_roots = NULL;

    object_class = getmem(sizeof(Object) +
                          CLS_SIZE * sizeof(OBJECT) +
                          2 * sizeof(char));
    register_root(&object_class, 1);

    SETNEXTOBJ(object_class, all_objects);
    all_objects = object_class;

    SETMARK(object_class, 0);
    SETNUMIDX(object_class, 0);
    SETNUMBIN(object_class, 0);

    init_class();

    SETCLASS(object_class, class_class);
    SET(object_class, CLS_NUMIVAR, MKNUM(OBJECT_SIZE));
    SET(object_class, CLS_SUPER, NULL);
    SET(object_class, CLS_IVARNAMES, NULL);

    BSET(object_class, 0, 2);
    BSET(object_class, 1, 0);

    /* Now that <object> and <class> exist properly, we can set up the rest
       of the built-in classes. Don't forget to register them. */

    init_object();
    init_null();
    init_integer();
    init_symbol();
    init_string();
    init_pair();
    init_vector();
    init_function();
    init_prim();
    init_gf();
    init_thread();
    init_float();

    init_stream();
    init_file();
    init_socket();
    init_xwindows();

    /* Now that all the classes are defined, we can create the singletons
       necessary to the runtime system. We can also add a list of instvarnames
       to <class>. */

    true = false = undefined = NULL;
    
    register_root(&true, 1);
    register_root(&false, 1);
    register_root(&undefined, 1);

        /* Note: no symbols until now have been defined, so make sure
           undefined is defined first, so that all other globally-unbound
           symbols can have undefined as their SYM_VALUE. */

    undefined = newstring("#<undefined>");

    true = newsym("#t");
    false = newsym("#f");
    SET(true, SYM_VALUE, true);
    SET(false, SYM_VALUE, false);

    SET(class_class, CLS_IVARNAMES,
        cons(newsym("super-class"),
            cons(newsym("instance-variable-count"),
                cons(newsym("instance-variable-names"), NULL))));

    /* Next, register methods on the built-in classes. */

    /* First, create the '*features* list. */

    SET(newsym("*features*"), SYM_VALUE, NULL);

    init_meth_class();
    init_meth_object();
    init_meth_null();
    init_meth_integer();
    init_meth_symbol();
    init_meth_string();
    init_meth_pair();
    init_meth_vector();
    init_meth_function();
    init_meth_prim();
    init_meth_gf();
    init_meth_thread();
    init_meth_misc();
    init_meth_float();

    init_meth_stream();
    init_meth_file();
    init_meth_socket();
    init_meth_xwindows();

    /* Now, make base classes available to the user. */

    SET(newsym("<class>"), SYM_VALUE, class_class);
    SET(newsym("<object>"), SYM_VALUE, object_class);
    SET(newsym("<null>"), SYM_VALUE, null_class);
    SET(newsym("<integer>"), SYM_VALUE, int_class);
    SET(newsym("<symbol>"), SYM_VALUE, symbol_class);
    SET(newsym("<string>"), SYM_VALUE, string_class);
    SET(newsym("<pair>"), SYM_VALUE, pair_class);
    SET(newsym("<vector>"), SYM_VALUE, vector_class);
    SET(newsym("<function>"), SYM_VALUE, function_class);
    SET(newsym("<primitive-function>"), SYM_VALUE, prim_class);
    SET(newsym("<generic-function>"), SYM_VALUE, gf_class);
    SET(newsym("<thread>"), SYM_VALUE, thread_class);
    SET(newsym("<continuation>"), SYM_VALUE, continuation_class);
    SET(newsym("<float>"), SYM_VALUE, float_class);
}

OBJECT all_objects;

void *getmem(word size) {
    void *p;

    if (size == 0)
      return NULL;

#if WANT_FIXED_HEAP
    if (curr_bytes_alloced + size >= max_bytes_alloced)
      gc();
#endif

    p = malloc(size);

    if (p != NULL)
        return p;

    gc();

    p = malloc(size);

    if (p != NULL)
        return p;

    fatal("Out of memory.\n");
    return NULL;
}

void *growmem(void *oldblk, word oldsize, int incr) {
    void *blk = getmem(oldsize + incr);

    if (oldblk != NULL) {
        memcpy(blk, oldblk, oldsize);
        freemem(oldblk);
    }

    return blk;
}

void freemem(void *blk) {
    free(blk);
}

void register_root(OBJECT *root, int size) {
    if (all_roots == NULL || all_roots->top == ROOTPAGE_SIZE) {
        ROOTPAGE page = (ROOTPAGE) malloc(sizeof(Rootpage));

        if (page == NULL)
            fatal("Out of space for rootpages.");

        page->prev = all_roots;
        all_roots = page;

        page->top = 0;
    }

    all_roots->root[all_roots->top].loc = root;
    all_roots->root[all_roots->top].size = size;

    all_roots->top++;
}

void temp_register(OBJECT *root, int size) {
    if (temp_roots == NULL || temp_roots->top == ROOTPAGE_SIZE) {
        ROOTPAGE page = (ROOTPAGE) malloc(sizeof(Rootpage));

        if (page == NULL)
            fatal("Out of space for temporary rootpages.");

        page->prev = temp_roots;
        temp_roots = page;

        page->top = 0;
    }

    temp_roots->root[temp_roots->top].loc = root;
    temp_roots->root[temp_roots->top].size = size;

    temp_roots->top++;
}

void deregister_root(int n) {
    while (n > 0) {
        if (temp_roots == NULL)
            return;

        temp_roots->top--;

        if (temp_roots->top == 0) {
            ROOTPAGE page = temp_roots;
            temp_roots = temp_roots->prev;
            free(page);
        }

        n--;
    }
}

PRIVATE void mark(OBJECT obj) {
    word i, size;

    while (1) {
        if (obj == NULL)
            return;

        if (GETCLASS(obj) == int_class)
            return;

        if (MARK(obj))
            return;

        SETMARK(obj, 1);

        size = CLSSIZE(obj) + NUMIDX(obj);

	if (size > 0) {
	  mark(CLASS(obj));
	  for (i = 0; i < size - 1; i++)
            mark(GET(obj, i));

	  obj = GET(obj, size - 1);
	} else
	  obj = CLASS(obj);
    }
}

void gc(void) {
    ROOTPAGE page;
    OBJECT obj, temp;
    int i;

    page = all_roots;

    while (page != NULL) {
        int i, j;

        for (i = 0; i < page->top; i++)
            for (j = 0; j < page->root[i].size; j++)
                mark(page->root[i].loc[j]);

        page = page->prev;
    }

    page = temp_roots;

    while (page != NULL) {
        int i, j;

        for (i = 0; i < page->top; i++)
            for (j = 0; j < page->root[i].size; j++)
                mark(page->root[i].loc[j]);

        page = page->prev;
    }

    for (i=0; i<SYMTAB_SIZE; i++)
        if (symtab[i] != NULL) {
            OBJECT sym = symtab[i];

            while (sym != NULL) {
                if (GET(sym, SYM_VALUE) != undefined)
                    mark(sym);

                sym = GET(sym, SYM_NEXT);
            }

            symtab[i] = NULL;
        }

    obj = all_objects;
    all_objects = NULL;

    while (obj != NULL) {
        temp = NEXTOBJ(obj);

        if (MARK(obj)) {
	  SETMARK(obj, 0);
	  SETNEXTOBJ(obj, all_objects);
	  all_objects = obj;
	    
	  if (instance(obj, symbol_class))
	    register_symbol(obj);
        } else {
#if WANT_FIXED_HEAP
	  curr_bytes_alloced -= sizeof(Object) +
	    (word) NUM(GET(CLASS(obj), CLS_NUMIVAR)) * sizeof(OBJECT) +
	      NUMIDX(obj) * sizeof(OBJECT) +
		NUMBIN(obj) * sizeof(char);
#endif
	  freemem(obj);
	}

        obj = temp;
    }
}

OBJECT NewObject(OBJECT class, word numidx, word numbin) {
    word size = (word) NUM(GET(class, CLS_NUMIVAR)) * sizeof(OBJECT) +
                numidx * sizeof(OBJECT) +
                numbin * sizeof(char);
    OBJECT obj = getmem(sizeof(Object) + size);

#if WANT_FIXED_HEAP
    curr_bytes_alloced += sizeof(Object) + size;
#endif

    SETNEXTOBJ(obj, all_objects);
    all_objects = obj;

    SETMARK(obj, 0);
    SETNUMIDX(obj, numidx);
    SETNUMBIN(obj, numbin);
    SETCLASS(obj, class);

    memset((char *) (&CLASS(obj) + 1), 0, size);

    return obj;
}

int subclass(OBJECT class, OBJECT super) {
    while (class != NULL) {
        if (class == super)
            return 1;

        class = GET(class, CLS_SUPER);
    }

    return 0;
}

