#include "oom.h"

#include "serial.h"
#include "version.h"

#include <stdlib.h>
#include <setjmp.h>

PRIVATE int only_coding;        /* Is this session only for coding the db? */
PRIVATE int db_initialised;     /* Has DB been "booted" yet? */

void banner(int usage) {
    printf(
        "============ OOM v"__VERSION__" Copyright (C) 1994 by Tony Garnock-Jones ============\n"
    );
    if (usage) printf(

"Usage: OOM <infile> <outfile> /f name [ { /c | /p n bbbb [/p n bbbb ...] } ]\n"
"\n"
"   <infile>        Database to read from\n"
"   <outfile>       Database to write to\n"
"   /p n bbbb       Allow connections from port n at baud bbbb (8,N,1)\n"
"   /c              Code the database only (no verb-lookups)\n"
"   /f name         Create a (temporary) connection to file \"name\"\n"
"\n"
"   In every case, console connections are always active.\n"

    );
}

void create_minimal_db(void) {
    OBJECT sysobj, rootclass, room1, wizard;
    OBJINFO o;

    sysobj = newobj(NULL,NULL);
    rootclass = newobj(NULL,NULL);
    room1 = newobj(rootclass, NULL);
    wizard = newobj(rootclass, NULL);

    o = findobj(GETONUM(sysobj));
    o->parent = rootclass;
    o->owner = wizard;
    o->location = MKONUM(-1);

    o = findobj(GETONUM(room1));
    o->owner = wizard;
    o->contents = cons(wizard, NULL);
    o->location = MKONUM(-1);

    o = findobj(GETONUM(wizard));
    o->owner = wizard;
    o->flags |= FO_PLAYER | FO_PROGRAMMER | FO_WIZARD;
    o->location = room1;

    o = findobj(GETONUM(rootclass));
    o->owner = wizard;
    o->children = cons(sysobj, o->children);
    o->location = MKONUM(-1);
}

extern void code_scanner(SCANSTATE s);
extern void mud_scanner(SCANSTATE s);

void coding_repl(void) {
    SCANSTATE s;
    PARSER p;

    c_reset(curinfo->conn);

    curinfo->player = MKONUM(3);
    curinfo->stack = newstack(INTERP_OBJSTACKSIZE);
    curinfo->ticks_left = -1;

    s = newscanner("CONSOLE", curinfo->conn);
    p = newparser(code_scanner, s);

    if (setjmp(curinfo->errjmp) != 0) {
        while (!check(p, ";")) parse_drop(p);
        parse_drop(p);
    }
    curinfo->errjmp_ok = 1;

    format("> ");
    while (1) {
        CODE code;

        if (shutting_down) break;

        if (check(p, "end-of-file")) {
            parse_drop(p);
            break;
        }

        code = parser(p);

        if (code == NULL) {
            format("\n> ");
            continue;
        }

        curinfo->code_owner = curinfo->player;
        curinfo->caller = NULL;
        curinfo->errstk = NULL;
        interp(NULL, code->buf, NULL, NULL, NULL);

        kill_code(code);

        while (stack_size(curinfo->stack) > 0)
            print(pop(curinfo->stack));
        format("\n> ");
    }

    killparser(p);
    killscanner(s);

    curinfo->player = NULL;
    curinfo->code_owner = NULL;
    curinfo->stack = NULL;
}

void get_mud_sentence(PARSER p, OBJECT *cmd, OBJECT *dir,
                      OBJECT *prep, OBJECT *indir) {
    OBJECT curr;
    *cmd = *dir = *prep = *indir = NULL;

    if (!check(p,"word") && !check(p,"preposition")) {
        parse_drop(p);
        return;
    }

    parse_drop(p);
    *cmd = cons(newstring(p->scanner->d.s), NULL);

    curr = NULL;
    while (1) {
        if (!check(p,"word") && !check(p,"preposition")) {
            parse_drop(p);
            return;
        }
        if (check(p, "preposition")) {
            parse_drop(p);
            break;
        }
        parse_drop(p);
        if (curr == NULL)
            *dir = curr = cons(newstring(p->scanner->d.s),NULL);
        else {
            SETCDR(curr, cons(newstring(p->scanner->d.s), NULL));
            curr = CDR(curr);
        }
    }

    *prep = cons(newstring(p->scanner->d.s), NULL);

    curr = NULL;
    while (check(p, "word") || check(p, "preposition")) {
        parse_drop(p);
        if (curr == NULL)
            *indir = curr = cons(newstring(p->scanner->d.s), NULL);
        else {
            SETCDR(curr, cons(newstring(p->scanner->d.s), NULL));
            curr = CDR(curr);
        }
    }
    parse_drop(p);
}

OBJECT getplayer(PARSER p) {
    OBJECT cmd, dir, prep, indir;
    OBJECT player = NULL;

    curinfo->logging_in = 1;

    curinfo->stack = newstack(INTERP_OBJSTACKSIZE);
    while (1) {
        OBJECT temp;

        if (check(p, "end-of-file")) {
            parse_drop(p);
            curinfo->stack = NULL;
            curinfo->logging_in = 0;
            return NULL;
        }

        get_mud_sentence(p, &cmd, &dir, &prep, &indir);

        temp = curinfo->code_owner;
        curinfo->player = curinfo->code_owner = MKONUM(3);

        player = callmethod(MKONUM(0),"do-login-command",4,cmd,dir,prep,indir);

        curinfo->player = NULL;
        curinfo->code_owner = temp;

        if (get_type(player) == T_OBJECT &&
            (findobj(GETONUM(player))->flags & FO_PLAYER) != 0)
                break;
    }
    curinfo->stack = NULL;
    curinfo->logging_in = 0;

    return player;
}

#define WIZARD(obj)     ((findobj(GETONUM(obj))->flags & FO_WIZARD) != 0)
#define PROGRAMMER(obj) ((findobj(GETONUM(obj))->flags & FO_PROGRAMMER) != 0)

int domudcommand(PARSER p) {
    OBJECT cmd, dir, prep, indir, res;

    cmd = dir = prep = indir = NULL;
    curinfo->stack = newstack(INTERP_OBJSTACKSIZE);

    if (check(p, "emote")) {
        parse_drop(p);

        cmd = cons(newstring("emote"),NULL);
        dir = cons(newstring(p->scanner->d.s),NULL);
    } else if (check(p, "say")) {
        parse_drop(p);

        cmd = cons(newstring("say"),NULL);
        dir = cons(newstring(p->scanner->d.s),NULL);
    } else if (check(p, "eval")) {
        PARSER ep;
        CODE code;

        parse_drop(p);

        if (!PROGRAMMER(curinfo->player) && !WIZARD(curinfo->player)) {
            while (!check(p, "end-of-line")) parse_drop(p);
            parse_drop(p);
            curinfo->stack = NULL;
            return 0;
        }

        ep = newparser(code_scanner, p->scanner);
        code = parser(ep);

        if (code != NULL) {
            curinfo->code_owner = curinfo->player;
            curinfo->caller = NULL;
            curinfo->errstk = NULL;
            interp(NULL, code->buf, NULL, NULL, NULL);
            kill_code(code);
        }

        killparser(ep);

        while (stack_size(curinfo->stack) > 0) {
            format("==> ");
            print(pop(curinfo->stack));
            format("\n");
        }

        curinfo->stack = NULL;
        return 1;
    } else get_mud_sentence(p, &cmd, &dir, &prep, &indir);

    res = callmethod(MKONUM(0), "do-mud-command", 4, cmd, dir, prep, indir);
    curinfo->stack = NULL;

    if (res != false) return 1; else return 0;
}

void mud_repl(void) {
    SCANSTATE s;
    PARSER p;

    c_reset(curinfo->conn);
    s = newscanner("CONSOLE", curinfo->conn);
    p = newparser(mud_scanner, s);

    setjmp(curinfo->errjmp);
    curinfo->errjmp_ok = 1;

    curinfo->stack = newstack(INTERP_OBJSTACKSIZE);
    curinfo->ticks_left = -1;

    if (!db_initialised) {
        db_initialised = 1;
        curinfo->code_owner = curinfo->player = MKONUM(3);
        callmethod(MKONUM(0), "db-initialize", 0);
        curinfo->code_owner = curinfo->player = NULL;
    }

    curinfo->code_owner = curinfo->player = getplayer(p);

    if (curinfo->player == NULL) {      /* end-of-file */
        killparser(p);
        killscanner(s);
        return;
    }

    callmethod(MKONUM(0), "do-mud-command", 4, NULL, NULL, NULL, NULL);

    curinfo->stack = NULL;

    setjmp(curinfo->errjmp);
    curinfo->errjmp_ok = 1;

    while (1) {
        if (shutting_down) break;

        if (check(p, "end-of-file")) {
            parse_drop(p);
            break;
        }

        curinfo->code_owner = curinfo->player;
        if (!domudcommand(p)) format("I don't understand that.\n");
    }

    killparser(p);
    killscanner(s);

    if (setjmp(curinfo->errjmp) == 0) {
        OBJECT temp = curinfo->player;
        curinfo->code_owner = curinfo->player = MKONUM(3);
        curinfo->stack = newstack(INTERP_OBJSTACKSIZE);
        callmethod(MKONUM(0), "do-logout", 1, temp);
        curinfo->stack = NULL;
    }

    curinfo->player = NULL;
    curinfo->code_owner = NULL;
}

void repl(void) {
    if (only_coding) coding_repl(); else mud_repl();
}

void make_conn(OBJECT conn) {
    curinfo->conn = conn;
    while (!shutting_down) repl();
}

void make_temp_conn(OBJECT conn) {
    curinfo->conn = conn;
    repl();     /* Only do this until the first end-of-file */
}

void make_serial_conn(OBJECT conn) {
    int pn;
    long br;

    pn = GETCINFO(conn);
    br = GETCONNL(conn);

    while (!shutting_down) {
        curinfo->conn = conn;
        serial_reset(conn);     /* Waits for carrier */
        if (shutting_down) break;
        fork(make_conn, conn, INTERP_STACKSIZE);
        while (serial_carrier(conn) && !shutting_down) swap();
        if (!shutting_down) conn = newconn(CONN_SERIAL, pn, (void *) br);
    }
}

void parse_switches(char *sw[]) {
    OBJECT conn;

    while (*sw) {
        char *s = *sw;
        sw++;
        if (*s != '/' && *s != '-') fatal("Invalid command line switch: %s",s);
        s++;
        switch (*s) {
            case 'f':
            case 'F':
                log("compiling file ~S into database",*sw);
                conn = newconn(CONN_FILE, 0, *sw++);
                fork(make_temp_conn, conn, INTERP_STACKSIZE);
                break;
            case 'c':
            case 'C':
                log("this session is coding-only");
                only_coding = 1;
                break;
            case 'p':
            case 'P': {
                int pn;
                long br;

                pn = atoi(*sw++);
                br = atol(*sw++);

                log("session allowing connects port ~N at ~N",
                    (long) pn,
                    (long) br);
                conn = newconn(CONN_SERIAL, pn, (void *) br);
                fork(make_serial_conn, conn, 512);
                break;
            }
            default:
                fatal("Invalid command line switch: %s",s-1);
        }
    }
}

void main(int argc, char *argv[]) {
    FILE *f;

    only_coding = 0;
    db_initialised = 0;

    init_log("oom.log");
    init_error();
    init_memory();
    init_symtab();
    init_process();
    init_connect();
    atexit(done_connect);

    init_interp();
    init_prim();

    if (argc < 3) {
        banner(1);
        exit(2);
    } else banner(0);

    if (argv[1][0] == '-') {
        log("creating minimal database on ~S", argv[2]);

        create_minimal_db();

        f = fopen(argv[2], "wb");
        dump_on(f);
        fclose(f);

        exit(0);
    }

    log("loading from ~S, saving to ~S", argv[1], argv[2]);

    f = fopen(argv[1], "rb");
    load_from(f);
    fclose(f);

    parse_switches(argv + 3);

    fork(make_conn, newconn(CONN_CONSOLE, 0, NULL), INTERP_STACKSIZE);

    while (getproccount() > 1) swap();

    f = fopen(argv[2], "wb");
    dump_on(f);
    fclose(f);
}

