#include "oom.h"

#include <alloc.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/*------------------------------------------------------------------------*/

PRIVATE OBJECT freelist;
PRIVATE long freelist_size;

void *getmem(word size) {
    void *p;

    p = malloc(size);
    if (p != NULL) return p;

    gc();
    p = malloc(size);
    if (p != NULL) return p;

    fatal("Out of memory (allocating: %u)", size);
    return NULL;
}

void *growmem(void *oldblk, word oldsize, int incr) {
    void *newblk = getmem(oldsize + incr);
    if (oldblk != NULL) memcpy(newblk, oldblk, oldsize);
    freemem(oldblk);
    return newblk;
}

void freemem(void *mem) {
    free(mem);
}

#define OBJPAGELEN  (1024 / sizeof(_object))
                /* 4096 minus 4 bytes for the NEXT pointer. */

PRIVATE struct __objpage {
    struct __objpage *next;
    _object object[OBJPAGELEN];
} *objpages;

PRIVATE int objpagecount;

PRIVATE OBJECT newOBJECT(void) {
    OBJECT t;
    if (freelist == NULL) {
        gc();
        if (freelist_size < (OBJPAGELEN >> 1)) {
            word i;
            struct __objpage *p;
            p = getmem(sizeof(struct __objpage));
            p->next = objpages;
            objpages = p;
            objpagecount++;
            for (i=0; i<OBJPAGELEN; i++) {
                p->object[i].d.cons.car = freelist;
                p->object[i].type = T_FREELIST;
                freelist = &(p->object[i]);
            }
        }
    }
    t = freelist;
    freelist = t->d.cons.car;
    return t;
}

#define PTRLT(x,y)      ((long)(x) < (long)(y))
#define PTRGT(x,y)      PTRLT(y,x)
#define PTRGE(x,y)      (!PTRLT(x,y))
#define PTRLE(x,y)      (!PTRGT(x,y))

PRIVATE void mark(OBJECT obj) {
    int a;

    while (1) {
        if (obj == NULL) return;

        a = get_type(obj);
        if (a & T_MARK) return;
        if (a == T_INT) return;
        if (a == T_OBJECT) return;
        if (a == T_PRIM) return;
        obj->type |= T_MARK;

        switch (a) {
            case T_CONS:
                mark(CAR(obj));
                obj = CDR(obj);
                continue;
            case T_FUNC:
                obj = GETENV(obj);
                continue;
            case T_VECTOR: {
                int i;
                for (i = 0; i < GETLEN(obj) - 1; i++)
                    mark(GETELEM(obj,i));
                obj = GETELEM(obj,GETLEN(obj) - 1);
                continue;
            }
            case T_CONN:
                connections[GETCTYPE(obj)].mark(obj);
                break;
        }
        return;
    }
}

PRIVATE void mark_array(OBJECT *start, OBJECT *end) {
    OBJECT tmp;
    struct __objpage *page;
    word i, n;

    if (PTRGT(start, end)) return;

    n = (word) (end - start) + 1;

    for (i=0; i<n; i++) {
        tmp = start[i];
        page = objpages;
        while (page != NULL) {
            if (PTRGE(tmp, &(page->object[0])) &&
                PTRLT(tmp, &(page->object[OBJPAGELEN])) &&
                (((((char *)tmp) - ((char *)(&(page->object[0])))) %
                    sizeof(_object)) == 0)) {
                        mark(tmp);
                        break;
                    }
            page = page->next;
        }
    }
}

#define MKFP(s,o)   ((void far *) (((long) s << 16) | ((long) o)))

PRIVATE void mark_processes(void) {
    PROCESS p = getproclist();
    char *stkbot;

    while (p != NULL) {
        if (p == curproc)
            stkbot = (char *) &stkbot; else
            stkbot = MKFP(p->jb->j_ss, p->jb->j_sp);
        if (p->stack.base != NULL) {
            mark_array((OBJECT *) stkbot,
                       (OBJECT *) p->stack.top);
            mark_array((OBJECT *) (stkbot + 2),
                       (OBJECT *) (((char *) p->stack.top) + 2));
        }
        if (p->info != NULL) {
            mark(p->info->player);
            mark(p->info->conn);
            mark(p->info->stack);
        }
        p = p->next;
    }
}

PRIVATE void killobj(OBJECT obj) {
    switch (obj->type) {
        case T_STRING:
        case T_SYM:
            freemem(GETSTR(obj));
            break;
        case T_VECTOR:
            freemem(obj->d.vector.vec);
            break;
        case T_CONN:
            connections[GETCTYPE(obj)].close(obj);
            break;
        case T_FUNC:
            freemem(obj->d.function.body);
            break;
    }
}

PRIVATE void sweep(void) {
    struct __objpage *page;
    word i;

    freelist = NULL;
    freelist_size = 0;
    page = objpages;
    while (page != NULL) {
        for (i=0; i<OBJPAGELEN; i++)
            if ((page->object[i].type & T_MARK) == 0) {
                killobj(&(page->object[i]));
                page->object[i].d.cons.car = freelist;
                page->object[i].type = T_FREELIST;
                freelist = &(page->object[i]);
                freelist_size++;
            } else page->object[i].type &= ~T_MARK;
        page = page->next;
    }
}

PRIVATE void mark_objtbl(void); /* Forward declaration */

void gc(void) {
    mark(true);
    mark(false);
    mark_objtbl();
    mark_processes();
    mark(shutdown_tasks);
    foreachglb(mark);
    sweep();
}

/*==============================    DUMPING    ==========================*/

PRIVATE long find_obj_num(OBJECT obj) {
    struct __objpage *page;
    long i;

    i = 0;
    page = objpages;
    while (page != NULL) {
        long n = ((long)obj - (long)page->object) / sizeof(_object);

        if (n >= 0 && n < OBJPAGELEN) return i+n;

        i += OBJPAGELEN;
        page = page->next;
    }
    fatal("Object didn't map to a page!");
    return -1;
}

PRIVATE void dump_objnum(OBJECT obj, FILE *f) {
    switch (get_type(obj)) {
        case T_NULL:
            fputc(0, f);
            break;
        case T_INT:
        case T_OBJECT:
        case T_PRIM:
            fputc(1, f);
            fwrite(&((long)obj), sizeof(long), 1, f);
            break;
        default: {
            long n;
            if (obj == true) fputc(3, f); else
                if (obj == false) fputc(4, f); else {
                    fputc(2, f);
                    n = find_obj_num(obj);
                    fwrite(&n, sizeof(long), 1, f);
                }
        }
    }
}

PRIVATE void dump_object(long n, OBJECT obj, FILE *f) {
    if (obj == true) return;
    if (obj == false) return;
    fwrite(&n, sizeof(long), 1, f);
    fwrite(&(obj->type), sizeof(char), 1, f);
    switch (obj->type) {
        case T_STRING: {
            word l = strlen(GETSTR(obj));
            fwrite(&l, sizeof(word), 1, f);
            /* FALL THROUGH */
        }
        case T_SYM:
            fputs(GETSTR(obj), f);
            fputc(0, f);
            break;
        case T_CONS:
            dump_objnum(CAR(obj), f);
            dump_objnum(CDR(obj), f);
            break;
        case T_FUNC: {
            char *ip = GETBODY(obj);
            word count = *(word *)(ip+1);

            dump_objnum(GETENV(obj), f);
            fputc(*ip, f);
            ip += 3;
            fwrite(&count, sizeof(word), 1, f);
            while (count > 0) {
                fputc(*ip++, f);
                count--;
            }
            break;
        }
        case T_VECTOR: {
            long i;
            i = GETLEN(obj);
            fwrite(&i, sizeof(long), 1, f);
            for (i=0; i<GETLEN(obj); i++) dump_objnum(GETELEM(obj,(word)i), f);
            break;
        }
        case T_CONN: {
            word i;
            i = GETCTYPE(obj);
            fwrite(&i, sizeof(word), 1, f);
            i = GETCINFO(obj);
            fwrite(&i, sizeof(word), 1, f);
            connections[GETCTYPE(obj)].dump(obj, f);
            break;
        }
    }
}

PRIVATE void dump_objects(FILE *f) {
    long i,b;
    struct __objpage *page;

    b = 0;
    page = objpages;
    while (page != NULL) {
        for (i=0; i<OBJPAGELEN; i++)
            if (page->object[(word) i].type != T_FREELIST &&
                page->object[(word) i].type != T_PRIM)
                    dump_object(i+b, &(page->object[(word) i]), f);
        b += OBJPAGELEN;
        page = page->next;
    }
    i = -1;
    fwrite(&i, sizeof(long), 1, f);
}

PRIVATE void dump_objtbl(FILE *f);  /* Forward reference */

void dump_on(FILE *f) {
    long i;

    gc();

    fputc(0xC0, f);
    fputc(0xDE, f); /* 0xC0DE */

    fputs("OOM100", f);
    fputc(0, f);

    i = objpagecount * OBJPAGELEN;
    fwrite(&i, sizeof(long), 1, f);

    dump_objects(f);
    dump_objtbl(f);
    fwrite(&next_task_id, sizeof(long), 1, f);
    dump_objnum(shutdown_tasks, f);
}

PRIVATE OBJECT find_num_obj(long n) {
    struct __objpage *p;

    p = objpages;
    while (n >= OBJPAGELEN) {
        p = p->next;
        n -= OBJPAGELEN;
    }

    return &(p->object[(word) n]);
}

PRIVATE OBJECT load_objnum(FILE *f) {
    char type = fgetc(f);
    switch (type) {
        case 0: return NULL;
        case 1: {
            long i;
            fread(&i, sizeof(long), 1, f);
            return (OBJECT) i;
        }
        case 2: {
            long i;
            fread(&i, sizeof(long), 1, f);
            return find_num_obj(i);
        }
        case 3: return true;
        case 4: return false;
    }
    return NULL;
}

PRIVATE void load_object(long i, FILE *f) {
    OBJECT x = find_num_obj(i);

    x->type = fgetc(f);
    switch (x->type) {
        case T_STRING:
        case T_SYM: {
            char *s, *p;
            if (x->type == T_STRING) {
                word l;
                fread(&l, sizeof(word), 1, f);
                s = getmem(l+1);
            } else s = getmem(256);
            p = s;
            do {
                *p = fgetc(f);
                p++;
            } while (*(p-1) != '\0');
            x->d.strsym.data = getmem(strlen(s)+1);
            strcpy(x->d.strsym.data, s);
            x->d.strsym.hash = hash_str(s);
            freemem(s);
            break;
        }
        case T_CONS:
            SETCAR(x, load_objnum(f));
            SETCDR(x, load_objnum(f));
            break;
        case T_FUNC: {
            char argc;
            word len;

            x->d.function.env = load_objnum(f);
            argc = fgetc(f);
            fread(&len, sizeof(word), 1, f);
            x->d.function.body = getmem(len + 3);
            x->d.function.body[0] = argc;
            x->d.function.body[1] = (char) (len & 0xFF);
            x->d.function.body[2] = (char) (len >> 8);
            fread(x->d.function.body + 3, sizeof(char), len, f);
            break;
        }
        case T_VECTOR: {
            long i;

            fread(&i, sizeof(long), 1, f);
            x->d.vector.length = i;
            x->d.vector.vec = getmem(GETLEN(x) * sizeof(OBJECT));
            for (i=0; i<GETLEN(x); i++)
                SETELEM(x,(word) i,load_objnum(f));
            break;
        }
        case T_CONN: {
            word t,i;
            fread(&t, sizeof(word), 1, f);
            fread(&i, sizeof(word), 1, f);
            x->d.connection.type = t;
            x->d.connection.info = i;
            connections[t].load(x, f);
            break;
        }
    }
    x->type |= T_MARK;
}

PRIVATE void load_oo_object(long num, FILE *f); /* Forward declaration */
PRIVATE void sweep_objtbl(void);    /* Forward declaration */
PRIVATE long nextnumber;

void load_from(FILE *f) {
    long i;
    struct __objpage *p;

    gc();

    if (fgetc(f) != 0xC0) return;
    if (fgetc(f) != 0xDE) return;

    while (fgetc(f) != 0);

    fread(&i, sizeof(long), 1, f);
    i += OBJPAGELEN - 1;
    i /= OBJPAGELEN;

    sweep();            /* Kills all non-oo objects */
    sweep_objtbl();     /* Kills all oo objects */
    sweepglbtab();      /* Kills all in global-symtab (prims) */

    while (objpagecount > i) {
        p = objpages->next;
        freemem(objpages);
        objpages = p;
        objpagecount--;
    }

    while (objpagecount < i) {
        word i;

        p = getmem(sizeof(struct __objpage));
        p->next = objpages;
        objpages = p;
        for (i=0; i<OBJPAGELEN; i++)
            p->object[i].type = T_FREELIST;
        objpagecount++;
    }

    init_interp();
    mark(true);
    mark(false);

    fread(&i, sizeof(long), 1, f);
    while (i != -1) {
        load_object(i, f);                  /* Loads + marks */
        fread(&i, sizeof(long), 1, f);
    }

    fread(&nextnumber, sizeof(long), 1, f); /* Loads next object number */

    fread(&i, sizeof(long), 1, f);
    while (i != -1) {
        load_oo_object(i, f);
        fread(&i, sizeof(long), 1, f);
    }

    fread(&next_task_id, sizeof(long), 1, f);
    shutdown_tasks = load_objnum(f);
    while (shutdown_tasks != NULL) {
        OBJECT player, code_owner, conn, id, args;
        OBJECT task = CAR(shutdown_tasks);
        PROCINFO pi;

        player = CAR(task);
        code_owner = CAR(CDR(task));
        conn = CAR(CDR(CDR(task)));
        id = CAR(CDR(CDR(CDR(task))));
        args = CDR(CDR(CDR(CDR(task))));

        pi = fork(fork_after, args, INTERP_STACKSIZE);
        pi->player = player;
        pi->code_owner = code_owner;
        pi->conn = conn;
        pi->task_id = GETINT(id);
        pi->ticks_left = TICK_QUOTA;

        shutdown_tasks = CDR(shutdown_tasks);
    }

    init_prim();
    foreachglb(mark);

    sweep();
}

/*=======================================================================*/

int streq_oo(OBJECT a, OBJECT b) {
    if (GETHASH(a) == GETHASH(b))
        if (streq_ss(GETSTR(a), GETSTR(b)))
            return 1;
    return 0;
}

OBJECT shallow_copy(OBJECT list) {
    OBJECT org, curr;

    org = NULL;
    while (list != NULL) {
        if (org == NULL) org = curr = cons(CAR(list), NULL); else {
            SETCDR(curr,cons(CAR(list), NULL));
            curr = CDR(curr);
        }
        list = CDR(list);
    }
    return org;
}

/*------------------------------------------------------------------------*/

int get_type(OBJECT o) {
    word intobj;

    if (o == NULL) return T_NULL;

    intobj = ((word)o) & 3;
    switch (intobj) {
        case 0: return o->type;
        case 1: return T_INT;
        case 2: return o->type;
        case 3: return ((long) o & 4) ? T_PRIM : T_OBJECT;
    }

    return T_NULL;
}

PRIVATE OBJECT newstrsym(int type, char *val) {
    OBJECT o = newOBJECT();

    o->type = type;
    o->d.strsym.data = getmem(strlen(val) + 1);
    strcpy(o->d.strsym.data, val);
    o->d.strsym.hash = hash_str(val);

    return o;
}

OBJECT newstring(char *val) {
    return newstrsym(T_STRING, val);
}

OBJECT newsym(char *val) {
    return newstrsym(T_SYM, val);
}

OBJECT cons(OBJECT car, OBJECT cdr) {
    OBJECT o = newOBJECT();

    o->type = T_CONS;
    o->d.cons.car = car;
    o->d.cons.cdr = cdr;

    return o;
}

OBJECT newfunc(OBJECT env, char argc, word codelen, char *codebuf) {
    OBJECT o = newOBJECT();

    o->type = T_FUNC;
    o->d.function.env = env;
    o->d.function.body = getmem(codelen + 3);
    o->d.function.body[0] = argc;
    o->d.function.body[1] = (char) (codelen & 0xFF);
    o->d.function.body[2] = (char) (codelen >> 8);
    memcpy(o->d.function.body + 3, codebuf, codelen);

    return o;
}

OBJECT newvector(long length) {
    OBJECT o = newOBJECT();
    int i;

    o->type = T_VECTOR;
    o->d.vector.length = length;
    o->d.vector.vec = getmem((word) length * sizeof(OBJECT));
    for (i=0; i<length; i++) SETELEM(o,i,NULL);

    return o;
}

OBJECT newconn(int type, int info, void *data) {
    OBJECT o = newOBJECT();

    o->type = T_CONN;
    o->d.connection.type = type;
    o->d.connection.info = info;
    o->d.connection.d.f = data;
    connections[type].open(o);

    return o;
}

/*------------------------------------------------------------------------*/

#define OBJTBLSIZE 431  /* Prime number */
OBJINFO objtbl[OBJTBLSIZE];

PRIVATE void mark_objtbl(void) {
    OBJINFO curr;
    word i;

    for (i=0; i<OBJTBLSIZE; i++) {
        curr = objtbl[i];
        while (curr != NULL) {
            mark(curr->parent);
            mark(curr->children);
            mark(curr->slotnames);
            mark(curr->slotvals);
            mark(curr->methnames);
            mark(curr->methvals);
            mark(curr->name);
            mark(curr->owner);
            mark(curr->location);
            mark(curr->contents);
            curr = curr->next;
        }
    }
}

PRIVATE void dump_objtbl(FILE *f) {
    OBJINFO curr;
    word i;
    long n;

    fwrite(&nextnumber, sizeof(long), 1, f);

    for (i=0; i<OBJTBLSIZE; i++) {
        curr = objtbl[i];
        while (curr != NULL) {
            fwrite(&(curr->number), sizeof(long), 1, f);
            fwrite(&(curr->flags), sizeof(long), 1, f);
            dump_objnum(curr->parent, f);
            dump_objnum(curr->children, f);
            dump_objnum(curr->slotnames, f);
            dump_objnum(curr->slotvals, f);
            dump_objnum(curr->methnames, f);
            dump_objnum(curr->methvals, f);
            dump_objnum(curr->name, f);
            dump_objnum(curr->owner, f);
            dump_objnum(curr->location, f);
            dump_objnum(curr->contents, f);
            curr = curr->next;
        }
    }

    n = -1;
    fwrite(&n, sizeof(long), 1, f);
}

PRIVATE void load_oo_object(long num, FILE *f) {
    OBJINFO o = getmem(sizeof(_objinfo));
    word hash;

    fread(&(o->flags), sizeof(long), 1, f);
    o->parent = load_objnum(f);
    o->children = load_objnum(f);
    o->slotnames = load_objnum(f);
    o->slotvals = load_objnum(f);
    o->methnames = load_objnum(f);
    o->methvals = load_objnum(f);
    o->name = load_objnum(f);
    o->owner = load_objnum(f);
    o->location = load_objnum(f);
    o->contents = load_objnum(f);

    o->number = num;
    hash = (word) num % OBJTBLSIZE;
    o->next = objtbl[hash];
    objtbl[hash] = o;
}

PRIVATE void sweep_objtbl(void) {
    word i;
    OBJINFO next;

    for (i=0; i<OBJTBLSIZE; i++) {
        while (objtbl[i] != NULL) {
            next = objtbl[i]->next;
            freemem(objtbl[i]);
            objtbl[i] = next;
        }
    }
}

OBJECT maxobj(void) {
    return MKONUM(nextnumber - 1);
}

OBJECT values_copy(OBJECT list, OBJECT newown) {
    OBJECT org, curr;
    OBJECT info;

    org = NULL;
    curr = NULL;
    while (list != NULL) {
        info = CDR(CAR(list));
        if (newown != NULL && (GETINT(CDR(info)) & F_CHANGEOWN) != 0)
            info = cons(newown, CDR(info));

        if (org == NULL)
            org = curr = cons(cons(CAR(CAR(list)), info), NULL);
        else {
            SETCDR(curr, cons(cons(CAR(CAR(list)), info), NULL));
            curr = CDR(curr);
        }

        list = CDR(list);
    }
    return org;
}

OBJECT newobj(OBJECT parent, OBJECT player) {
    OBJINFO o = getmem(sizeof(_objinfo));
    OBJINFO p;
    word hash;

    if (parent == NULL) p = NULL; else
                        p = findobj(GETONUM(parent));

    if (p != NULL &&
        player != NULL &&
        player != p->owner &&
        ((findobj(GETONUM(player))->flags & FO_WIZARD) == 0) &&
        ((p->flags & FO_COPY) == 0))
            error("no-permission", newsym("clone"));

    do o->number = nextnumber++; while (validobjnum(o->number));

    o->flags = p ? p->flags : FO_COPY;

    o->parent = parent;
    o->children = NULL;
    if (p != NULL) p->children = cons(MKONUM(o->number), p->children);

    o->slotnames = p ? p->slotnames : NULL;
    o->slotvals = p ? values_copy(p->slotvals, player) : NULL;

    o->methnames = NULL;
    o->methvals = NULL;

    o->name = p ? p->name : NULL;
    o->owner = player;
    o->location = p ? p->location : NULL;
    o->contents = NULL;

    if (validobjnum(GETONUM(o->location))) {
        OBJINFO oi = findobj(GETONUM(o->location));
        oi->contents = cons(MKONUM(o->number), oi->contents);
    }

    hash = (word) (o->number % OBJTBLSIZE);
    o->next = objtbl[hash];
    objtbl[hash] = o;

    return MKONUM(o->number);
}

void freeobj(OBJECT o) {
    long onum = GETONUM(o);
    word hash = (word) (onum % OBJTBLSIZE);
    OBJINFO prev, curr;

    prev = NULL;
    curr = objtbl[hash];

    while (curr != NULL) {
        if (curr->number == onum) {
            if (prev == NULL) objtbl[hash] = curr->next; else
                              prev->next = curr->next;
            freemem(curr);
            return;
        }
        curr = curr->next;
    }
}

OBJINFO findobj(long onum) {
    OBJINFO curr;

    if (onum < 0) error("invalid-object", MKONUM(onum));

    curr = objtbl[(word) (onum % OBJTBLSIZE)];

    while (curr != NULL) {
        if (curr->number == onum)
            return curr;
        curr = curr->next;
    }

    error("invalid-object",MKONUM(onum));
    return NULL;
}

int validobjnum(long onum) {
    OBJINFO curr;

    if (onum < 0) return 0;

    curr = objtbl[(word) (onum % OBJTBLSIZE)];

    while (curr != NULL) {
        if (curr->number == onum) return 1;
        curr = curr->next;
    }

    return 0;
}

OBJECT findslot(OBJINFO o, char *name) {
    OBJECT currname, currval;

    currname = o->slotnames;
    currval = o->slotvals;
    while (currname != NULL) {
        if (streq_os(CAR(currname), name))
            return CAR(currval);
        currname = CDR(currname);
        currval = CDR(currval);
    }
    return NULL;
}

OBJECT findmeth(OBJINFO o, char *name, OBJECT *info, OBJECT *found_in) {
    OBJECT currname, currval;
    OBJECT currobj;
    OBJINFO oi;

    currobj = MKONUM(o->number);
    do {
        oi = findobj(GETONUM(currobj));
        currname = oi->methnames;
        currval = oi->methvals;
        while (currname != NULL) {
            if (streq_os(CAR(currname), name)) {
                *found_in = currobj;
                *info = CAR(currval);
                return CAR(CAR(currval));
            }
            currname = CDR(currname);
            currval = CDR(currval);
        }
        currobj = oi->parent;
    } while (currobj != NULL);
    return NULL;
}

PRIVATE void updatepointers(OBJINFO o, OBJECT old, OBJECT new, OBJECT val,
                            OBJECT info) {
    OBJECT cn,cv, pn,pv;
    OBJECT child;

    if ((GETINT(CDR(info)) & F_CHANGEOWN) != 0)
        info = cons(o->owner, CDR(info));

    pn = pv = NULL;
    cn = o->slotnames;
    cv = o->slotvals;

    while (1) {
        if (cn == new) error("updatepointers() is still screwed.", NULL);
        if (cn == old) {
            if (pn == NULL) {
                o->slotnames = new;
                o->slotvals = cons(cons(val, info), o->slotvals);
            } else {
                SETCDR(pn, new);
                SETCDR(pv, cons(cons(val, info), CDR(pv)));
            }
            break;
        }
        pn = cn; pv = cv;
        cn = CDR(cn);
        cv = CDR(cv);
    }

    child = o->children;
    while (child != NULL) {
        updatepointers(findobj(GETONUM(CAR(child))), old, new, val, info);
        child = CDR(child);
    }
}

void addslot(OBJINFO o, OBJECT name, OBJECT val, OBJECT owner, OBJECT flags) {
    OBJECT info = cons(owner, flags);
    OBJECT child = o->children;
    OBJECT oldnames = o->slotnames;

    o->slotnames = cons(name, o->slotnames);
    o->slotvals = cons(cons(val, info), o->slotvals);

    while (child != NULL) {
        updatepointers(findobj(GETONUM(CAR(child))), oldnames, o->slotnames,
            val, info);
        child = CDR(child);
    }
}

void addmeth(OBJINFO o, OBJECT name, OBJECT val, OBJECT owner, OBJECT flags) {
    o->methnames = cons(name, o->methnames);
    o->methvals = cons(cons(val, cons(owner, flags)), o->methvals);
}

void init_memory(void) {
    word i;

    freelist = NULL;
    freelist_size = 0;
    nextnumber = 0;
    objpages = NULL;

    for (i=0; i<OBJTBLSIZE; i++)
        objtbl[i] = NULL;
}

