#include "oom.h"

#include <stdlib.h>
#include <string.h>

PRIVATE CODE new_code(PARSER p) {
    CODE c = getmem(sizeof(_code));

    c->parser = p;
    c->buf = getmem(1024);
    c->pos = 0;
    c->buflen = 1024;
    c->scope = NULL;
    return c;
}

PRIVATE void gen(CODE c, char ch) {
    if (c->pos >= c->buflen) {
        c->buf = growmem(c->buf, c->buflen, 256);
        c->buflen += 256;
    }
    c->buf[c->pos++] = ch;
}

PRIVATE void gen_str(CODE c, char *s) {
    while (*s) {
        gen(c, *s);
        s++;
    }
    gen(c, 0);
}

PRIVATE void gen_word(CODE c, word w) {
    gen(c, (char) (w & 0xFF));
    gen(c, (char) (w >> 8));
}

PRIVATE void patch_word(CODE c, word pos, word w) {
    c->buf[pos] = (char) (w & 0xFF);
    c->buf[pos+1] = (char) (w >> 8);
}

PRIVATE void gen_long(CODE c, long l) {
    gen_word(c, (word) (l & 0xFFFF));
    gen_word(c, (word) (l >> 16));
}

void kill_code(CODE code) {
    if (!code) return;
    while (code->scope) code->scope = killscope(code->scope);
    freemem(code->buf);
    freemem(code);
}

/* ----------------------------------------------------------- PARSER PROPER */

#define CHECK(c)            check(code->parser, c)
#define CHECKDROP(c,e)      check_drop(code->parser, c, e)
#define DROP()              parse_drop(code->parser)
#define SCAN_S              (code->parser->scanner->d.s)
#define SCAN_L              (code->parser->scanner->d.l)

PRIVATE int expr_parse(CODE code); /* Forward reference */

PRIVATE int compile_varref(CODE code, char *id) {
    word snum, ofs, i;

    if (lookup(code->scope, id, &snum, &ofs)) {
        gen(code, OP_REF_LOCAL);
        gen_word(code, snum);
        gen_word(code, ofs);
        return 1;
    }

    i = 0;
    while (primtab[i].name != NULL) {
        if (streq_ss(id, primtab[i].name)) {
            gen(code, OP_REF_PRIM);
            gen_word(code, i);
            return 1;
        }
        i++;
    }

    error("Variable not bound", newsym(id));
    return 0;
}

#define NORMAL  0
#define SLOT    1
#define METHOD  2
#define PARENT  3

PRIVATE int id_parse(CODE code) {
    char currid[256];
    int have_id;
    int idtype = NORMAL;

    if (CHECK("(")) {
        DROP();
        if (!expr_parse(code)) return 0;
        if (!CHECKDROP(")","Unmatched close-paren")) return 0;
        currid[0] = '\0';
        have_id = 0;
    } else if (CHECK("identifier")) {
        DROP();
        strcpy(currid, SCAN_S);
        idtype = NORMAL;
        have_id = 1;
    } else if (CHECK("#")) {
        long neg = 1;

        DROP();

        if (CHECK("-")) {
            DROP();
            neg = -1;
        }

        if (!CHECKDROP("number","Number expected after # in object-reference"))
            return 0;
        gen(code, OP_CONST_OBJECT);
        gen_long(code, (SCAN_L) * neg);
        currid[0] = '\0';
        have_id = 0;
    } else if (CHECK("$")) {
        DROP();
        if (!CHECKDROP("identifier","ID expected in system-object slot"))
            return 0;
        gen(code, OP_CONST_OBJECT);
        gen_long(code, 0);
        strcpy(currid, SCAN_S);
        idtype = SLOT;
        have_id = 1;
    } else {
        error("Syntax error in id_parse",NULL);
        return 0;
    }

    while (1) {
        if (CHECK("(")) {
            char argc = 0;

            DROP();

            if (have_id) {
                switch (idtype) {
                    case PARENT:
                        error("Cannot call parent: it isn't a function",NULL);
                        return 0;
                    case METHOD:
                        gen(code, OP_PRE_METHCALL);
                        break;
                    case SLOT:
                        gen(code, OP_REF_SLOT);
                        gen_str(code, currid);
                        break;
                    case NORMAL:
                        if (!compile_varref(code, currid)) return 0;
                        break;
                }
            }

            if (CHECK(")")) DROP(); else
                while (1) {
                    if (!expr_parse(code)) return 0;
                    argc++;
                    if (CHECK(")")) {
                        DROP();
                        break;
                    }
                    if (!CHECKDROP(",","Comma expected separating "
                                       "args in funcall"))
                        return 0;
                }

            if (have_id && idtype == METHOD) {
                gen(code, OP_CALL_METHOD);
                gen(code, argc);
                gen_str(code, currid);
            } else {
                gen(code, OP_CALL);
                gen(code, argc);
            }

            have_id = 0;
            continue;
        }

        if (CHECK("=")) {
            DROP();

            if (!have_id) {
                error("LValue required in assignment", NULL);
                return 0;
            }
            switch (idtype) {
                case METHOD:
                    error("Cannot assign to method-slot", NULL);
                    return 0;
                case PARENT:
                    error("Cannot assign to .parent", NULL);
                    return 0;
            }

            if (!expr_parse(code)) return 0;

            switch (idtype) {
                case SLOT:
                    gen(code, OP_SET_SLOT);
                    gen_str(code, currid);
                    break;
                case NORMAL: {
                    word snum, ofs;

                    if (!lookup(code->scope, currid, &snum, &ofs)) {
                        error("Variable not settable (not found?)", NULL);
                        return 0;
                    }
                    gen(code, OP_SET_LOCAL);
                    gen_word(code, snum);
                    gen_word(code, ofs);
                    break;
                }
            }
            break;
        }

        if (have_id) {
            switch (idtype) {
                case PARENT:
                    gen(code, OP_REF_PARENT);
                    break;
                case METHOD:
                    gen(code, OP_REF_METHOD);
                    gen_str(code, currid);
                    break;
                case SLOT:
                    gen(code, OP_REF_SLOT);
                    gen_str(code, currid);
                    break;
                case NORMAL:
                    if (!compile_varref(code, currid)) return 0;
                    break;
            }
        }

        if (CHECK(":")) {
            DROP();

            if (have_id && idtype == METHOD) {
                error("Methods cannot be accessed as objects", NULL);
                return 0;
            }

            if (!CHECKDROP("identifier","Method-identifier expected "
                                        "after ':'"))
                return 0;

            strcpy(currid, SCAN_S);
            idtype = METHOD;
            have_id = 1;
            continue;
        }

        if (CHECK(".")) {
            DROP();

            if (have_id && idtype == METHOD) {
                error("Methods cannot be accessed as objects", NULL);
                return 0;
            }

            if (CHECK("parent")) {
                DROP();
                idtype = PARENT;
            } else {
                if (!CHECKDROP("identifier","Slot-ID expected after '.'"))
                    return 0;
                strcpy(currid, SCAN_S);
                idtype = SLOT;
            }

            have_id = 1;
            continue;
        }

        break;
    }
    return 1;
}

PRIVATE int constant_parse(CODE code) {
    if (CHECK("not")) {
        DROP();
        if (!constant_parse(code)) return 0;
        gen(code, OP_NOT);
        return 1;
    }

    if (CHECK("null")) {
        DROP();
        gen(code, OP_CONST_NULL);
        return 1;
    }

    if (CHECK("true")) {
        DROP();
        gen(code, OP_CONST_TRUE);
        return 1;
    }

    if (CHECK("false")) {
        DROP();
        gen(code, OP_CONST_FALSE);
        return 1;
    }

    if (CHECK("string")) {
        DROP();
        gen(code, OP_CONST_STR);
        gen_str(code, SCAN_S);
        return 1;
    }

    if (CHECK("'")) {
        DROP();

        if (!CHECKDROP("identifier","identifier must follow single-quote"))
            return 0;

        gen(code, OP_CONST_SYM);
        gen_str(code, SCAN_S);
        return 1;
    }

    if (CHECK("[")) {
        int listlen = 0;

        DROP();

        if (CHECK("]")) {
            DROP();
            gen(code, OP_CONST_NULL);
            return 1;
        }

        while (1) {
            if (!expr_parse(code)) return 0;
            listlen++;

            if (CHECK("]")) {
                DROP();
                break;
            }

            if (!CHECKDROP(",","Comma expected between expressions in a list"))
                return 0;
        }

        gen(code, OP_CONST_NULL);
        if (listlen == 1) gen(code, OP_CONS); else {
            gen(code, OP_NCONS);
            gen_word(code, listlen);
        }
        return 1;
    }

    if (CHECK("-")) {
        DROP();
        if (!CHECKDROP("number","Number must follow unary negation symbol"))
            return 0;
        gen(code, OP_CONST_NUM);
        gen_long(code, -SCAN_L);
        return 1;
    }

    if (CHECK("number")) {
        DROP();
        gen(code, OP_CONST_NUM);
        gen_long(code, SCAN_L);
        return 1;
    }

    return id_parse(code);
}

PRIVATE int mul_arith_parse(CODE code) {
    if (!constant_parse(code)) return 0;

    while (CHECK("*") || CHECK("/") || CHECK("%")) {
        int op = ( CHECK("*") ? OP_STAR :
                 ( CHECK("%") ? OP_PERCENT : OP_SLASH ) );

        DROP();
        if (!constant_parse(code)) return 0;
        gen(code, op);
    }
    return 1;
}

PRIVATE int add_arith_parse(CODE code) {
    if (!mul_arith_parse(code)) return 0;

    while (CHECK("+") || CHECK("-")) {
        int op = ( CHECK("+") ? OP_PLUS : OP_MINUS );

        DROP();
        if (!mul_arith_parse(code)) return 0;
        gen(code, op);
    }
    return 1;
}

PRIVATE int neq_op_parse(CODE code) {
    if (!add_arith_parse(code)) return 0;

    while (CHECK(">") || CHECK("<") || CHECK(">=") || CHECK("<=")) {
        int op;

        if (CHECK(">")) op = OP_GT;
        if (CHECK("<")) op = OP_LT;
        if (CHECK(">=")) op = OP_GE;
        if (CHECK("<=")) op = OP_LE;

        DROP();
        if (!add_arith_parse(code)) return 0;
        gen(code, op);
    }
    return 1;
}

PRIVATE int eq_op_parse(CODE code) {
    if (!neq_op_parse(code)) return 0;

    while (CHECK("==") || CHECK("!=")) {
        int op = ( CHECK("!=") ? OP_NE : OP_EQ );

        DROP();
        if (!neq_op_parse(code)) return 0;
        gen(code, op);
    }
    return 1;
}

PRIVATE int and_op_parse(CODE code) {
    if (!eq_op_parse(code)) return 0;

    while (CHECK("and")) {
        DROP();
        if (!eq_op_parse(code)) return 0;
        gen(code, OP_AND);
    }
    return 1;
}

PRIVATE int or_op_parse(CODE code) {
    if (!and_op_parse(code)) return 0;

    while (CHECK("or")) {
        DROP();
        if (!and_op_parse(code)) return 0;
        gen(code, OP_OR);
    }
    return 1;
}

char def_arglist_parse(CODE code) {
    char argc = 0;

    if (CHECK(")")) DROP(); else
        while (1) {
            if (CHECK("identifier")) {
                argc++;
                addbinding(code->scope, SCAN_S);
                DROP();
            }

            if (CHECK(")")) {
                DROP();
                break;
            }

            if (!CHECKDROP(",","Comma expected to separate arglist-names"))
                return -1;
        }
    return argc;
}

int expr_parse(CODE code) {
    if (CHECK("define")) {
        char s[256];
        word snum, ofs;

        DROP();

        if (CHECK("slot")) {
            DROP();

            if (!expr_parse(code)) return 0;

            if (!CHECKDROP("identifier","slot-identifier expected"))
                return 0;
            strcpy(s,SCAN_S);

            if (CHECK("as")) {
                DROP();
                if (!expr_parse(code)) return 0;
            } else gen(code, OP_CONST_NULL);

            gen(code, OP_DEF_SLOT);
            gen_str(code, s);
            return 1;
        }

        if (CHECK("method")) {
            DROP();

            if (!expr_parse(code)) return 0;

            if (!CHECKDROP("identifier","Self-ID reqd for self-clause"))
                return 0;

            code->scope = newscope(code->scope);
            addbinding(code->scope, SCAN_S);

            if (!CHECKDROP(":","Colon punctuation required between "
                               "self-clause and method-name"))
                return 0;

            if (!CHECKDROP("identifier","Method-name required"))
                return 0;
            strcpy(s,SCAN_S);

            if (CHECK("as")) {
                DROP();
                code->scope = killscope(code->scope);
                if (!expr_parse(code)) return 0;
            } else if (CHECK("(")) {
                char argc;
                word patchaddr, pos;

                DROP();
                argc = def_arglist_parse(code); /* Checks for close-paren */
                if (argc == -1) return 0;
                argc++; /* For "self" param */

                if (!CHECKDROP("as","Keyword 'as' expected after "
                                    "method-header"))
                    return 0;

                gen(code, OP_CONST_FUNCTION);
                gen(code, argc);
                patchaddr = code->pos;
                gen_word(code, 0);
                pos = code->pos;
                if (!expr_parse(code)) return 0;
                gen(code, OP_RET);
                patch_word(code, patchaddr, code->pos - pos);
                code->scope = killscope(code->scope);
            } else {
                error("Parameter-list or keyword 'as' expected "
                      "after method name", NULL);
                code->scope = killscope(code->scope);
                return 0;
            }

            gen(code, OP_DEF_METHOD);
            gen_str(code, s);

            return 1;
        }

        if (!CHECKDROP("identifier","identifier, 'method', or 'slot' "
                                    "expected after define"))
            return 0;

        strcpy(s, SCAN_S);

        if (!lookup(code->scope, s, &snum, &ofs))
            addbinding(code->scope, s);
        else
            if (snum > 0) addbinding(code->scope, s);

        if (CHECK("as")) {
            DROP();
            if (!expr_parse(code)) return 0;
        } else gen(code, OP_CONST_NULL);

        if (code->scope == NULL)
            error("Cannot define variables at toplevel",NULL);

        lookup(code->scope, s, &snum, &ofs);

        gen(code, OP_SET_LOCAL);
        gen_word(code, snum);
        gen_word(code, ofs);
        return 1;
    }

    if (CHECK("function")) {
        char argc = 0;
        word patchaddr, pos;

        DROP();

        if (!CHECKDROP("(","Parameter-list expected after function name"))
            return 0;

        code->scope = newscope(code->scope);
        argc = def_arglist_parse(code);
        if (argc == -1) return 0;

        gen(code, OP_CONST_FUNCTION);
        gen(code, argc);
        patchaddr = code->pos;
        gen_word(code, 0);
        pos = code->pos;
        if (!expr_parse(code)) return 0;
        gen(code, OP_RET);
        patch_word(code, patchaddr, code->pos - pos);

        code->scope = killscope(code->scope);
        return 1;
    }

    if (CHECK("if")) {
        word fjmp, endjmp;

        DROP();

        if (!expr_parse(code)) return 0;
        gen(code, OP_JUMP_FALSE);
        fjmp = code->pos;
        gen_word(code, 0);

        if (!CHECKDROP("then","Keyword 'then' expected after test in if"))
            return 0;

        if (!expr_parse(code)) return 0;

        gen(code, OP_JUMP);
        endjmp = code->pos;
        gen_word(code, 0);
        patch_word(code, fjmp, code->pos - fjmp);

        if (CHECK("else")) {
            DROP();
            if (!expr_parse(code)) return 0;
        } else gen(code, OP_CONST_NULL);

        patch_word(code, endjmp, code->pos - endjmp);
        return 1;
    }

    if (CHECK("while") || CHECK("until")) {
        int isuntil = CHECK("until");
        word wtest, wend;

        DROP();

        wtest = code->pos;
        if (!expr_parse(code)) return 0;
        if (isuntil)
            gen(code, OP_JUMP_TRUE); else
            gen(code, OP_JUMP_FALSE);
        wend = code->pos;
        gen_word(code, 0);

        if (!CHECKDROP("do","do required after while- or until-test"))
            return 0;

        if (!expr_parse(code)) return 0;

        gen(code, OP_POP);
        gen(code, OP_JUMP);
        gen_word(code, wtest - code->pos);
        patch_word(code, wend, code->pos - wend);
        gen(code, OP_CONST_NULL);
        return 1;
    }

    if (CHECK("after")) {
        long num_seconds;
        word patchaddr, pos;

        DROP();

        if (!CHECKDROP("number", "Positive number expected "
                                 "after 'after' keyword"))
            return 0;
        num_seconds = SCAN_L;

        gen(code, OP_CONST_FUNCTION);
        gen(code, 0);
        patchaddr = code->pos;
        gen_word(code, 0);
        pos = code->pos;
        if (!expr_parse(code)) return 0;
        gen(code, OP_RET);
        patch_word(code, patchaddr, code->pos - pos);

        gen(code, OP_FORK_AFTER);
        gen_long(code, num_seconds);

        return 1;
    }

    if (CHECK("begin")) {
        word pos;

        DROP();

        if (CHECK("end")) {
            DROP();
            return 1;
        }

        code->scope = newscope(code->scope);

        gen(code, OP_ENTER_SCOPE);
        pos = code->pos;
        gen_word(code, 0);

        while (1) {
            if (!expr_parse(code)) return 0;
            if (!CHECKDROP(";","Semicolon expected after expression in block"))
                return 0;
            if (CHECK("end")) {
                DROP();
                break;
            }
            gen(code, OP_POP);
        }

        patch_word(code, pos, code->scope->size);

        gen(code, OP_LEAVE_SCOPE);
        code->scope = killscope(code->scope);

        return 1;
    }

    if (CHECK("return")) {
        DROP();

        if (CHECK(";")) gen(code, OP_CONST_NULL); else
            if (!expr_parse(code)) return 0;
        gen(code, OP_RET);

        return 1;
    }

    return or_op_parse(code);
}

CODE parser(PARSER p) {
    CODE code = new_code(p);

    if (!CHECK("end-of-file") && expr_parse(code)) {
        CHECKDROP(";","Semi-colon expected to terminate toplevel expression");
        gen(code, OP_RET);
        return code;
    }

    kill_code(code);
    return NULL;
}

