#include "oom.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <time.h>
#include <ctype.h>

#define LISTP(x)    (get_type(x) == T_CONS || (x) == NULL)

OBJECT printon(OBJECT conn, OBJECT o) {
    if (get_type(conn) != T_CONN) error("type-error", newsym("print-on"));
    switch (get_type(o)) {
        case T_NULL: c_printf(conn,"[]"); break;
        case T_INT: c_printf(conn,"%ld",GETINT(o)); break;
        case T_STRING:
        case T_SYM: c_puts(GETSTR(o), conn); break;
        case T_CONS:
            c_printf(conn,"[");
            while (o != NULL && get_type(o) == T_CONS) {
                printon(conn, CAR(o));
                o = CDR(o);
                if (o != NULL) {
                    c_printf(conn,", ");
                    if (get_type(o) != T_CONS) {
                        c_printf(conn,"| ");
                        printon(conn, o);
                    }
                }
            }
            c_printf(conn,"]");
            break;
        case T_FUNC: c_printf(conn,"!!function(%d)",GETBODY(o)[0]); break;
        case T_PRIM:
            c_printf(conn,"!!prim(%s)",primtab[GETPNUM(o)].name);
            break;
        case T_VECTOR:
            c_printf(conn,"{");
            if (GETLEN(o) > 0) {
                word i;
                for (i=0; i<GETLEN(o)-1; i++) {
                    printon(conn, GETELEM(o,i));
                    c_printf(conn," ");
                }
                printon(conn, GETELEM(o,GETLEN(o)-1));
            }
            c_printf(conn,"}");
            break;
        case T_CONN:
            c_printf(conn, "!!connection(%d,%d)", GETCTYPE(o), GETCINFO(o));
            break;
        case T_OBJECT: c_printf(conn,"#%ld",GETONUM(o)); break;
        default:
            fatal("Unknown object-type in print: %d (%p)",get_type(o),o);
    }

    return NULL;
}

OBJECT print(OBJECT o) {
    return printon(curinfo->conn, o);
}

PRIVATE void strapp(char **s, int *len, int *buflen, char what) {
    if (*len >= *buflen) {
        *s = growmem(*s, *buflen, 256);
        *buflen += 256;
    }
    (*s)[(*len)++] = what;
}

OBJECT readfromfun(OBJECT conn) {
    char *str = NULL;
    int len = 0;
    int buflen = 0;
    char ch;
    OBJECT result;

    if (get_type(conn) != T_CONN) error("type-error", newsym("read-from"));

    while ((ch = c_getc(conn)) != '\n') strapp(&str, &len, &buflen, ch);
    strapp(&str, &len, &buflen, '\0');
    result = newstring(str);
    freemem(str);
    return result;
}

OBJECT readfun(void) {
    return readfromfun(curinfo->conn);
}

OBJECT disconfun(OBJECT conn) {
    if (get_type(conn) != T_CONN) error("type-error", newsym("disconnect"));
    if ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
        error("no-permission", newsym("disconnect"));
    c_disconnect(conn);
    return NULL;
}

PRIVATE OBJECT clone(OBJECT parent) {
    OBJECT o;

    if (get_type(parent) != T_OBJECT) error("type-error", newsym("clone"));

    o = newobj(parent, curinfo->player);
    callmethod(o, "initialize", 0);

    return o;
}

PRIVATE OBJECT deletefun(OBJECT l, OBJECT o) {
    OBJECT org, prev;

    if (!LISTP(l)) error("type-error", newsym("delete!"));

    org = l;
    prev = NULL;
    while (l != NULL) {
        if (equalfun(CAR(l), o) != false) {
            if (prev == NULL) return CDR(org);
            SETCDR(prev, CDR(l));
            return org;
        }
        prev = l;
        l = CDR(l);
    }

    return org;
}

PRIVATE OBJECT recycle(OBJECT what) {
    OBJECT parent, curr;
    OBJINFO wi;

    if (get_type(what) != T_OBJECT) error("type-error", newsym("recycle"));

    wi = findobj(GETONUM(what));

    if (curinfo->code_owner != wi->owner &&
        ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0))
            error("no-permission", newsym("recycle"));

    callmethod(what, "recycle", 0);

    parent = wi->parent;

    if (get_type(parent) == T_OBJECT) {
        OBJECT child = wi->children;
        OBJECT pc = findobj(GETONUM(parent))->children;
        OBJECT prev;

        while (child != NULL) {
            findobj(GETONUM(CAR(child)))->parent = parent;
            pc = cons(CAR(child), pc);
            child = CDR(child);
        }
        findobj(GETONUM(parent))->children = pc;

        prev = NULL;
        child = pc;
        while (child != NULL) {
            if (CAR(child) == what) {
                if (prev == NULL)
                    findobj(GETONUM(parent))->children = CDR(child); else
                    SETCDR(prev, CDR(child));
                break;
            }
            prev = child;
            child = CDR(child);
        }
    }

    if (validobjnum(GETONUM(wi->location))) {
        OBJINFO ii = findobj(GETONUM(wi->location));
        ii->contents = deletefun(ii->contents, what);
    }

    curr = wi->contents;
    while (curr != NULL) {
        findobj(GETONUM(CAR(curr)))->location = MKONUM(-1);
        curr = CDR(curr);
    }

    freeobj(what);
    return NULL;
}

PRIVATE OBJECT gettaskidfun(void) {
    return newint(curinfo->task_id);
}

PRIVATE OBJECT ticksleft(void) {
    return newint(curinfo->ticks_left);
}

PRIVATE PROCINFO findtask(long id) {
    PROCESS p = getproclist();
    while (p != NULL) {
        if (p->info != NULL)
            if (p->info->task_id == id)
                return p->info;
        p = p->next;
    }
    return NULL;
}

PRIVATE OBJECT validtaskfun(OBJECT n) {
    long taskid = GETINT(n);

    if (get_type(n) != T_INT) error("type-error", newsym("valid-task"));
    if (findtask(taskid) != NULL) return true;
    return false;
}

PRIVATE OBJECT killtask(OBJECT num) {
    PROCINFO pi;

    if (get_type(num) != T_INT) error("type-error", newsym("kill-task"));

    pi = findtask(GETINT(num));
    if (pi == NULL) error("invalid-argument",num);

    if (((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0) &&
        curinfo->player != pi->player)
            error("no-permission",newsym("kill-task"));

    pi->ticks_left = 0;
    return NULL;
}

PRIVATE OBJECT timefun(void) {
    time_t timer;

    time(&timer);
    return newint(timer);
}

PRIVATE OBJECT timestrfun(void) {
    time_t timer;
    time(&timer);
    return newstring(ctime(&timer));
}

PRIVATE OBJECT shutdown(void) {
    if ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
        error("no-permission",newsym("shutdown"));

    log("server shut down by ~O (as ~O)",
        curinfo->player, curinfo->code_owner);

    shutting_down = 1;
    return NULL;
}

PRIVATE OBJECT killserver(void) {
    if ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
        error("no-permission",newsym("kill-server!!"));

    log("server killed by ~O (as ~O)", curinfo->player, curinfo->code_owner);

    exit(1);
    return NULL;    /* Keeps the compiler happy... */
}

PRIVATE OBJECT syslog(OBJECT str) {
    if ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
        error("no-permission",newsym("log"));

    if (get_type(str) != T_STRING) error("type-error", newsym("log"));
    log("> ~S", GETSTR(str));

    return true;
}

PRIVATE long listlen(OBJECT list) {
    long count = 0;

    while (list != NULL) {
        count++;
        list = CDR(list);
    }

    return count;
}

PRIVATE OBJECT listtovect(OBJECT list) {
    OBJECT vec;
    int i;

    if (!LISTP(list)) error("type-error", newsym("vector"));

    vec = newvector(listlen(list));

    for (i=0; i<GETLEN(vec); i++) {
        SETELEM(vec,i,CAR(list));
        list = CDR(list);
    }

    return vec;
}

PRIVATE OBJECT mkvectfun(OBJECT len, OBJECT val) {
    OBJECT vec;
    int i;

    if (get_type(len) != T_INT) error("type-error", newsym("make-vector"));

    vec = newvector(GETINT(len));
    for (i=0; i<GETLEN(vec); i++) SETELEM(vec, i, val);

    return vec;
}

PRIVATE OBJECT mkstrfun(OBJECT len, OBJECT val) {
    OBJECT str;
    char *s;
    int i;

    if (get_type(len) != T_INT) error("type-error", newsym("make-string"));
    if (get_type(val) != T_STRING &&
        val != NULL)
            error("type-error", newsym("make-string"));

    s = getmem((int) GETINT(len) + 1);

    for (i=0; i<GETINT(len); i++) s[i] = (val == NULL ? ' ' : GETSTR(val)[0]);
    s[(int) GETINT(len)] = '\0';

    str = newstring(s);

    freemem(s);
    return str;
}

PRIVATE OBJECT head(OBJECT p) {
    if (!LISTP(p)) error("type-error", newsym("head"));

    return CAR(p);
}

PRIVATE OBJECT sethead(OBJECT p, OBJECT v) {
    if (!LISTP(p)) error("type-error", newsym("set-head"));

    SETCAR(p, v);
    return v;
}

PRIVATE OBJECT tail(OBJECT p) {
    if (!LISTP(p)) error("type-error", newsym("tail"));

    return CDR(p);
}

PRIVATE OBJECT settail(OBJECT p, OBJECT v) {
    if (!LISTP(p)) error("type-error", newsym("set-tail"));

    SETCDR(p, v);
    return v;
}

PRIVATE OBJECT lenfun(OBJECT l) {
    switch (get_type(l)) {
        case T_NULL: return newint(0);
        case T_STRING: return newint(strlen(GETSTR(l)));
        case T_CONS: return newint(listlen(l));
        case T_VECTOR: return newint(GETLEN(l));
        default:
            error("type-error", newsym("length"));
    }
    return NULL;
}

PRIVATE OBJECT appendfun(OBJECT l, OBJECT o) {
    OBJECT org = l;

    if (get_type(l) != T_CONS)
        if (l == NULL) return cons(o, NULL); else
            error("type-error", newsym("append!"));

    while (CDR(l) != NULL) l = CDR(l);
    SETCDR(l, cons(o, NULL));

    return org;
}

PRIVATE OBJECT getelemfun(OBJECT l, OBJECT i) {
    long idx;

    if (get_type(i) != T_INT) error("type-error", newsym("element"));
    idx = GETINT(i);

    switch (get_type(l)) {
        case T_NULL:
            error("out-of-range", newsym("element"));

        case T_STRING: {
            char ch[2];
            ch[0] = GETSTR(l)[(word) idx];
            ch[1] = '\0';
            return newstring(ch);
        }

        case T_CONS: {
            while (l != NULL) {
                if (idx == 0) return CAR(l);
                idx--;
                l = CDR(l);
            }
            error("out-of-range", newsym("element"));
        }

        case T_VECTOR:
            if (idx >= GETLEN(l) || idx < 0)
                error("out-of-range", newsym("element"));
            return GETELEM(l, (word) idx);

        default:
            error("type-error", newsym("element"));
    }
    return NULL;
}

PRIVATE OBJECT setelemfun(OBJECT l, OBJECT i, OBJECT o) {
    long idx;

    if (get_type(i) != T_INT) error("type-error", newsym("set-elem!"));

    idx = GETINT(i);

    switch (get_type(l)) {
        case T_NULL:
            error("out-of-range", newsym("element"));

        case T_STRING:
            if (idx >= strlen(GETSTR(l)))
                error("out-of-range", newsym("set-elem!"));
            if (get_type(o) != T_STRING)
                error("type-error", newsym("set-elem!"));
            GETSTR(l)[(word) idx] = GETSTR(o)[0];
            break;

        case T_CONS:
            while (l != NULL) {
                if (idx == 0) {
                    SETCAR(l, o);
                    break;
                }
                idx--;
                l = CDR(l);
            }
            if (l == NULL) error("out-of-range", newsym("set-elem!"));
            break;

        case T_VECTOR:
            if (idx >= GETLEN(l) || idx < 0)
                error("out-of-range", newsym("set-elem!"));
            SETELEM(l, (word) idx, o);
            break;

        default:
            error("type-error", newsym("set-elem!"));
    }
    return o;
}

PRIVATE OBJECT listidxfun(OBJECT l, OBJECT o) {
    long idx;

    if (!LISTP(l)) error("type-error", newsym("listidx"));

    idx = 0;

    while (l != NULL) {
        if (equalfun(CAR(l), o) != false) return newint(idx);
        idx++;
        l = CDR(l);
    }

    return false;
}

PRIVATE OBJECT sectionfun(OBJECT what, OBJECT idx, OBJECT len) {
    if (get_type(idx) != T_INT) error("type-error", newsym("section"));
    if (get_type(len) != T_INT) error("type-error", newsym("section"));

    switch (get_type(what)) {
        case T_NULL:
            error("out-of-range", newsym("section"));

        case T_STRING: {
            char *s;
            long length;

            if (GETINT(idx) > strlen(GETSTR(what)) ||
                GETINT(idx) < 0 ||
                GETINT(len) < 0)
                    error("out-of-range", newsym("section"));

            length = GETINT(len);
            if (length + GETINT(idx) > strlen(GETSTR(what)))
                length = strlen(GETSTR(what)) - GETINT(idx);
            s = getmem((word) length+1);
            memcpy(s, GETSTR(what) + (word) GETINT(idx), (word) length);
            s[(word) length] = '\0';
            what = newstring(s);
            freemem(s);

            return what;
        }

        case T_VECTOR: {
            OBJECT vec; 
            word i,f;

            if (GETINT(idx) > GETLEN(what) ||
                GETINT(idx) < 0 ||
                GETINT(idx) + GETINT(len) > GETLEN(what) ||
                GETINT(len) < 0)
                    error("out-of-range", newsym("section"));

            vec = newvector(GETINT(len));
            f = (word) GETINT(idx) + (word) GETINT(len);
            for (i = (word) GETINT(idx); i<f; i++)
                SETELEM(vec, i - (word) GETINT(idx), GETELEM(what, i));

            return vec;
        }

        case T_CONS: {
            OBJECT curr = what, org, prev;
            long count;

            count = GETINT(idx);
            while (curr != NULL && count > 0) {
                count--;
                curr = CDR(curr);
            }

            if (count > 0) error("out-of-range", newsym("section"));

            count = GETINT(len);
            org = NULL;
            prev = NULL;

            while (curr != NULL && count > 0) {
                if (prev == NULL) org = prev = cons(CAR(curr), NULL); else {
                    SETCDR(prev, cons(CAR(curr), NULL));
                    prev = CDR(prev);
                }
                curr = CDR(curr);
                count--;
            }

            if (count > 0) error("out-of-range", newsym("section"));

            return org;
        }

        default:
            error("type-error", newsym("section"));
    }
    return NULL;
}

PRIVATE OBJECT strcmpfun(OBJECT a, OBJECT b) {
    if (get_type(a) != T_STRING ||
        get_type(b) != T_STRING)
            error("type-error", newsym("strcmp"));

    return newint(strcmp(GETSTR(a), GETSTR(b)));
}

PRIVATE OBJECT stridxfun(OBJECT s, OBJECT sub) {
    char *idx;

    if (get_type(s) != T_STRING ||
        get_type(sub) != T_STRING)
            error("type-error", newsym("stridx"));

    idx = strstr(GETSTR(s), GETSTR(sub));

    if (idx == NULL) return false;
    return newint(idx-GETSTR(s));
}

PRIVATE OBJECT strsubfun(OBJECT s, OBJECT what, OBJECT with) {
    char *result = NULL;
    int len = 0;
    int buflen = 0;
    char *src, *rep, *txt;
    int replen;

    if (get_type(s) != T_STRING) error("type-error", newsym("strsub"));
    if (get_type(what) != T_STRING) error("type-error", newsym("strsub"));
    if (get_type(with) != T_STRING) error("type-error", newsym("strsub"));

    src = GETSTR(s);
    rep = GETSTR(what);
    replen = strlen(rep);

    while (*src != '\0') {
        if (!strncmp(src, rep, replen)) {
            src += replen;
            txt = GETSTR(with);
            while (*txt != '\0') strapp(&result, &len, &buflen, *txt++);
        } else strapp(&result, &len, &buflen, *src++);
    }
    strapp(&result, &len, &buflen, '\0');

    s = newstring(result);
    freemem(result);

    return s;
}

PRIVATE OBJECT toupperfun(OBJECT s) {
    char *res;
    if (get_type(s) != T_STRING) error("type-error", newsym("toupper"));
    res = getmem(strlen(GETSTR(s)) + 1);
    strcpy(res, GETSTR(s));
    strupr(res);
    s = newstring(res);
    freemem(res);
    return s;
}

PRIVATE OBJECT tolowerfun(OBJECT s) {
    char *res;
    if (get_type(s) != T_STRING) error("type-error", newsym("tolower"));
    res = getmem(strlen(GETSTR(s)) + 1);
    strcpy(res, GETSTR(s));
    strlwr(res);
    s = newstring(res);
    freemem(res);
    return s;
}

#define BOOL(x)     ((x)?true:false)

OBJECT equalfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);
    char btype = get_type(b);

    if (a == b) return true;

    if (atype != btype) return false;
    switch (atype) {
        case T_STRING:
        case T_SYM:
            return BOOL(streq_oo(a,b));

        case T_CONS:
            while (a != NULL && b != NULL) {
                if (equalfun(CAR(a),CAR(b)) == false) return false;
                a = CDR(a);
                b = CDR(b);
                if (get_type(a) != T_CONS) {
                    if (get_type(b) != T_CONS)
                        return equalfun(a,b);
                    return false;
                }
                if (get_type(b) != T_CONS) return false;
            }
            if (a != NULL || b != NULL) return false;
            return true;

        case T_VECTOR: {
            word i;

            if (GETLEN(a) != GETLEN(b)) return false;

            for (i=0; i<GETLEN(a); i++)
                if (equalfun(GETELEM(a,i), GETELEM(b,i)) == false)
                    return false;
            return true;
        }
    }
    return false;
}

OBJECT gtfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);
    char btype = get_type(b);

    if (atype != btype) return false;
        /* error("type-error", newsym("greater-than")); */

    if (a == b && atype != T_OBJECT) return false;

    switch (atype) {
        case T_INT: return BOOL( GETINT(a) > GETINT(b) );
        case T_STRING:
        case T_SYM: return BOOL( stricmp(GETSTR(a), GETSTR(b)) > 0 );
        case T_OBJECT: return ltfun(b, a);
    }

    error("type-error", newsym("greater-than"));
    return false;
}

OBJECT ltfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);
    char btype = get_type(b);

    if (atype != btype) return false;
        /* error("type-error", newsym("less-than")); */

    if (a == b && atype != T_OBJECT) return false;

    switch (atype) {
        case T_INT: return BOOL( GETINT(a) < GETINT(b) );
        case T_STRING:
        case T_SYM: return BOOL( stricmp(GETSTR(a), GETSTR(b)) < 0 );
        case T_OBJECT:
            while (a != NULL)
                if (a == b) return true; else a = findobj(GETONUM(a))->parent;
            return false;
    }

    error("type-error", newsym("less-than"));
    return false;
}

OBJECT addfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);

    if (atype != get_type(b) &&
        !(LISTP(a) && LISTP(b))) error("type-error", newsym("add"));

    switch (atype) {
        case T_NULL:
            if (LISTP(b)) return b;
            break;

        case T_INT:
            return newint(GETINT(a) + GETINT(b));

        case T_STRING: {
            char *s;
            OBJECT tmp;
            s = getmem(strlen(GETSTR(a))+strlen(GETSTR(b))+1);
            strcpy(s, GETSTR(a));
            strcat(s, GETSTR(b));
            tmp = newstring(s);
            freemem(s);
            return tmp;
        }

        case T_CONS: {
            OBJECT org, prev;
            org = prev = NULL;

            while (get_type(a) == T_CONS) {
                if (prev == NULL) org = prev = cons(CAR(a), NULL); else {
                    SETCDR(prev, cons(CAR(a), NULL));
                    prev = CDR(prev);
                }
                a = CDR(a);
            }
            if (a != NULL) error("type-error", newsym("add"));
            while (get_type(b) == T_CONS) {
                if (prev == NULL) org = prev = cons(CAR(b), NULL); else {
                    SETCDR(prev, cons(CAR(b), NULL));
                    prev = CDR(prev);
                }
                b = CDR(b);
            }
            if (b != NULL) error("type-error", newsym("add"));
            return org;
        }
    }
    error("type-error", newsym("add"));
    return newint(0);
}

OBJECT subfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);

    if (atype != get_type(b) || atype != T_INT)
        error("type-error", newsym("subtract"));

    return newint(GETINT(a) - GETINT(b));
}

OBJECT mulfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);

    if (atype != get_type(b) || atype != T_INT)
        error("type-error", newsym("multiply"));

    return newint(GETINT(a) * GETINT(b));
}

OBJECT divfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);

    if (atype != get_type(b) || atype != T_INT)
        error("type-error", newsym("divide"));

    if (GETINT(b) == 0) error("division-by-zero", newsym("divide"));

    return newint(GETINT(a) / GETINT(b));
}

OBJECT modfun(OBJECT a, OBJECT b) {
    char atype = get_type(a);

    if (atype != get_type(b) || atype != T_INT)
        error("type-error", newsym("modulo"));

    if (GETINT(b) == 0) error("division-by-zero", newsym("modulo"));

    return newint(GETINT(a) % GETINT(b));
}

PRIVATE OBJECT absfun(OBJECT i) {
    long num;
    if (get_type(i) != T_INT) error("type-error", newsym("abs"));
    num = GETINT(i);
    if (num < 0) num = -num;
    return newint(num);
}

PRIVATE OBJECT randomfun(OBJECT scale) {
    if (get_type(scale) != T_INT) error("type-error", newsym("random"));
    return newint(random(GETINT(scale)));
}

PRIVATE OBJECT connectionfun(void) {
    return curinfo->conn;
}

PRIVATE OBJECT playerfun(void) {
    return curinfo->player;
}

PRIVATE OBJECT pushhandlerfun(OBJECT func) {
    if (get_type(func) != T_FUNC) error("type-error", newsym("push-handler"));
    curinfo->errstk = cons(func, curinfo->errstk);
    return func;
}

PRIVATE OBJECT pophandlerfun(void) {
    OBJECT func;
    if (curinfo->errstk == NULL) return NULL;
    func = CAR(curinfo->errstk);
    curinfo->errstk = CDR(curinfo->errstk);
    return func;
}

extern void code_scanner(SCANSTATE s);
extern CODE parser(PARSER p);

PRIVATE OBJECT evalfun(OBJECT str) {
    SCANSTATE s;
    PARSER p;
    CODE code;
    OBJECT stk, res;

    if (get_type(str) != T_STRING) error("type-error", newsym("evaluate"));

    stk = curinfo->stack;
    curinfo->stack = newstack(INTERP_OBJSTACKSIZE);

    s = newscanner("EVAL-STR",newconn(CONN_STRING, 0, GETSTR(str)));
    p = newparser(code_scanner, s);

    c_reset(s->conn);

    code = parser(p);

    if (code != NULL) {
        interp(NULL, code->buf, NULL, NULL, NULL);
        kill_code(code);
    }

    res = pop(curinfo->stack);

    killparser(p);
    killscanner(s);

    curinfo->stack = stk;

    return res;
}

PRIVATE OBJECT callerfun(void) {
    return curinfo->caller;
}

#include "version.h"
PRIVATE OBJECT getverfun(void) {
    return newstring(__VERSION__);
}

PRIVATE OBJECT errfun(OBJECT errmsg, OBJECT errobj) {
    if (get_type(errmsg) != T_SYM) error("type-error", newsym("error"));

    error(GETSTR(errmsg), errobj);
    return NULL;
}

PRIVATE OBJECT abortfun(void) {
    resetjmp();
    return NULL;
}

PRIVATE OBJECT setcodeownerfun(OBJECT owner) {
    if (get_type(owner) != T_OBJECT)
        error("type-error", newsym("set-code-owner"));

    if (curinfo->code_owner != owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
            error("no-permission", newsym("set-code-owner"));

    curinfo->code_owner = owner;
    return NULL;
}

extern OBJECT enter_scope(OBJECT prev, word size);  /* In INTERP.C */
extern void s_setprev(OBJECT scope, OBJECT prev);   /* In INTERP.C */

PRIVATE OBJECT simmethcallfun(OBJECT self, OBJECT class, OBJECT owner,
                              OBJECT meth, OBJECT args) {
    OBJECT scope;
    OBJECT oldcaller, oldowner;
    int i;
    int argc;

    if ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
        error("no-permission", newsym("simulate-method-call"));

    if (get_type(self) != T_OBJECT)
        error("type-error", newsym("simulate-method-call"));
    if (get_type(class) != T_OBJECT)
        error("type-error", newsym("simulate-method-call"));
    if (get_type(meth) != T_FUNC)
        error("type-error", newsym("simulate-method-call"));
    if (!LISTP(args)) error("type-error", newsym("simulate-method-call"));

    argc = (int) listlen(args);
    scope = enter_scope(NULL, argc + 1);

    i = 1;
    while (args != NULL) {
        SETELEM(scope, i, CAR(args));
        args = CDR(args);
        i++;
    }
    SETELEM(scope, 0, self);

    s_setprev(scope, GETENV(meth));

    if (argc + 1 != GETBODY(meth)[0]) error("wrong-args", meth);

    oldowner = curinfo->code_owner;
    oldcaller = curinfo->caller;
    curinfo->code_owner = owner;
    curinfo->caller = oldowner;

    interp(meth, GETBODY(meth) + 3, self, class, scope);

    curinfo->caller = oldcaller;
    curinfo->code_owner = oldowner;

    return pop(curinfo->stack);
}

PRIVATE OBJECT playerpfun(OBJECT p) {
    if (get_type(p) != T_OBJECT) return false;
    if (findobj(GETONUM(p))->flags & FO_PLAYER) return true;
    return false;
}

PRIVATE OBJECT mkplayerfun(OBJECT p, OBJECT setting) {
    long f;

    if (get_type(p) != T_OBJECT) error("type-error", newsym("make-player"));

    f = findobj(GETONUM(p))->flags;
    if (setting == false) f &= ~FO_PLAYER; else f |= FO_PLAYER;
    findobj(GETONUM(p))->flags = f;

    return NULL;
}

PRIVATE OBJECT wizardpfun(OBJECT p) {
    if (get_type(p) != T_OBJECT) return false;
    if (findobj(GETONUM(p))->flags & FO_WIZARD) return true;
    return false;
}

PRIVATE OBJECT mkwizfun(OBJECT p, OBJECT setting) {
    long f;

    if (get_type(p) != T_OBJECT) error("type-error", newsym("make-wizard"));

    f = findobj(GETONUM(p))->flags;
    if (setting == false) f &= ~FO_WIZARD; else f |= FO_WIZARD;
    findobj(GETONUM(p))->flags = f;

    return NULL;
}

PRIVATE OBJECT progpfun(OBJECT p) {
    if (get_type(p) != T_OBJECT) return false;
    if (findobj(GETONUM(p))->flags & FO_PROGRAMMER) return true;
    return false;
}

PRIVATE OBJECT mkprogfun(OBJECT p, OBJECT setting) {
    long f;

    if (get_type(p) != T_OBJECT)
        error("type-error", newsym("make-programmer"));

    f = findobj(GETONUM(p))->flags;
    if (setting == false) f &= ~FO_PROGRAMMER; else f |= FO_PROGRAMMER;
    findobj(GETONUM(p))->flags = f;

    return NULL;
}

PRIVATE OBJECT getobjinfofun(OBJECT o) {
    OBJINFO oi;
    OBJECT info;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("object-info"));

    oi = findobj(GETONUM(o));
    info = NULL;
    if (oi->flags & FO_READ) info = cons(newsym("read"), info);
    if (oi->flags & FO_WRITE) info = cons(newsym("write"), info);
    if (oi->flags & FO_COPY) info = cons(newsym("copy"), info);

    return cons(oi->owner, info);
}

PRIVATE OBJECT setobjinfofun(OBJECT o, OBJECT info) {
    OBJINFO oi;
    OBJECT sv;

    if (get_type(o) != T_OBJECT)
        error("type-error", newsym("set-object-info"));
    if (!LISTP(info)) error("type-error", newsym("set-object-info"));

    oi = findobj(GETONUM(o));

    if (oi->owner != curinfo->code_owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
            error("no-permission", newsym("set-object-info"));

    oi->owner = CAR(info);
    oi->flags = 0;
    info = CDR(info);
    while (info != NULL) {
        if (get_type(CAR(info)) != T_SYM)
            error("type-error", newsym("set-object-info"));

        if (streq_os(CAR(info), "read")) oi->flags |= FO_READ; else
        if (streq_os(CAR(info), "write")) oi->flags |= FO_WRITE; else
        if (streq_os(CAR(info), "copy")) oi->flags |= FO_COPY; else
            error("type-error", newsym("set-object-info"));

        info = CDR(info);
    }

    sv = oi->slotvals;
    while (sv != NULL) {
        if (GETINT(CDR(CDR(CAR(sv)))) & F_CHANGEOWN)
            SETCDR(CAR(sv), cons(oi->owner, CDR(CDR(CAR(sv)))));
        sv = CDR(sv);
    }

    return NULL;
}

PRIVATE OBJECT getnamefun(OBJECT o) {
    if (get_type(o) != T_OBJECT) error("type-error", newsym("name"));
    return findobj(GETONUM(o))->name;
}

PRIVATE OBJECT setnamefun(OBJECT o, OBJECT n) {
    OBJINFO oi;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("set-name"));
    if (get_type(n) != T_STRING) error("type-error", newsym("set-name"));

    oi = findobj(GETONUM(o));

    if (curinfo->code_owner != oi->owner &&
        ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0))
            error("no-permission", newsym("set-name"));

    oi->name = n;
    return n;
}

PRIVATE OBJECT getlocfun(OBJECT o) {
    if (get_type(o) != T_OBJECT) error("type-error", newsym("location"));
    return findobj(GETONUM(o))->location;
}

PRIVATE OBJECT shallowcopy(OBJECT list) {
    OBJECT org, prev;

    org = NULL;
    prev = NULL;
    while (list != NULL) {
        if (prev == NULL) org = prev = cons(CAR(list), NULL); else {
            SETCDR(prev, cons(CAR(list), NULL));
            prev = CDR(prev);
        }
        list = CDR(list);
    }

    return org;
}

PRIVATE OBJECT getcontentsfun(OBJECT o) {
    if (get_type(o) != T_OBJECT) error("type-error", newsym("contents"));
    return shallowcopy(findobj(GETONUM(o))->contents);
}

PRIVATE OBJECT validobjfun(OBJECT o) {
    if (get_type(o) != T_OBJECT) return false;
    return BOOL(validobjnum(GETONUM(o)));
}

PRIVATE void del_specific_slot(OBJECT o, OBJECT slotname, OBJECT nextname) {
    OBJECT cn,cv, pn,pv;
    OBJINFO oi = findobj(GETONUM(o));

    pn = pv = NULL;
    cn = oi->slotnames;
    cv = oi->slotvals;
    while (cn != slotname) {
        if (cn == nextname) break;
        if (cn == NULL) return; /* It isn't in here, and not in it's kids. */
        pn = cn; cn = CDR(cn);
        pv = cv; cv = CDR(cv);
    }

    if (cn == nextname) {
        if (pv != NULL) SETCDR(pv, CDR(cv));
    } else {
        if (pn == NULL) {
            oi->slotnames = CDR(cn);
            oi->slotvals = CDR(cv);
        } else {
            SETCDR(pn, CDR(cn));
            SETCDR(pv, CDR(cv));
        }
    }

    cn = oi->children;
    while (cn != NULL) {
        del_specific_slot(CAR(cn), slotname, nextname);
        cn = CDR(cn);
    }
}

PRIVATE OBJECT remslotfun(OBJECT o, OBJECT slotname) {
    OBJECT sn,sv,np,vp,pn;
    OBJINFO oi;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("remove-slot"));
    if (get_type(slotname) != T_SYM)
        error("type-error", newsym("remove-slot"));

    oi = findobj(GETONUM(o));

    if (curinfo->code_owner != oi->owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0 &&
        (oi->flags & FO_WRITE) == 0)
            error("no-permission", newsym("remove-slot"));

    pn = findobj(GETONUM(oi->parent))->slotnames;
    sn = oi->slotnames;
    sv = oi->slotvals;
    np = vp = NULL;
    while (sn != pn) {
        if (streq_oo(CAR(sn), slotname)) {
            OBJECT child;

            if (np == NULL) oi->slotnames = CDR(sn); else SETCDR(np,CDR(sn));
            if (vp == NULL) oi->slotvals = CDR(sv); else SETCDR(vp,CDR(sv));

            child = oi->children;
            while (child != NULL) {
                del_specific_slot(CAR(child), sn, CDR(sn));
                child = CDR(child);
            }

            return NULL;
        }
        np = sn;
        sn = CDR(sn);
        vp = sv;
        sv = CDR(sv);
    }
    error("slot-not-found", newsym("remove-slot"));
    return NULL;
}

PRIVATE OBJECT remmethfun(OBJECT o, OBJECT methname) {
    OBJECT mn, mv, np, vp;
    OBJINFO oi;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("remove-method"));
    if (get_type(methname) != T_SYM)
        error("type-error", newsym("remove-method"));

    oi = findobj(GETONUM(o));

    if (curinfo->code_owner != oi->owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0 &&
        (oi->flags & FO_WRITE) == 0)
            error("no-permission", newsym("remove-method"));

    mn = oi->methnames;
    mv = oi->methvals;
    np = vp = NULL;
    while (mn != NULL) {
        if (streq_oo(CAR(mn), methname)) {
            if (np == NULL) oi->methnames = CDR(mn); else SETCDR(np,CDR(mn));
            if (vp == NULL) oi->methvals = CDR(mv); else SETCDR(vp,CDR(mv));
            return NULL;
        }
        np = mn;
        mn = CDR(mn);
        vp = mv;
        mv = CDR(mv);
    }
    error("method-not-found", newsym("remove-method"));
    return NULL;
}

PRIVATE OBJECT childrenfun(OBJECT o) {
    if (get_type(o) != T_OBJECT) error("type-error", newsym("children"));
    return shallowcopy(findobj(GETONUM(o))->children);
}

PRIVATE OBJECT getslotsfun(OBJECT o) {
    OBJINFO oi;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("slots"));

    oi = findobj(GETONUM(o));

    if (curinfo->code_owner != oi->owner &&
        ((oi->flags & FO_READ) == 0))
            error("no-permission", newsym("slots"));

    return shallowcopy(oi->slotnames);
}

PRIVATE OBJECT getslotvalfun(OBJECT o, OBJECT sn) {
    OBJECT cell;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("slot-val"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("slot-val"));

    cell = findslot(findobj(GETONUM(o)), GETSTR(sn));
    if (cell == NULL) return NULL;

    if (CAR(CDR(cell)) != curinfo->code_owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
            if ((GETINT(CDR(CDR(cell))) & F_READ) == 0)
                error("no-permission", newsym("slot-val"));

    return CAR(cell);
}

PRIVATE OBJECT setslotvalfun(OBJECT o, OBJECT sn, OBJECT val) {
    OBJECT cell;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("set-slot-val"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("set-slot-val"));

    cell = findslot(findobj(GETONUM(o)), GETSTR(sn));
    if (cell == NULL) return false;

    if (CAR(CDR(cell)) != curinfo->code_owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
            if ((GETINT(CDR(CDR(cell))) & F_WRITE) == 0)
                error("no-permission", newsym("set-slot-val"));

    SETCAR(cell, val);
    return true;
}

PRIVATE OBJECT getslotinfofun(OBJECT o, OBJECT sn) {
    OBJECT cell, info;
    long flags;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("slot-info"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("slot-info"));

    cell = findslot(findobj(GETONUM(o)),GETSTR(sn));
    if (cell == NULL) error("slot-not-found", newsym("slot-info"));

    info = NULL;
    flags = GETINT(CDR(CDR(cell)));

    if (curinfo->code_owner != CAR(CDR(cell)) &&
        ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0) &&
        ((flags & F_READ) == 0))
            error("no-permission", newsym("slot-info"));

    if (flags & F_CHANGEOWN) info = cons(newsym("changeown"), info);
    if (flags & F_WRITE) info = cons(newsym("write"), info);
    if (flags & F_READ) info = cons(newsym("read"), info);

    return cons(CAR(CDR(cell)), info);
}

PRIVATE void fixup_owners(OBJECT o, OBJECT sn, OBJECT info, long flags) {
    OBJINFO oi = findobj(GETONUM(o));
    OBJECT c = findslot(oi, GETSTR(sn));

    if (flags & F_CHANGEOWN)
        SETCDR(c, cons(oi->owner, CDR(info)));
    else
        SETCDR(c, info);

    c = oi->children;
    while (c != NULL) {
        fixup_owners(CAR(c), sn, info, flags);
        c = CDR(c);
    }
}

PRIVATE OBJECT setslotinfofun(OBJECT o, OBJECT sn, OBJECT info) {
    OBJECT cell;
    long flags;
    OBJECT owner;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("set-slot-info"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("set-slot-info"));

    cell = findslot(findobj(GETONUM(o)),GETSTR(sn));
    if (cell == NULL) error("slot-not-found", newsym("set-slot-info"));

    if (curinfo->code_owner != CAR(CDR(cell)) &&
        ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0) &&
        ((GETINT(CDR(CDR(cell))) & F_WRITE) == 0))
            error("no-permission", newsym("set-slot-info"));

    flags = 0;
    owner = CAR(info);
    info = CDR(info);
    while (info != NULL) {
        if (streq_os(CAR(info), "read")) flags |= F_READ;
        if (streq_os(CAR(info), "write")) flags |= F_WRITE;
        if (streq_os(CAR(info), "changeown")) flags |= F_CHANGEOWN;
        info = CDR(info);
    }
    info = CDR(cell);
    SETCAR(info, owner);
    SETCDR(info, newint(flags));

    fixup_owners(o, sn, info, flags);
    return NULL;
}

PRIVATE OBJECT getmethvalfun(OBJECT o, OBJECT sn) {
    OBJECT val, cell, class;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("method-val"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("method-val"));

    val = findmeth(findobj(GETONUM(o)), GETSTR(sn), &cell, &class);
    if (val == NULL) return false;

    if (CAR(CDR(cell)) != curinfo->code_owner &&
        (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0)
            if ((GETINT(CDR(CDR(cell))) & F_READ) == 0)
                error("no-permission", newsym("method-val"));

    return cons(val, cons(CAR(CDR(cell)), cons(class, NULL)));
}

PRIVATE OBJECT getmethinfofun(OBJECT o, OBJECT sn) {
    OBJECT cell, info, c;
    long flags;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("method-info"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("method-info"));

    if (findmeth(findobj(GETONUM(o)),GETSTR(sn),&cell,&c) == NULL)
        error("method-not-found", newsym("method-info"));

    info = NULL;
    flags = GETINT(CDR(CDR(cell)));

    if (curinfo->code_owner != CAR(CDR(cell)) &&
        ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0) &&
        ((flags & F_READ) == 0))
            error("no-permission", newsym("method-info"));

    if (flags & F_WRITE) info = cons(newsym("write"), info);
    if (flags & F_READ) info = cons(newsym("read"), info);

    return cons(CAR(CDR(cell)), info);
}

PRIVATE OBJECT setmethinfofun(OBJECT o, OBJECT sn, OBJECT info) {
    OBJECT cell, c;
    long flags;
    OBJECT owner;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("set-method-info"));
    if (get_type(sn) != T_SYM) error("type-error", newsym("set-method-info"));
    if (!LISTP(info)) error("type-error", newsym("set-method-info"));

    if (findmeth(findobj(GETONUM(o)),GETSTR(sn),&cell,&c) == NULL)
        error("method-not-found", newsym("set-method-info"));

    if (curinfo->code_owner != CAR(CDR(cell)) &&
        ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0) &&
        ((GETINT(CDR(CDR(cell))) & F_WRITE) == 0))
            error("no-permission", newsym("set-method-info"));

    flags = 0;
    owner = CAR(info);
    info = CDR(info);
    while (info != NULL) {
        if (streq_os(CAR(info), "read")) flags |= F_READ;
        if (streq_os(CAR(info), "write")) flags |= F_WRITE;
        info = CDR(info);
    }
    info = cons(owner, newint(flags));

    SETCDR(cell, info);
    return NULL;
}

PRIVATE OBJECT getmethodsfun(OBJECT o) {
    OBJINFO oi;

    if (get_type(o) != T_OBJECT) error("type-error", newsym("methods"));

    oi = findobj(GETONUM(o));

    if (curinfo->code_owner != oi->owner &&
        ((oi->flags & FO_READ) == 0))
            error("no-permission", newsym("methods"));

    return shallowcopy(findobj(GETONUM(o))->methnames);
}

PRIVATE int indirect_member(OBJECT list, OBJECT what) {
    while (list != NULL) {
        if (CAR(list) == what) return 1;
        if (get_type(CAR(list)) == T_CONS)
            return indirect_member(CAR(list), what);
        list = CDR(list);
    }
    return 0;
}

PRIVATE OBJECT movefun(OBJECT what, OBJECT where) {
    OBJINFO ai, ei;
    OBJECT oldloc;
    OBJECT oldown;
    int callerwizard;

    if (get_type(what) != T_OBJECT) error("type-error", newsym("move"));
    if (get_type(where) != T_OBJECT) error("type-error", newsym("move"));

    ai = findobj(GETONUM(what));
    if (GETONUM(where) != -1) ei = findobj(GETONUM(where)); else ei = NULL;

    callerwizard =
        (int) (findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD);

    if (curinfo->code_owner != ai->owner && !callerwizard)
        error("no-permission", newsym("move"));

    oldown = curinfo->code_owner;
    curinfo->code_owner = MKONUM(3);

    if (ei != NULL) {
        OBJECT res = callmethod(where, "accept", 1, what);
        if (!callerwizard && res == false)
            error("move-rejected", NULL);
    }

    if (indirect_member(ai->contents, where) || what == where)
        error("move-recursive", NULL);

    oldloc = ai->location;
    ai->location = where;

    if (GETONUM(oldloc) != -1) {
        OBJINFO ol = findobj(GETONUM(oldloc));
        ol->contents = deletefun(ol->contents, what);
    }

    if (ei != NULL) ei->contents = cons(what, ei->contents);

    if (GETONUM(oldloc) != -1) callmethod(oldloc, "exitfunc", 2, what, where);

    if (ei != NULL)
        if (!validobjnum(GETONUM(where))) {
            curinfo->code_owner = oldown;
            return NULL;
        }
    if (!validobjnum(GETONUM(what))) {
        curinfo->code_owner = oldown;
        return NULL;
    }
    if (ai->location != where) {
        curinfo->code_owner = oldown;
        return NULL;
    }

    if (ei != NULL) callmethod(where, "enterfunc", 2, what, oldloc);

    curinfo->code_owner = oldown;
    return NULL;
}

PRIVATE OBJECT setparentfun(OBJECT obj, OBJECT parent) {
    OBJECT currname, currval;
    OBJECT prevname, prevval;
    OBJINFO oi, pi, opi;

    if (get_type(obj) != T_OBJECT) error("type-error", newsym("set-parent"));
    if (get_type(parent) != T_OBJECT)
        error("type-error", newsym("set-parent"));

    if ((findobj(GETONUM(curinfo->code_owner))->flags & FO_WIZARD) == 0) {
        if (curinfo->code_owner != findobj(GETONUM(obj))->owner)
            error("no-permission", newsym("set-parent"));

        if (curinfo->code_owner != findobj(GETONUM(parent))->owner)
            if ((findobj(GETONUM(parent))->flags & FO_COPY) == 0)
                error("no-permission", newsym("set-parent"));
    }

    oi = findobj(GETONUM(obj));
    pi = findobj(GETONUM(parent));
    opi = findobj(GETONUM(oi->parent));

    currname = oi->slotnames; prevname = NULL;
    currval = oi->slotvals; prevval = NULL;
    while (currname != NULL) {
        if (currname == opi->slotnames) {
            if (prevname == NULL) {
                oi->slotnames = pi->slotnames;
                oi->slotvals = values_copy(pi->slotvals, obj);
            } else {
                SETCDR(prevname, pi->slotnames);
                SETCDR(prevval, values_copy(pi->slotvals, obj));
            }
            break;
        }
        prevname = currname; currname = CDR(currname);
        prevval = currval; currval = CDR(currval);
    }

    if (currname == NULL) {
        if (prevname == NULL) {
            oi->slotnames = pi->slotnames;
            oi->slotvals = values_copy(pi->slotvals, obj);
        } else {
            SETCDR(prevname, pi->slotnames);
            SETCDR(prevval, values_copy(pi->slotvals, obj));
        }
    }

    opi->children = deletefun(opi->children, obj);

    if (listidxfun(pi->children, obj) == false)
        pi->children = cons(obj, pi->children);

    oi->parent = parent;

    return NULL;
}

PRIVATE int random_printable(void) {
    unsigned char ch;

    do ch = random(256); while (!isalnum(ch) || ch < 32);
        /* while !printable or !control-character, get a new one */
    return (int) ch;
}

PRIVATE OBJECT cryptfun(OBJECT str, OBJECT salt) {
    char *s;

    if (get_type(str) != T_STRING) error("type-error", newsym("crypt"));
    if (get_type(salt) != T_STRING) error("type-error", newsym("crypt"));

    s = getmem(strlen(GETSTR(str))+3);
    strcpy(s + 2, GETSTR(str));

    if (strlen(GETSTR(salt)) < 2) {
        s[0] = random_printable();
        s[1] = random_printable();
    } else {
        s[0] = GETSTR(salt)[0];
        s[1] = GETSTR(salt)[1];
    }

    str = newstring(crypt(s, s+2, s));
    freemem(s);
    return str;
}

PRIVATE OBJECT asnumfun(OBJECT o) {
    switch (get_type(o)) {
        case T_INT: return o;
        case T_OBJECT: return newint(GETONUM(o));
        case T_STRING: return newint(atol(GETSTR(o)));
        default:
            error("type-error", newsym("as-num"));
    }
    return newint(0);
}

PRIVATE OBJECT asobjfun(OBJECT o) {
    switch (get_type(o)) {
        case T_INT: return MKONUM(GETINT(o));
        case T_OBJECT: return o;
        case T_STRING: {
            char *s = GETSTR(o);
            if (*s == '#') s++;
            return MKONUM(atol(s));
        }
        default:
            error("type-error", newsym("as-obj"));
    }
    return MKONUM(-1);
}

PRIVATE OBJECT asstrfun(OBJECT o) {
    char s[256];
    switch (get_type(o)) {
        case T_NULL: sprintf(s,"[]"); break;
        case T_INT: sprintf(s,"%ld",GETINT(o)); break;
        case T_OBJECT: sprintf(s,"#%ld",GETONUM(o)); break;
        case T_STRING: return o;
        case T_SYM: return newstring(GETSTR(o));
        case T_CONS: strcpy(s,"[a list]"); break;
        case T_VECTOR: strcpy(s,"{a vector}"); break;
        case T_FUNC: strcpy(s,"!!function"); break;
        case T_PRIM: strcpy(s,"!!prim"); break;
        case T_CONN: strcpy(s,"!!connection"); break;
    }
    return newstring(s);
}

PRIVATE OBJECT aslistfun(OBJECT o) {
    word i;
    OBJECT l = NULL;
    if (get_type(o) != T_VECTOR) error("type-error", newsym("as-list"));
    for ( i = GETLEN(o)-1 ; i > 0 ; i-- ) l = cons(GETELEM(o, i), l);
    return l;
}

PRIVATE OBJECT typeoffun(OBJECT o) {
    switch (get_type(o)) {
        case T_NULL: return newsym("null");
        case T_INT: return newsym("number");
        case T_STRING: return newsym("string");
        case T_SYM: return newsym("symbol");
        case T_CONS: return newsym("pair");
        case T_FUNC: return newsym("function");
        case T_PRIM: return newsym("primitive");
        case T_VECTOR: return newsym("vector");
        case T_CONN: return newsym("connection");
        case T_OBJECT: return newsym("object");
    }
    fatal("Unknown type in type-of: %d",get_type(o));
    return NULL;
}

_prim primtab[] = {
    { 1, "print", print },                          /**/
    { 0, "read", readfun },                         /**/
    { 2, "print-on", printon },                     /**/
    { 1, "read-from", readfromfun },                /**/
    { 1, "disconnect", disconfun },                 /**/

    { 1, "clone", clone },                          /**/
    { 1, "recycle", recycle },                      /**/
    { 0, "max-object", maxobj },                    /**/

    { 0, "task-id", gettaskidfun },                 /**/
    { 0, "ticks-left", ticksleft },                 /**/
    { 1, "valid-task", validtaskfun },              /**/
    { 1, "kill-task", killtask },                   /**/

    { 0, "time", timefun },                         /**/
    { 0, "time-string", timestrfun },               /**/

    { 0, "shutdown", shutdown },                    /**/
    { 0, "kill-server!!", killserver },             /**/
    { 1, "log", syslog },                           /**/

    { 1, "vector", listtovect },                    /**/
    { 2, "make-vector", mkvectfun },                /**/
    { 2, "make-string", mkstrfun },                 /**/
    { 2, "splice", cons },                          /**/
    { 1, "head", head },                            /**/
    { 2, "set-head", sethead },                     /**/
    { 1, "tail", tail },                            /**/
    { 2, "set-tail", settail },                     /**/
    { 1, "length", lenfun },                        /**/
    { 2, "append!", appendfun },                    /**/
    { 2, "delete!", deletefun },                    /**/
    { 2, "element", getelemfun },                   /**/
    { 3, "set-elem!", setelemfun },                 /**/
    { 2, "listidx", listidxfun },                   /**/
    { 3, "section", sectionfun },                   /**/
    { 2, "strcmp", strcmpfun },                     /**/
    { 2, "stridx", stridxfun },                     /**/
    { 3, "strsub", strsubfun },                     /**/
    { 1, "toupper", toupperfun },                   /**/
    { 1, "tolower", tolowerfun },                   /**/

    { 2, "equal?", equalfun },                      /**/
    { 2, "greater-than", gtfun },                   /**/
    { 2, "less-than", ltfun },                      /**/

    { 2, "add", addfun },                           /**/
    { 2, "subtract", subfun },                      /**/
    { 2, "multiply", mulfun },                      /**/
    { 2, "divide", divfun },                        /**/
    { 2, "modulo", modfun },                        /**/
    { 1, "abs", absfun },                           /**/
    { 1, "random", randomfun },                     /**/

    { 0, "connection", connectionfun },             /**/
    { 0, "player", playerfun },                     /**/
    { 1, "push-handler", pushhandlerfun },          /**/
    { 0, "pop-handler", pophandlerfun },            /**/

    { 1, "evaluate", evalfun },                     /**/
    { 0, "caller", callerfun },                     /**/
    { 0, "server-version", getverfun },             /**/
    { 2, "error", errfun },                         /**/
    { 0, "abort", abortfun },                       /**/
    { 1, "set-code-owner", setcodeownerfun },       /**/
    { 5, "simulate-method-call", simmethcallfun },  /**/

    { 1, "player?", playerpfun },                   /**/
    { 2, "make-player", mkplayerfun },              /**/
    { 1, "wizard?", wizardpfun },                   /**/
    { 2, "make-wizard", mkwizfun },                 /**/
    { 1, "programmer?", progpfun },                 /**/
    { 2, "make-programmer", mkprogfun },            /**/
    { 1, "object-info", getobjinfofun },            /**/
    { 2, "set-object-info", setobjinfofun },        /**/
    { 1, "name", getnamefun },                      /**/
    { 2, "set-name", setnamefun },                  /**/
    { 1, "location", getlocfun },                   /**/
    { 1, "contents", getcontentsfun },              /**/
    { 1, "valid", validobjfun },                    /**/
    { 2, "remove-slot", remslotfun },               /**/
    { 2, "remove-method", remmethfun },             /**/
    { 1, "children", childrenfun },                 /**/
    { 1, "slots", getslotsfun },                    /**/
    { 2, "slot-val", getslotvalfun },               /**/
    { 3, "set-slot-val", setslotvalfun },           /**/
    { 2, "slot-info", getslotinfofun },             /**/
    { 3, "set-slot-info", setslotinfofun },         /**/
    { 2, "method-val", getmethvalfun },             /**/
    { 2, "method-info", getmethinfofun },           /**/
    { 1, "methods", getmethodsfun },                /**/
    { 3, "set-method-info", setmethinfofun },       /**/
    { 2, "move", movefun },                         /**/
    { 2, "set-parent", setparentfun },              /**/

    { 2, "crypt", cryptfun },                       /**/

    { 1, "as-num", asnumfun },                      /**/
    { 1, "as-obj", asobjfun },                      /**/
    { 1, "as-str", asstrfun },                      /**/
    { 1, "as-list", aslistfun },                    /**/
    { 1, "type-of", typeoffun },                    /**/

    { 0, NULL, NULL }
};

void init_prim(void) {
    randomize();
}

