#include "oom.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

PRIVATE char *keywords[] = {
    "define",
    "slot",
    "method",
    "as",

    "if",
    "then",
    "else",

    "begin",
    "end",

    "while",
    "until",
    "do",

    "return",

    "after",

    "function",

    "not",
    "or",
    "and",

    "null",
    "true",
    "false",

    "parent",

    NULL
};

PRIVATE int keyword(char *s) {
    int i = 0;

    while (keywords[i] != NULL) {
        if (streq_ss(s,keywords[i])) return 1;
        i++;
    }
    return 0;
}

#define START           0
#define ID              1
#define NUM             2
#define STRING          3

#define GO(st)  { state = st; continue; }

#define EMIT_P(c)   EMIT_S(s,c,NULL)
#define D_EMIT_P(c) { scan_drop(s); EMIT_S(s,c,NULL); }
#define SECOND(se,t,f)          \
    if (scan_peek(s) == se)     \
        t else                  \
        f

#define PUNCTSTR    "+-*/%<>!=;:.,'#[](){}$\""
#define DELIMSTR            "=;:.,'#[](){}$\""

void code_scanner(SCANSTATE s) {
    int state = START;
    char c;
    char *str;
    long num = 0;
    int radix = 10;

    while (1) {
        switch (state) {
            case START:
                c = scan_peek(s);

                if (c_eof(s->conn)) EMIT_S(s,"end-of-file",NULL)

                if (isspace(c)) {
                    scan_drop(s);
                    if (c == '\n') s->linenum++;
                    GO(START);
                }

                if (isdigit(c)) GO(NUM);
                if (strchr(PUNCTSTR,c) == NULL) GO(ID);

                scan_drop(s);

                if (c == '"') GO(STRING);
                
                switch (c) {
                    case '+': EMIT_P("+")
                    case '-':
                        if (scan_peek(s) == '-') {
                            scan_drop(s);
                            while (scan_peek(s) != '\n') scan_drop(s);
                            GO(START);
                        }
                        EMIT_P("-")
                    case '*': EMIT_P("*")
                    case '/': EMIT_P("/")
                    case '%': EMIT_P("%")
                    case '<': SECOND('=',D_EMIT_P("<="),EMIT_P("<"))
                    case '>': SECOND('=',D_EMIT_P(">="),EMIT_P(">"))
                    case '!': SECOND('=',D_EMIT_P("!="),EMIT_P("!"))
                    case '=': SECOND('=',D_EMIT_P("=="),EMIT_P("="))
                    case ';': EMIT_P(";")
                    case ':': EMIT_P(":")
                    case '.': EMIT_P(".")
                    case ',': EMIT_P(",")
                    case '\'': EMIT_P("'")
                    case '#': EMIT_P("#")
                    case '[': EMIT_P("[")
                    case ']': EMIT_P("]")
                    case '(': EMIT_P("(")
                    case ')': EMIT_P(")")
                    case '{': EMIT_P("{")
                    case '}': EMIT_P("}")
                    case '$': EMIT_P("$")
                }
                break;
            case ID:
                c = scan_peek(s);
                if ((strchr(DELIMSTR,c) == NULL) && !isspace(c)) {
                    scan_shift(s);
                    GO(ID);
                }
                str = scan_buf(s);
                if (keyword(str)) EMIT_S(s, str, NULL) else
                                  EMIT_S(s, "identifier", str)
            case NUM:
                c = toupper(scan_peek(s));
                if (isdigit(c) || (c >= 'A' && c <= 'F')) {
                    scan_drop(s);
                    num *= radix;
                    if (c >= 'A')
                        num += c - 'A' + 10; else
                        num += c - '0';
                    GO(NUM);
                }
                if (c == 'X') {
                    scan_drop(s);
                    if (radix == 16 || num != 0)
                        error("Invalid format of hexadecimal number",NULL);
                    radix = 16;
                    GO(NUM);
                }
                EMIT_L(s, "number", num)
            case STRING:
                c = scan_peek(s);
                if (c == '"') {
                    scan_drop(s);
                    while (isspace(c = scan_peek(s))) {
                        if (c == '\n') s->linenum++;
                        scan_drop(s);
                    }
                    if (c == '"') {
                        scan_drop(s);
                        GO(STRING);
                    }
                    EMIT_S(s, "string", scan_buf(s))
                }
                if (c == '\\') {
                    scan_drop(s);
                    c = scan_peek(s);
                    scan_drop(s);
                    switch (c) {
                        case 'r': scan_insert(s,'\r'); break;
                        case 'n': scan_insert(s,'\n'); break;
                        case 't': scan_insert(s,'\t'); break;
                        case 'b': scan_insert(s,'\b'); break;
                        case 'a': scan_insert(s,'\a'); break;
                        case '"':
                        case '\\': scan_insert(s,c); break;
                    }
                } else if (c == '\n')
                    error("String split across lines", NULL);
                else
                    scan_shift(s);
                break;
        }
    }
}

