#include "oom.h"

#include "serial.h"
#include "com.h"

#include <time.h>

#define BUFLEN 512

PRIVATE COMPORT port1, port2;

PRIVATE char *buf[2];
PRIVATE int len[2], buflen[2];
PRIVATE int pos[2], eof[2], ungotten[2];
PRIVATE int reading[2];

void serial_startup(void) {
    int i;

    port1 = newcomport(1, 2400, 8, 'N', 1);
    port2 = newcomport(2, 2400, 8, 'N', 1);

    for (i=0; i<2; i++) {
        pos[i] = len[i] = buflen[i] = eof[i] = 0;
        buf[i] = NULL;
        reading[i] = 0;
        ungotten[i] = -1;
    }
}

void serial_shutdown(void) {
    killcomport(port1);
    killcomport(port2);
}

void serial_open(OBJECT conn) {
    if (GETCINFO(conn) == 1) {
        killcomport(port1);
        port1 = newcomport(1, GETCONNL(conn), 8, 'N', 1);
    } else {
        killcomport(port2);
        port2 = newcomport(2, GETCONNL(conn), 8, 'N', 1);
    }
}

void serial_close(OBJECT conn) {
}

void serial_reset(OBJECT conn) {
    int pn = GETCINFO(conn)-1;
    COMPORT p = ((pn == 0) ? port1 : port2);
    long t;

    while (!comport_carrier(p) && !shutting_down) swap();
    if (shutting_down) exit_proc();

    t = clock();
    while (clock() - t < 36) swap();

    while (comport_avail(p)) comport_getc(p);

    pos[pn] = eof[pn] = len[pn] = buflen[pn] = 0;
    if (buf[pn] != NULL) freemem(buf[pn]);
    buf[pn] = NULL;
    reading[pn] = 0;
    ungotten[pn] = '\n';
}

PRIVATE void clean_up_carrier_drop(void) {
    if (setjmp(curinfo->errjmp) == 0 && !curinfo->logging_in) {
        OBJECT temp = curinfo->player;

        curinfo->code_owner = curinfo->player = MKONUM(3);
        curinfo->stack = newstack(INTERP_OBJSTACKSIZE);

        callmethod(MKONUM(0), "do-logout", 1, temp);

        curinfo->stack = NULL;
    }
}

void serial_disconnect(OBJECT conn) {
    COMPORT p = ((GETCINFO(conn) == 1) ? port1 : port2);

    if (GETCINFO(conn) == 0) return;    /* Already disconnected. */

    GETCINFO(conn) = 0;
    clean_up_carrier_drop();
    comport_setdtr(p, 0);
}

void serial_dump(OBJECT conn, FILE *f) {
}

void serial_load(OBJECT conn, FILE *f) {
    GETCINFO(conn) = 0; /* Serial connections are not allowed to be reloaded */
}

void serial_mark(OBJECT conn) {
}

PRIVATE void check_carrier(OBJECT conn) {
    COMPORT port;
    if (GETCINFO(conn) == 0) exit_proc();
    port = ((GETCINFO(conn) == 1) ? port1 : port2);
    if (!comport_carrier(port) || shutting_down) {
        GETCINFO(conn) = 0;
        clean_up_carrier_drop();
        exit_proc();
    }
}

PRIVATE int cg(OBJECT conn) {
    COMPORT port = ((GETCINFO(conn) == 1) ? port1 : port2);
    while (!comport_avail(port)) {
        check_carrier(conn);
        swap();
    }
    check_carrier(conn);
    return comport_getc(port);
}

int serial_getc(OBJECT conn) {
    int ch;
    int pn = GETCINFO(conn)-1;

    if (ungotten[pn] != -1) {
        int tmp = ungotten[pn];
        ungotten[pn] = -1;
        return tmp;
    }

    while (reading[pn]) swap();

    if (len[pn] > pos[pn]) {
        int tmp = buf[pn][pos[pn]];
        pos[pn]++;
        if (tmp == 26) {
            eof[pn] = 1;
            return ' ';
        }
        return tmp;
    }

    pos[pn] = eof[pn] = len[pn] = buflen[pn] = 0;
    if (buf[pn] != NULL) freemem(buf[pn]);
    buf[pn] = NULL;
    ungotten[pn] = -1;

    reading[pn] = 1;

    ch = cg(conn);

    while (ch != '\r' || len[pn] == 0) {
        switch (ch) {
            case 0:
                ch = cg(conn);
                break;
            case 3:
                break;
            case 8:
                if (len[pn] > 0) {
                    len[pn]--;

                    switch (buf[pn][len[pn]]) {
                        case 9: c_printf(conn, "\b\b\b\b"); break;
                        case 26: c_printf(conn, "\b\b  \b\b"); break;
                        default: c_printf(conn, "\b \b"); break;
                    }
                }
                break;
            case 27:
                c_printf(conn, "\\\n");
                len[pn] = 0;
                break;
            case 26:
                c_printf(conn, "^Z");
                /* FALL THROUGH */
            default:
                if (len[pn] >= buflen[pn]) {
                    buf[pn] = growmem(buf[pn], buflen[pn], 256);
                    buflen[pn] += 256;
                }
                buf[pn][len[pn]++] = ch;
                if (ch == 9)
                    c_printf(conn, "    ");
                else
                    if (ch == 26)
                        ;
                    else
                        c_putc(ch, conn);
        }
        if (ch != '\r') ch = cg(conn);
        if (ch == '\r') {
            c_putc('\n', conn);
            if (len[pn] >= buflen[pn]) {
                buf[pn] = growmem(buf[pn], buflen[pn], 256);
                buflen[pn] += 256;
            }
            buf[pn][len[pn]++] = '\n';
        }
    }
    reading[pn] = 0;
    pos[pn] = 1;
    if (buf[pn][0] == 26) {
        eof[pn] = 1;
        return ' ';
    }
    return buf[pn][0];
}

void serial_putc(int ch, OBJECT conn) {
    COMPORT port = ((GETCINFO(conn) == 1) ? port1 : port2);
    check_carrier(conn);
    while (!comport_tx_ready(port)) {
        check_carrier(conn);
        swap();
    }
    check_carrier(conn);
    comport_putc(ch, port);
    if (ch == '\n') comport_putc('\r', port);
        /* For stupid CRLF terminals. */
        /* This should really be either a #define or a runtime check. */
}

void serial_ungetc(int ch, OBJECT conn) {
    check_carrier(conn);
    ungotten[GETCINFO(conn)-1] = ch;
}

int serial_ready(OBJECT conn) {
    int pn = GETCINFO(conn) - 1;
    check_carrier(conn);
    return len[pn] > pos[pn] || comport_avail(pn == 0 ? port1 : port2);
}

int serial_carrier(OBJECT conn) {
    if (shutting_down) exit_proc();
    if (GETCINFO(conn) == 0) return 0;
    return comport_carrier(GETCINFO(conn) == 1 ? port1 : port2);
}

int serial_eof(OBJECT conn) {
    check_carrier(conn);
    return eof[GETCINFO(conn)-1];
}

