#include "slang.h"
#include <ctype.h>

#define MAX_TOKEN_LEN   100
#define MAX_STR_LEN     256

PRIVATE char readc(OBJECT f) {
    if (l_feof(f)) return EOF;
    return l_getc(f);
}

PRIVATE char trim(OBJECT f) {
    bool cmt = FALSE;
    char c;

    while (TRUE) {
        c = readc(f);
        if (c == EOF) return c;
        if (cmt) {
            if (c == '\n') cmt = FALSE;
        } else
            if (c == ';') cmt = TRUE; else
            if (!isspace(c)) return c;
    }
}

PRIVATE OBJECT do_read(OBJECT f);       /* Prototype */

PRIVATE OBJECT read_list(OBJECT f) {
    char c;
    OBJECT org = NULL, pp = NULL;
    OBJECT r;

    while (TRUE) {
        c = trim(f);
        if (c == EOF) {
            error("EOF in a list!",NULL);
            return UNDEFD;
        }
        if (c == ']' || c == '}') {
            if (!NULLP(pp)) {
                CDR(pp) = EOL;
                return org;
            }
            return EOL;
        }
        l_ungetc(c,f);
        r = do_read(f);
        if (SYMP(r) && streq_os(r,".")) {
            r = do_read(f);
            c = trim(f);
            if (c == EOF) {
                error("EOF in a list!",NULL);
                return UNDEFD;
            }
            if (c != ']' && c != '}') {
                error("Missing close-bracket in dotted list or pair",NULL);
                return UNDEFD;
            }
            if (!NULLP(pp)) {
                CDR(pp) = r;
                return org;
            }
            return r;
        }
        if (!NULLP(pp)) {
            CDR(pp) = cons(r,EOL);
            pp = CDR(pp);
        } else org = pp = cons(r,EOL);
    }
}

PRIVATE OBJECT read_string(OBJECT f) {
    char c;
    char buf[MAX_STR_LEN];
    int buf_pos = 0;

    while ((c = readc(f)) != '"' && c != EOF) {
        if (c == '\\') {
            c = readc(f);
            if (c == EOF) {
                error("EOF after a \\",NULL);
                return UNDEFD;
            }
            switch (c) {
                case 'n': case 'N': c = '\n'; break;
                case 't': case 'T': c = '\t'; break;
                case 'r': case 'R': c = '\r'; break;
                default:
                    l_ungetc(c,f);
                    c = '\\';
            }
        }
        if (c == EOF) {
            error("EOF in a string",NULL);
            return UNDEFD;
        }
        buf[buf_pos++] = c;
        buf[buf_pos] = 0;
        if (buf_pos >= MAX_STR_LEN - 1) {
            error("String exceeds limits on length",NULL);
            return UNDEFD;
        }
    }
    return newstrobj(T_STRING,buf);
}

PRIVATE OBJECT intern_token(char *s) {
    if (isdigit(*s) || (*s == '-' && isdigit(*(s+1)))) {
        if (strchr(s,'.')) return newfloat((FLOAT) atof(s));
        return newint(atol(s));
    }
    strlwr(s);
    if (streq_os(S_TRUE,s)) return S_TRUE;
    if (streq_os(S_FALSE,s)) return S_FALSE;
    if (streq_os(MARK,s)) return MARK;
    return newstrobj(T_SYM,s);
}

extern void listtovectfun(void);

PRIVATE OBJECT read_object_buf = NULL;

PRIVATE OBJECT do_read(OBJECT f) {
    char c;
    char buf[MAX_TOKEN_LEN];
    int buf_pos = 0;

    if (!NULLP(read_object_buf)) {
        OBJECT r = read_object_buf;
        read_object_buf = NULL;
        return r;
    }

    buf[0] = 0;
    c = trim(f);
    if (c == EOF) {
        error("EOF in read",NULL);
        return UNDEFD;
    }
    switch (c) {
        case '[': return read_list(f);
        case '{': {
            OBJECT l = read_list(f);
            read_object_buf = newstrobj(T_SYM,"make-static");
            return l;
        }
        case ']':
        case '}':
            error("Too many close-brackets in",f);
            return UNDEFD;
        case '#':
            c = readc(f);
            switch (c) {
                case '[':
                    push(read_list(f));
                    listtovectfun();
                    return pop();
                case '{': return newclosure(read_list(f),UNDEFD);
            }
            l_ungetc(c,f);
            c = '#';
            break;
        case '!':
            c = readc(f);
            switch (c) {
                case '{': {
                    OBJECT l = read_list(f);
                    read_object_buf = newstrobj(T_SYM,"make-recursive");
                    return l;
                }
            }
            l_ungetc(c,f);
            c = '!';
            break;
        case '"': return read_string(f);
    }
    buf[buf_pos] = c;
    for (buf_pos = 1; buf_pos <= MAX_TOKEN_LEN - 1; buf_pos++) {
        c = readc(f);
        if (c == EOF || isspace(c)) {
            buf[buf_pos] = 0;
            return intern_token(buf);
        }
        if (strchr("!#{[]}';\"",c)) {
            OBJECT r;
            l_ungetc(c,f);
            buf[buf_pos] = 0;
            r = intern_token(buf);
            return r;
        }
        buf[buf_pos] = c;
    }
    error("Token larger than max-token-length",NULL);
    return NULL;
}

OBJECT readobject(OBJECT f) {
    char c;

    if (NULLP(read_object_buf)) {
        c = trim(f);
        if (c == EOF) return EOF_VAL;
        l_ungetc(c,f);
    }

    return do_read(f);
}

