#include "slang.h"
#include <time.h>

PRIVATE OBJECT freelist = EOL;
long cells_free = 0;

#define PTRLT(x,y)      ((long)(x) < (long)(y))
#define PTRGT(x,y)      PTRLT(y,x)
#define PTRGE(x,y)      (!PTRLT(x,y))
#define PTRLE(x,y)      (!PTRGT(x,y))

#define HEAP_SIZE   1000            /* @ 9 bytes */
#define HEAP_COUNT  4
PRIVATE OBJECT heap_org[HEAP_COUNT] = {EOL};
PRIVATE OBJECT heap_end[HEAP_COUNT] = {EOL};

void init_memman(void) {
    OBJECT h;
    if (NULLP(heap_org[0]) && NULLP(heap_end[0])) {
        long i,j;

        for (j=0; j<HEAP_COUNT; j++) {
            heap_org[j] = (OBJECT) malloc(HEAP_SIZE * sizeof(_object));
            if (!heap_org[j])
                fatal("Cannot allocate heap; decrease size + recompile");
            heap_end[j] = heap_org[j] + HEAP_SIZE;
            h = heap_org[j];
            for (i=0; i<HEAP_SIZE; i++,h++) {
                LTYPE(h) = T_FREELIST;
                GCMARK(h) = FALSE;
                CAR(h) = EOL;
                CDR(h) = freelist;
                freelist = h;
                cells_free++;
            }
        }
    }
}

long cells_used = 0;

OBJECT newobj(void) {
    OBJECT n;

    if (NULLP(freelist)) gc();
    if (NULLP(freelist)) fatal("Out of memory: cannot grow freelist");
    n = freelist;
    freelist = CDR(freelist);
    LTYPE(n) = T_NULL;
    CDR(n) = EOL;
    cells_used++;
    cells_free--;
    return n;
}

void *getmem(word size) {
    void *p = malloc(size);
    if (!p) {
        gc();               /* Attempt recovery by freeing unused strs+syms */
        p = malloc(size);
        if (!p) fatal("Out of memory in getmem");
    }
    return p;
}

PRIVATE void mark_array(OBJECT *start, OBJECT *end); /* Prototype */

PRIVATE void mark(OBJECT obj) {
    while (TRUE) {
        if (GCMARK(obj)) return;
        if (NULLP(obj)) return;
        GCMARK(obj) = TRUE;
        switch (TYPE(obj)) {
            case T_CONS:
                mark(CAR(obj));
                obj = CDR(obj);
                continue;
            case T_CLOSURE:
                mark(BODY(obj));
                obj = ENV(obj);
                continue;
            case T_VECTOR: {
                long i;
                for (i=0; i<VLEN(obj)-1; i++) mark(IDX(obj,i));
                obj = IDX(obj,VLEN(obj)-1);
                continue;
            }
            case T_CONTINUATION:
                mark_array(STACK(obj),
                           (OBJECT *) (((char *)STACK(obj))+
                                        CLEN(obj)*sizeof(OBJECT)));
                mark_array((OBJECT *) (((char *)STACK(obj))+1),
                           (OBJECT *) (((char *)STACK(obj))+
                                        CLEN(obj)*sizeof(OBJECT)+1));
                mark_array((OBJECT *) (((char *)STACK(obj))+2),
                           (OBJECT *) (((char *)STACK(obj))+
                                        CLEN(obj)*sizeof(OBJECT)+2));
                mark_array((OBJECT *) (((char *)STACK(obj))+3),
                           (OBJECT *) (((char *)STACK(obj))+
                                        CLEN(obj)*sizeof(OBJECT)+3));
                break;
        }
        return;
    }
}

PRIVATE void mark_array(OBJECT *start, OBJECT *end) {
    OBJECT tmp;
    long i,j,n;
    if (start > end) {
        OBJECT *t = start;
        start = end;
        end = t;
    }
    n = end - start + 1;
    for (i=0; i<n; i++) {
        tmp = start[i];
        for (j=0; j<HEAP_COUNT; j++) {
            if ((PTRGE(tmp,heap_org[j])) &&
                (PTRLT(tmp,heap_end[j])) &&
                (((((long)tmp) - ((long)heap_org[j])) % sizeof(_object)) == 0) &&
                (!TYPEP(tmp,T_FREELIST))) {
                    mark(tmp);
                    break;
                }
        }
    }
}

PRIVATE void killobj(OBJECT obj) {
    switch (TYPE(obj)) {
        case T_STRING:
        case T_SYM: free(SDATA(obj)); break;
        case T_VECTOR: free(DATA(obj)); break;
        case T_CONTINUATION:
            free((char *)JMPPTR(obj));  /* <-- Horrible assumptions here :-( */
            break;
        case T_FILE:
            fclose(FHANDLE(obj));
            free(FNAME(obj));
            break;
    }
}

PRIVATE void sweep(void) {
    long i,j;
    OBJECT h;

    for (j=0; j<HEAP_COUNT; j++) {
        h = heap_org[j];
        for (i=0; i<HEAP_SIZE; i++,h++) {
            if (!GCMARK(h) && !TYPEP(h,T_FREELIST)) {
                killobj(h);
                LTYPE(h) = T_FREELIST;
                CAR(h) = EOL;
                CDR(h) = freelist;
                freelist = h;
                cells_free++;
                gc_freed++;
            } else GCMARK(h) = FALSE;
        }
    }
}

OBJECT *stack_start = EOL;
long gc_ticks = 0;
long gc_freed = 0;

void gc(void) {
    OBJECT stack_end = NULL;
    long ticks = clock();
    long i;

    disable_alarm();
    mark(errobj);
    for (i=0; i<MAX_SPC_SYM; i++) mark(spc_sym[i]);
    mark(getglbtab());
    mark(getstk());
    mark_array(stack_start,&stack_end);
    mark_array((OBJECT *) (((char *)stack_start)+1),
               (OBJECT *) (((char *) &stack_end)+1));
    mark_array((OBJECT *) (((char *)stack_start)+2),
               (OBJECT *) (((char *) &stack_end)+2));
    mark_array((OBJECT *) (((char *)stack_start)+3),
               (OBJECT *) (((char *) &stack_end)+3));
    sweep();
    enable_alarm();
    gc_ticks += clock()-ticks;
}

PRIVATE int stacksize(OBJECT *start) {
    OBJECT s;
    return start - &s + 1;
}

extern void applyfun(void);

void call_cc(void) {
    long j;
    OBJECT proc = pop();
    OBJECT cont = newobj();
    OBJECT *src, *dst;

    LTYPE(cont) = T_CONTINUATION;
    CLEN(cont) = stacksize(stack_start);
    (char *)JMPPTR(cont) =
        getmem(CLEN(cont)*sizeof(OBJECT) + sizeof(jmp_buf));

    src = stack_start - CLEN(cont);
    dst = STACK(cont);
    for (j=CLEN(cont); --j >= 0; ) *dst++ = *src++;

    if (setjmp(JMP(cont))) return;
    push(copy_list(getstk()));
    push(cont);
    consfun();
    push(proc);
    applyfun();
}

PRIVATE void grow_stack(void) {
    char filler[100];
    set_cc();
}

void set_cc(void) {
    OBJECT cont = pop(), retval;
    long j;
    OBJECT *src, *dst;

    if (!CONSP(cont))
        error("set-cc: expects pair car=stack cdr=continuation; got",cont);
    dst = stack_start - CLEN(CDR(cont));
    if (PTRLE(dst-1,&dst)) {
        push(cont);
        grow_stack();
    }
    retval = pop();
    setstk(CAR(cont));
    cont = CDR(cont);
    if (!CONTP(cont))
        error("set-cc: expects continuation in cdr of pair; got",cont);
    push(retval);
    src = STACK(cont);
    for (j=CLEN(cont); --j >= 0; ) *dst++ = *src++;

    longjmp(JMP(cont),1);
}

