#include "slang.h"

#define RET(x)      { push(x); return; }

OBJECT spc_sym[MAX_SPC_SYM];

PRIVATE void addprim(char *name, OBJECT (*proc)(), int type) {
    addglobal(newstrobj(T_SYM,name),newprim(type,proc,name));
}

PRIVATE void writestr(char *s) {        /* Write a str */
    printf("\"");
    while (*s) {
        switch (*s) {                   /* Convert back to escaped chars */
            case 10: printf("\\n"); break;
            case 13: printf("\\r"); break;
            case 9: printf("\\t"); break;
            case '\\': printf("\\\\"); break;
            default: printf("%c",*s); break;
        }
        s++;
    }
    printf("\"");
}

PRIVATE void dispobj(OBJECT obj, bool write) {      /* Display an object */
    if (NULLP(obj)) {               /* The NULL object */
        printf("[]");
        return;
    }
    if (EQP(obj,UNDEFD)) {
        printf("#<undefined>");
        return;
    }
    if (EQP(obj,MARK)) return;
    if (EQP(obj,EOF_VAL)) {         /* Show the EOF object */
        printf("#<EOF>");
        return;
    }
    switch (TYPE(obj)) {            /* Normal object... */
        case T_INT: printf("%ld",IDATA(obj)); break;
        case T_FLOAT: printf("%.50g",FDATA(obj)); break;
        case T_STRING:
            if (write) writestr(SDATA(obj)); else
                       printf("%s",SDATA(obj));
            break;
        case T_SYM: printf("%s",SDATA(obj)); break;
        case T_CONS:
            printf("[");
            while (!NULLP(obj) && CONSP(obj)) {
                dispobj(CAR(obj),TRUE);         /* Show current */
                obj = CDR(obj);                 /* Get next */
                if (!NULLP(obj)) {
                    printf(" ");                /* Print spacing */
                    if (!CONSP(obj)) {
                        printf(". ");               /* Handle it */
                        dispobj(obj,TRUE);
                    }
                }
            }
            printf("]");
            break;
        case T_CLOSURE:
            if (EQP(ENV(obj),UNDEFD)) printf("#<CLOSURE-DYN "); else
                                      printf("#<CLOSURE-LEX ");
            dispobj(BODY(obj),TRUE); printf(">");
            break;

        case T_FPRIM:
        case T_MPRIM:
        case T_PRIM: printf("#<PRIMITIVE-PROCEDURE %s>",PNAME(obj)); break;

        case T_VECTOR: {
            long i;
            printf("#[");
            for (i=0; i<VLEN(obj); i++) {
                dispobj(IDX(obj,i),TRUE);
                if (i+1 < VLEN(obj)) printf(" ");
            }
            printf("]");
            break;
        }
        case T_CONTINUATION:
            printf("#<CONTINUATION %p>",obj);
            break;
        default: {
            char s[100];
            sprintf(s,"Unknown type %u in dispobj!",TYPE(obj));
            fatal(s);
        }
    }
}

OBJECT dispfun(void) {
    dispobj(pop(),FALSE);
}

OBJECT writefun(void) {           /* Write external rep. of an obj */
    dispobj(pop(),TRUE);
}

OBJECT loadfun(void) {
    OBJECT name = pop();
    OBJECT file,r;

    if (!STRINGP(name)) error("load: expects string; got",name);
    file = l_fopen(SDATA(name),"rb");
    if (EQP(file,S_FALSE)) {
        push(S_FALSE);
        return;
    }
    r = readobject(file);
    while (!EQP(r,EOF_VAL)) {
        eval(r,EOL);
        r = readobject(file);
    }
    push(S_TRUE);
}

OBJECT consfun(void) {
    OBJECT cdr = pop();
    push(cons(pop(),cdr));
}

OBJECT applyfun(OBJECT env) {
    OBJECT proc = pop();
    if (!PROCP(proc)) error("apply: expects procedure; got",proc);
    switch (TYPE(proc)) {
        case T_CLOSURE:
            if (!EQP(ENV(proc),UNDEFD))
                env = addframe(ENV(proc)); else
                if (!NULLP(CAR(CAR(env)))) env = addframe(env);
            proc = BODY(proc);
            while (!NULLP(proc)) {
                eval(CAR(proc),env);
                proc = CDR(proc);
            }
            break;
        case T_PRIM:
            PROC(proc)(); break;
        case T_FPRIM:
            PROC(proc)(env); break;
        case T_MPRIM:
            if (!NULLP(PROC(proc)(&proc,&env))) {
                eval(proc,env);
                return;
            }
        default:
            error("apply: cannot apply this:",proc);
    }
}

OBJECT listtovectfun(void) {
    OBJECT lst = pop();
    OBJECT v;
    long i,l;
    if (!LISTP(lst)) error("list->vector: expects list; got",lst);
    l = list_length(lst);
    v = newvector(l);
    for (i=0; i<l; i++, lst=CDR(lst)) IDX(v,i) = CAR(lst);
    push(v);
}

OBJECT makestaticfun(OBJECT env) {
    OBJECT lst = pop();
    if (!LISTP(lst)) error("make-static: expects list; got",lst);
    push(newclosure(lst,env));
}

OBJECT makerecfun(OBJECT env) {
    OBJECT lst = pop();
    OBJECT sym = pop();
    OBJECT prc;
    if (!LISTP(lst))
        error("make-recursive: expects list as second arg; got",lst);
    if (!SYMP(sym))
        error("make-recursive: expects symbol as first arg; got",sym);
    env = addframe(env);
    prc = newclosure(lst,env);
    defsym(env,sym,prc);
    push(sym);
    push(prc);
}

OBJECT deffun(OBJECT env) {
    OBJECT val = pop();
    OBJECT sym = pop();
    if (!SYMP(sym)) error("define: expects symbol; got",sym);
    if (NULLP(env)) addglobal(sym,val); else
                    defsym(env,sym,val);
}

OBJECT swapfun(void) {
    OBJECT x = pop();
    OBJECT y = pop();
    push(x); push(y);
}

OBJECT dupfun(void) {
    OBJECT x = pop();
    push(x); push(x);
}

OBJECT dropfun(void) {
    pop();
}

OBJECT lookupfun(OBJECT env) {
    OBJECT sym = pop();
    bool found;
    OBJECT r = refsym(sym,env,&found);
    if (!found) error("lookup: symbol not defined:",sym);
    push(r);
}

OBJECT gcfun(void) {
    gc();
}

PRIVATE OBJECT rectify(OBJECT n) {
    if (FLOATP(n) && ((FLOAT)(long)FDATA(n) == FDATA(n))) {
        LTYPE(n) = T_INT;
        IDATA(n) = FDATA(n);
    }
    return n;
}

OBJECT plusfun(void) {
    OBJECT b = pop();
    OBJECT a = pop();
    if (!NUMP(a)) error("+: expects number as first arg; got",a);
    if (!NUMP(b)) error("+: expects number as second arg; got",b);
    push(rectify(newfloat(NDATA(a)+NDATA(b))));
}

OBJECT subfun(void) {
    OBJECT b = pop();
    OBJECT a = pop();
    if (!NUMP(a)) error("-: expects number as first arg; got",a);
    if (!NUMP(b)) error("-: expects number as second arg; got",b);
    push(rectify(newfloat(NDATA(a)-NDATA(b))));
}

OBJECT mulfun(void) {
    OBJECT b = pop();
    OBJECT a = pop();
    if (!NUMP(a)) error("*: expects number as first arg; got",a);
    if (!NUMP(b)) error("*: expects number as second arg; got",b);
    push(rectify(newfloat(NDATA(a)*NDATA(b))));
}

OBJECT divfun(void) {
    OBJECT b = pop();
    OBJECT a = pop();
    if (!NUMP(a)) error("/: expects number as first arg; got",a);
    if (!NUMP(b)) error("/: expects number as second arg; got",b);
    push(rectify(newfloat((FLOAT)NDATA(a)/(FLOAT)NDATA(b))));
}

OBJECT gtfun(void) {
    OBJECT y = pop();
    OBJECT x = pop();
    if (!NUMP(x)) error(">: expects number as first arg; got",x);
    if (!NUMP(y)) error(">: expects number as second arg; got",y);
    push(S_BOOL(NDATA(x)>NDATA(y)));
}

OBJECT modfun(void) {
    OBJECT b = pop();
    OBJECT a = pop();
    if (!INTP(a)) error("mod: expects integer as first arg; got",a);
    if (!INTP(b)) error("mod: expects integer as second arg; got",b);
    push(newint(IDATA(a)%IDATA(b)));
}

OBJECT iffun(OBJECT *formptr, OBJECT *envptr) {
    OBJECT tp = pop();
    OBJECT val = pop();
    if (!EQP(val,S_FALSE)) return m_apply(tp,formptr,envptr);
    return NULL;
}

OBJECT ifelsefun(OBJECT *formptr, OBJECT *envptr) {
    OBJECT fp = pop();
    OBJECT tp = pop();
    OBJECT val = pop();
    if (!EQP(val,S_FALSE)) return m_apply(tp,formptr,envptr);
    return m_apply(fp,formptr,envptr);
}

OBJECT eqpfun(void) {
    OBJECT a = pop(), b = pop();
    if (EQP(a,b)) RET(S_TRUE);
    if (TYPE(a) != TYPE(b)) RET(S_FALSE);
    if (SYMP(a)) RET(S_BOOL(streq_oo(a,b)));
    RET(S_FALSE);
}

OBJECT eqvpfun(void) {
    OBJECT a = pop(), b = pop();
    if (EQP(a,b)) RET(S_TRUE);
    if (TYPE(a) != TYPE(b)) RET(S_FALSE);
    switch (TYPE(a)) {
        case T_INT: RET(S_BOOL(IDATA(a)==IDATA(b)));
        case T_FLOAT: RET(S_BOOL(FDATA(a)==FDATA(b)));
        case T_SYM: RET(S_BOOL(streq_oo(a,b)));
        case T_PRIM: case T_FPRIM: case T_MPRIM:
            RET(S_BOOL(PROC(a)==PROC(b)));
    }
    RET(S_FALSE);
}

OBJECT equalpfun(void) {
    OBJECT a = pop(), b = pop();
    if (EQP(a,b)) RET(S_TRUE);
    if (NUMP(a)&&NUMP(b)) RET(S_BOOL(NDATA(a)==NDATA(b)));
    if (TYPE(a) != TYPE(b)) RET(S_FALSE);
    switch (TYPE(a)) {
        case T_STRING:
        case T_SYM: RET(S_BOOL(streq_oo(a,b)));
        case T_CONS:
            while (!NULLP(a) && !NULLP(b)) {
                push(CAR(a)); push(CAR(b)); equalpfun();
                if (EQP(pop(),S_FALSE)) RET(S_FALSE);
                a = CDR(a);
                b = CDR(b);
                if (!CONSP(a)) {
                    if (!CONSP(b)) {
                        push(a); push(b); equalpfun();
                        return;
                    }
                }
                if (!CONSP(b)) RET(S_FALSE);
            }
            if (!NULLP(a) || !NULLP(b)) RET(S_FALSE);
            RET(S_TRUE);
        case T_VECTOR: {
            long i;

            if (VLEN(a) != VLEN(b)) RET(S_FALSE);
            for (i=VLEN(a); i>0; i--) {
                push(IDX(a,i-1));
                push(IDX(b,i-1));
                equalpfun();
                if (EQP(pop(),S_FALSE)) RET(S_FALSE);
            }
            RET(S_TRUE);
        }
        case T_PRIM: case T_FPRIM: case T_MPRIM:
            RET(S_BOOL(PROC(a)==PROC(b)));
    }
}

OBJECT memqfun(void) {
    OBJECT lst = pop();
    OBJECT x = pop();
    while (CONSP(lst)) {
        push(CAR(lst));
        push(x);
        eqpfun();
        if (EQP(pop(),S_TRUE)) RET(lst);
        lst = CDR(lst);
    }
    if (!NULLP(lst)) error("memq: improper list",NULL);
    RET(S_FALSE);
}

OBJECT memvfun(void) {
    OBJECT lst = pop();
    OBJECT x = pop();
    while (CONSP(lst)) {
        push(CAR(lst));
        push(x);
        eqvpfun();
        if (EQP(pop(),S_TRUE)) RET(lst);
        lst = CDR(lst);
    }
    if (!NULLP(lst)) error("memv: improper list",NULL);
    RET(S_FALSE);
}

OBJECT memberfun(void) {
    OBJECT lst = pop();
    OBJECT x = pop();
    while (CONSP(lst)) {
        push(CAR(lst));
        push(x);
        equalpfun();
        if (EQP(pop(),S_TRUE)) RET(lst);
        lst = CDR(lst);
    }
    if (!NULLP(lst)) error("member: improper list",NULL);
    RET(S_FALSE);
}

OBJECT assqfun(void) {
    OBJECT lst = pop();
    OBJECT x = pop();
    while (CONSP(lst)) {
        if (!CONSP(CAR(lst)))
            error("assq: alist contains non-pair",CAR(lst));
        push(CAR(CAR(lst)));
        push(x);
        eqpfun();
        if (EQP(pop(),S_TRUE)) RET(CAR(lst));
        lst = CDR(lst);
    }
    if (!NULLP(lst)) error("assq: improper list",NULL);
    RET(S_FALSE);
}

OBJECT assvfun(void) {
    OBJECT lst = pop();
    OBJECT x = pop();
    while (CONSP(lst)) {
        if (!CONSP(CAR(lst)))
            error("assv: alist contains non-pair",CAR(lst));
        push(CAR(CAR(lst)));
        push(x);
        eqvpfun();
        if (EQP(pop(),S_TRUE)) RET(CAR(lst));
        lst = CDR(lst);
    }
    if (!NULLP(lst)) error("assv: improper list",NULL);
    RET(S_FALSE);
}

OBJECT assocfun(void) {
    OBJECT lst = pop();
    OBJECT x = pop();
    while (CONSP(lst)) {
        if (!CONSP(CAR(lst)))
            error("assoc: alist contains non-pair",CAR(lst));
        push(CAR(CAR(lst)));
        push(x);
        equalpfun();
        if (EQP(pop(),S_TRUE)) RET(CAR(lst));
        lst = CDR(lst);
    }
    if (!NULLP(lst)) error("assoc: improper list",NULL);
    RET(S_FALSE);
}

OBJECT carfun(void) {
    OBJECT x = pop();
    if (!CONSP(x)) error("car: expects pair; got",x);
    push(CAR(x));
}

OBJECT cdrfun(void) {
    OBJECT x = pop();
    if (!CONSP(x)) error("cdr: expects pair; got",x);
    push(CDR(x));
}

OBJECT makelistfun(void) {
    OBJECT x = pop();
    OBJECT r = EOL;
    while (!EQP(x,MARK)) {
        r = cons(x,r);
        x = pop();
    }
    RET(r);
}

OBJECT makevectfun(void) {
    makelistfun();
    listtovectfun();
}

OBJECT setcarfun(void) {
    OBJECT x = pop();
    if (!CONSP(x)) error("set-car: expects pair; got",x);
    CAR(x) = pop();
}

OBJECT setcdrfun(void) {
    OBJECT x = pop();
    if (!CONSP(x)) error("set-cdr: expects pair; got",x);
    CDR(x) = pop();
}

OBJECT setfun(OBJECT env) {
    OBJECT val = pop();
    OBJECT sym = pop();
    if (!SYMP(sym)) error("set: expects sym as second arg; got",sym);
    setsym(sym,val,env);
}

OBJECT callcc(void) {
    call_cc();
}

OBJECT setcc(void) {
    set_cc();
}

PRIVATE void alarm_handler() {
    eval(newstrobj(T_SYM,"alarm-procedure"),EOL);
}

OBJECT alarmfun(void) {
    OBJECT proc = pop();
    OBJECT time = pop();

    if (!PROCP(proc)) error("alarm: expects procedure as arg2; got",proc);
    if (!NUMP(time)) error("alarm: expects number as arg1; got",time);
    addglobal(newstrobj(T_SYM,"alarm-procedure"),proc);
    set_alarm(NDATA(time),alarm_handler);
}

void init_prim(void) {
    UNDEFD = newobj();
    S_TRUE = newstrobj(T_SYM,"#t");
    S_FALSE = newstrobj(T_SYM,"#f");
    addglobal(S_TRUE,S_TRUE);
    addglobal(S_FALSE,S_FALSE);
    EOF_VAL = newobj();
    MARK = newstrobj(T_SYM,"mark");
    addglobal(MARK,MARK);

    F_STDIN = newobj();
    LTYPE(F_STDIN) = T_FILE;
    FNAME(F_STDIN) = "stdin";
    FHANDLE(F_STDIN) = stdin;

    addprim("disp",dispfun,T_PRIM);
    addprim("write",writefun,T_PRIM);
    addprim("load",loadfun,T_PRIM);
    addprim("cons",consfun,T_PRIM);
    addprim("apply",applyfun,T_FPRIM);
    addprim("list->vector",listtovectfun,T_PRIM);
    addprim("make-static",makestaticfun,T_FPRIM);
    addprim("make-recursive",makerecfun,T_FPRIM);
    addprim("define",deffun,T_FPRIM);
    addprim("swap",swapfun,T_PRIM);
    addprim("dup",dupfun,T_PRIM);
    addprim("drop",dropfun,T_PRIM);
    addprim("lookup",lookupfun,T_FPRIM);
    addprim("gc",gcfun,T_PRIM);
    addprim("+",plusfun,T_PRIM);
    addprim("-",subfun,T_PRIM);
    addprim("*",mulfun,T_PRIM);
    addprim("/",divfun,T_PRIM);
    addprim("mod",modfun,T_PRIM);
    addprim("if",iffun,T_MPRIM);
    addprim("ifelse",ifelsefun,T_MPRIM);
    addprim("eq?",eqpfun,T_PRIM);
    addprim("eqv?",eqvpfun,T_PRIM);
    addprim("equal?",equalpfun,T_PRIM);
    addprim("memq",memqfun,T_PRIM);
    addprim("memv",memvfun,T_PRIM);
    addprim("member",memberfun,T_PRIM);
    addprim("assq",assqfun,T_PRIM);
    addprim("assv",assvfun,T_PRIM);
    addprim("assoc",assocfun,T_PRIM);
    addprim("car",carfun,T_PRIM);
    addprim("cdr",cdrfun,T_PRIM);
    addprim("make-list",makelistfun,T_PRIM);
    addprim("make-vector",makevectfun,T_PRIM);
    addprim("set-car",setcarfun,T_PRIM);
    addprim("set-cdr",setcdrfun,T_PRIM);
    addprim("set",setfun,T_FPRIM);
    addprim("call/cc",callcc,T_PRIM);
    addprim("set-cc",setcc,T_PRIM);
    addprim("alarm",alarmfun,T_PRIM);
    addprim(">",gtfun,T_PRIM);
}
