;;@title "G+, a higher-level GTK+ interface"
;;
;; <synopsis>(require 'g+)</synopsis>
;; G+ is based on the ideas in JLib, a library for building GUI widget
;; trees which comes with Jscheme.

(require 'gtk)

;;@section "Core macros and functions"

;;@macro (g+predicate-case (varname ...) ((predicate ...) body ...) ...)
;; Expands into a <function>cond</function> expression which tests
;; each <parameter>varname</parameter> against the corresponding
;; <parameter>predicate</parameter>, executing the
;; <parameter>body</parameter> of the first clause for which all the
;; <parameter>predicate</parameter>s return true. (A clause may also
;; have the keyword <literal>else</literal> instead of a list of
;; predicates, with effect similar to <function>cond</function> and
;; <function>case</function>.)
(define-macro (g+predicate-case varlist . clauses)
  `(cond ,@(map (lambda (clause)
		  (if (eq? (car clause) 'else)
		      clause
		      `((and ,@(map list (car clause) varlist))
			,@(cdr clause))))
		clauses)))

;;@macro (g+define-ctor name (base-ctor args ...))
;; Expands into a definition of <parameter>name</parameter>, a
;; function which accepts <parameter>args ...</parameter> and passes
;; them to <parameter>base-ctor</parameter>, keeping the result,
;; passing the result to <function>g+:configure</function> with any
;; extra arguments supplied, and then returning the result of the call
;; to <parameter>base-ctor</parameter>.
;;
;; For example:
;; <programlisting>
;; (g+define-ctor X (A B C D))
;; </programlisting>
;; expands into:
;; <programlisting>
;; (define (X B C D . g+args)
;;   (let ((x (A B C D)))
;;     (g+:configure x g+args)
;;     x))
;; </programlisting>
(define-macro (g+define-ctor name realctor-pattern)
  `(define (,name ,@(cdr realctor-pattern) . g+args)
     (let ((x ,(if (list? realctor-pattern)
		   realctor-pattern
		   `(apply ,@(drop-right realctor-pattern 0)
			   ,(cdr (last-pair realctor-pattern))))))
       (g+:configure x g+args)
       x)))

;;@section "Constructors and modifiers"

;;@
;; Given an object <parameter>x</parameter>, and a list of
;; <parameter>items</parameter>, takes different actions depending on
;; the types of <parameter>x</parameter> and each
;; <parameter>item</parameter> in turn. In general, if
;; <parameter>x</parameter> is some kind of container, and an
;; <parameter>item</parameter> is some kind of widget or object
;; appropriate for containment within that container, it will be
;; placed inside it. If an <parameter>item</parameter> is a string,
;; and <parameter>x</parameter> has some kind of intuitively-default
;; text-string property on it, the property will be set. If an
;; <parameter>item</parameter> is a procedure, the procedure will be
;; called with <parameter>x</parameter> as its single argument.
;;
;; This function is the core of the G+ library, and is the main idea
;; taken from JLib: the heavy use of lambdas makes for a fairly clean
;; way of building an extensible optional-argument and -property
;; system.
(define g+:configure
  (letrec ((anything? (lambda _ #t))
	   (eachitem (lambda (x item)
		       (g+predicate-case (x item)

			 ((GtkMenuShell? GtkMenuItem?)
			  (gtk-menu-shell-append x item))
			 ((GtkMenuItem? GtkMenuShell?)
			  (gtk-menu-item-set-submenu x item))
			 ((GtkOptionMenu? GtkMenuShell?)
			  (gtk-option-menu-set-menu x item))
			 ((GtkTreeView? GtkTreeViewColumn?)
			  (gtk-tree-view-append-column x item))

			 ((GtkScrolledWindow? GtkWidget?)
			  (if (and (gobject? item)
				   (not (zero? (gsignal-lookup
						"set-scroll-adjustments"
						(gobject-type item)))))
			      (gtk-container-add x item)
			      (gtk-scrolled-window-add-with-viewport x item)))
			 ((GtkContainer? GtkWidget?)
			  (gtk-container-add x item))

			 ((anything? procedure?)
			  (item x))

			 ((GtkWindow? string?)
			  (gtk-window-set-title x item))
			 ((anything? string?)
			  (g+writestring x item))

			 ((anything? list?)
			  (g+:configure x item))

			 ((anything? boolean?)
			  'no-op)

			 (else
			  (error "g+:configure: unknown x/item pair"
				 x
				 item))))))
    (lambda (x items)
      (for-each (lambda (item) (eachitem x item))
		items)
      x)))

;;@ Returns a function that when applied to a GObject, sets a property
;;on its argument. For use with <function>g+:configure</function> (and
;;by extension constructors defined with
;;<function>g+define-ctor</function>).
(define (g+property name value)
  (lambda (x)
    (gobject-set-property! x name value)))

;;@ Returns a function that when applied to a GObject, installs a
;;signal-handler on it using <function>gsignal-connect</function>.
(define (g+signal name handler)
  (lambda (x)
    (gsignal-connect x name handler)))

;;@ Returns a function that when applied to a GtkBox, packs all the
;;<parameter>widgets</parameter> into it using
;;<function>gtk-box-pack-start</function>.
(define (g+pack-start expand fill padding . widgets)
  (lambda (box)
    (for-each (cute gtk-box-pack-start box <> expand fill padding) widgets)))

;;@ Returns a function that when applied to a GtkBox, packs all the
;;<parameter>widgets</parameter> into it using
;;<function>gtk-box-pack-end</function>.
(define (g+pack-end expand fill padding . widgets)
  (lambda (box)
    (for-each (cute gtk-box-pack-end box <> expand fill padding) widgets)))

;;@ Returns a function that when applied to a GtkWidget, sets the
;;tooltip on that widget in the <parameter>tooltips</parameter> set to
;;be <parameter>text</parameter>.
(define (g+tip tooltips text)
  (lambda (x)
    (gtk-tooltips-set-tip tooltips x text #f)))

;;@ Returns a function that when applied to a GtkLabel, sets its
;;markup and mnemonic keysequence according to
;;<parameter>markup-mnemonic</parameter>.
(define (g+label-markup markup-mnemonic)
  (lambda (x)
    (gtk-label-set-markup-with-mnemonic x markup-mnemonic)))

;;@ Returns a function that when applied to a GtkLabel, sets its
;;markup according to <parameter>markup</parameter>.
(define (g+label-markup* markup)
  (lambda (x)
    (gtk-label-set-markup x markup)))

;;@function (g+button mnemonic ...)
(g+define-ctor g+button (gtk-button-new-with-mnemonic mnemonic))
;;@function (g+button* ...)
(g+define-ctor g+button* (gtk-button-new))
;;@function (g+stock-button stock-id ...)
;;
;;These three constructors use <function>g+:configure</function> to
;;build variants on GtkButton.
(g+define-ctor g+stock-button (gtk-button-new-from-stock stock-id))

;;@function (g+label mnemonic ...)
(g+define-ctor g+label (gtk-label-new-with-mnemonic mnemonic))
;;@function (g+label text ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkLabel.
(g+define-ctor g+label* (gtk-label-new text))

;;@function (g+entry ...)
(g+define-ctor g+entry (gtk-entry-new))
;;@function (g+entry/max-length max-length ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkEntry.
(g+define-ctor g+entry/max-length (gtk-entry-new-with-max-length max-length))

;;@function (g+window type ...)
;;
;; Builds a GtkWindow using <function>gtk-window-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+window (gtk-window-new type))

;;@function (g+dialog ...)
;;
;; Builds a GtkDialog using <function>gtk-dialog-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+dialog (gtk-dialog-new))

;;@function (g+vbox homogeneous spacing ...)
(g+define-ctor g+vbox (gtk-vbox-new homogeneous spacing))
;;@function (g+hbox homogeneous spacing ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkBox.
(g+define-ctor g+hbox (gtk-hbox-new homogeneous spacing))

;;@function (g+vbutton-box ...)
(g+define-ctor g+vbutton-box (gtk-vbutton-box-new))
;;@function (g+hbutton-box ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkButtonBox.
(g+define-ctor g+hbutton-box (gtk-hbutton-box-new))

;;@function (g+vpaned ...)
(g+define-ctor g+vpaned (gtk-vpaned-new))
;;@function (g+hpaned ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkPaned.
(g+define-ctor g+hpaned (gtk-hpaned-new))

;;@function (g+menu ...)
(g+define-ctor g+menu (gtk-menu-new))
;;@function (g+menu-bar ...)
(g+define-ctor g+menu-bar (gtk-menu-bar-new))
;;@function (g+menu-item mnemonic ...)
(g+define-ctor g+menu-item (gtk-menu-item-new-with-mnemonic mnemonic))
;;@function (g+menu-item* ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkMenu and GtkOptionMenu.
(g+define-ctor g+menu-item* (gtk-menu-item-new))

;;@function (g+option-menu ...)
;;
;; Builds a GtkOptionMenu using
;; <function>gtk-option-menu-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+option-menu (gtk-option-menu-new))

;;@function (g+tooltips ...)
;;
;; Builds a GtkTooltips object using
;; <function>gtk-tooltips-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+tooltips (gtk-tooltips-new))

;;@function (g+toolbar ...)
;;
;; Builds a GtkToolbar object using
;; <function>gtk-toolbar-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+toolbar (gtk-toolbar-new))

;;@function (g+calendar ...)
;;
;; Builds a GtkCalendar object using
;; <function>gtk-calendar-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+calendar (gtk-calendar-new))

;;@function (g+check-button mnemonic ...)
(g+define-ctor g+check-button (gtk-check-button-new-with-mnemonic mnemonic))
;;@function (g+check-button* ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkCheckButton.
(g+define-ctor g+check-button* (gtk-check-button-new))

;;@function (g+radio-button group-or-null-gobject mnemonic ...)
(g+define-ctor g+radio-button (gtk-radio-button-new-with-mnemonic-from-widget
			       group-or-null-gobject
			       mnemonic))
;;@function (g+radio-button* group-or-null-gobject ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkRadioButton.
(g+define-ctor g+radio-button* (gtk-radio-button-new-from-widget group-or-null-gobject))

;;@function (g+adjustment current min max stepincr pageincr pagesize ...)
(g+define-ctor g+adjustment (gtk-adjustment-new current min max stepincr pageincr pagesize))
;;@function (g+hscrollbar adjustment ...)
(g+define-ctor g+hscrollbar (gtk-hscrollbar-new adjustment))
;;@function (g+vscrollbar adjustment ...)
(g+define-ctor g+vscrollbar (gtk-vscrollbar-new adjustment))
;;@function (g+hscale adjustment ...)
(g+define-ctor g+hscale (gtk-hscale-new adjustment))
;;@function (g+vscale adjustment ...)
(g+define-ctor g+vscale (gtk-vscale-new adjustment))
;;@function (g+spin-button adjustment climbrate numdigits ...)
(g+define-ctor g+spin-button (gtk-spin-button-new adjustment climbrate numdigits))
;;@function (g+spin-button/range min max step ...)
;;
;;These constructors use <function>g+:configure</function> to build
;;variants on GtkAdjustment, GtkScrollbar, GtkScale and GtkSpinButton.
(g+define-ctor g+spin-button/range (gtk-spin-button-new-with-range min max step))

;;@function (g+arrow arrow-type shadow-type ...)
;;
;; Builds a GtkArrow object using <function>gtk-arrow-new</function>
;; and <function>g+:configure</function>.
(g+define-ctor g+arrow (gtk-arrow-new arrow-type shadow-type))

;;@function (g+scrolled-window hscrollbar vscrollbar ...)
;;
;; Builds a GtkScrolledWindow object using
;; <function>gtk-scrolled-window-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+scrolled-window (gtk-scrolled-window-new hscrollbar vscrollbar))

;;@function (g+table rows columns homogeneous)
;;
;; Builds a GtkTable using <function>gtk-table-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+table (gtk-table-new rows columns homogeneous))

;;@ Uses <function>gtk-table-attach-defaults</function> to place a
;;widget within a GtkTable.
(define (g+table-cell left right top bottom widget)
  (lambda (table)
    (gtk-table-attach-defaults table widget left right top bottom)))

;;@ Uses <function>gtk-table-attach</function> to place a widget
;;within a GtkTable.
(define (g+table-cell* left right top bottom xoptions yoptions xpadding ypadding widget)
  (lambda (table)
    (gtk-table-attach table widget left right top bottom xoptions yoptions xpadding ypadding)))

;;@function (g+notebook ...)
;;
;; Builds a GtkNotebook object using
;; <function>gtk-notebook-new</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+notebook (gtk-notebook-new))

;;@ Returns a function that when applied to a GtkNotebook, appends a
;;page to it using <function>gtk-notebook-append-page</function>.
(define (g+notebook-page label-widget page-widget)
  (lambda (nb)
    (gtk-notebook-append-page nb page-widget label-widget)))

;;@ Returns a function that when applied to a GtkNotebook, appends a
;;page to it using <function>gtk-notebook-append-page-menu</function>.
(define (g+notebook-page* label-widget menu-widget page-widget)
  (lambda (nb)
    (gtk-notebook-append-page-menu page-widget label-widget menu-widget)))

;;@function (g+list-store typelist rows ...)
;;
;; Builds a GtkListStore object using
;; <function>g+:make-list-store</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+list-store (g+:make-list-store typelist rows))

;;@ Creates a new GtkListStore, and creates <literal>(length
;;<parameter>typelist</parameter>)</literal> columns. Each element of
;;<parameter>typelist</parameter> should be a GType record. The
;;<parameter>rows</parameter> should contain zero or more lists of
;;entries to put in the list store. Each row must contain items that
;;correspond to the GTypes passed in <parameter>typelist</parameter>.
(define (g+:make-list-store typelist rows)
  (let ((x (apply gtk-list-store-new typelist)))
    (g+:list-store-append! x typelist rows)
    x))

;;@ Appends <parameter>rows</parameter> to <parameter>ls</parameter>,
;;using the list of GType records in <parameter>typelist</parameter>
;;to build the intermediate GValues.
(define (g+:list-store-append! ls typelist rows)
  (if (not (null? rows))	; cut down on gratuitous garbage
      (let ((iter (gtk-tree-iter-new))
	    (indices (iota (length typelist))))
	(for-each (lambda (row)
		    (gtk-list-store-append ls iter)
		    (for-each (lambda (index item type)
				(gtk-list-store-set-value ls iter index
							  (object->gvalue type item)))
			      indices
			      row
			      typelist))
		  rows))))

;;@function (g+tree-store typelist rows ...)
;;
;; Builds a GtkTreeStore object using
;; <function>g+:make-tree-store</function> and
;; <function>g+:configure</function>.
(g+define-ctor g+tree-store (g+:make-tree-store typelist rows))

;;@ Creates a new GtkTreeStore, and creates <literal>(length
;;<parameter>typelist</parameter>)</literal> columns. Each element of
;;<parameter>typelist</parameter> should be a GType record. The
;;<parameter>rows</parameter> should contain zero or more lists of
;;entries to put in at the root of the tree. Each row must contain
;;items that correspond to the GTypes passed in
;;<parameter>typelist</parameter>, followed by child rows (that follow
;;the same definition).
;;
;;For example:
;;<programlisting>
;;(g+:make-tree-store (list gtype:string gtype:int)
;;                    '(("A" 100
;;                           ("AA" 110
;;                                 ("AAA" 111))
;;                           ("AB" 120))
;;                      ("B" 200
;;                           ("BA" 210)
;;                           ("BB" 220))))
;;</programlisting>
(define (g+:make-tree-store typelist rows)
  (let* ((x (apply gtk-tree-store-new typelist)))
    (g+:tree-store-append! x typelist (null-gboxed) rows)
    x))

;;@ Appends <parameter>rows</parameter> to <parameter>ts</parameter>,
;;under the parent element at <parameter>parent-iter</parameter> (pass
;;in <literal>(null-gboxed)</literal> to refer to the root element),
;;using the list of GType records in <parameter>typelist</parameter>
;;to build the intermediate GValues.
(define (g+:tree-store-append! ts typelist parent-iter rows)
  (let* ((column-count (length typelist))
	 (indices (iota column-count)))
    (define (append-rows parent-iter rows)
      (if (not (null? rows))
	  (let ((iter (gtk-tree-iter-new)))
	    (for-each (lambda (row)
			(gtk-tree-store-append ts iter parent-iter)
			(for-each (lambda (index item type)
				    (gtk-tree-store-set-value ts iter index
							      (object->gvalue type item)))
				  indices
				  row
				  typelist)
			(append-rows iter (drop row column-count)))
		      rows))))
    (append-rows parent-iter rows)))

;;@function (g+tree-view tree-model ...)
;; Wraps <function>gtk-tree-view-new-with-model</function> with a
;; <function>g+:configure</function> step.
(g+define-ctor g+tree-view (gtk-tree-view-new-with-model model)) ; a list/tree-store is a model

;;@ Creates and returns a configured instance of
;; GtkTreeViewColumn.
;;
;; <parameter>title</parameter> should be the text used as the column
;; heading. <parameter>renderer</parameter> should be either one of
;; the symbols <literal>(text toggle pixbuf)</literal>, or an instance
;; of GtkCellRenderer. <parameter>column-id</parameter> should be the
;; column from the GtkTreeModel to fetch data to render from. (To
;; render the data in the first column on the GtkTreeModel, pass in 0;
;; the third column, pass 2; etc.)
;;
;; <parameter>updater</parameter> may supply a function which will be
;; called when the content of the cell renderer is edited by the
;; user. Set it to <literal>#f</literal> if you don't want to install
;; a handler for edited cells. <parameter>editable-column</parameter>
;; may supply a GtkTreeModel column number which contains GBoolean
;; information specifying whether the cell rendered by this column at
;; a particular row should be user-editable or not. Supply
;; <literal>#f</literal> if you want the cell to be left in its
;; default state with regard to editability.
;;
;; Both <parameter>updater</parameter> and
;; <parameter>editable-column</parameter> are only relevant if
;; <parameter>renderer</parameter> is a symbol - if it's a
;; GtkCellRenderer instance, this function has no way of working out
;; how to set updater or edit-column properties, so it leaves it up to
;; its caller.
(define (g+tree-view-column title renderer column-id updater editable-column . g+args)
  (let ((x (gtk-tree-view-column-new))
	(r (case renderer
	     ((text)	(gtk-cell-renderer-text-new))
	     ((toggle)	(gtk-cell-renderer-toggle-new))
	     ((pixbuf)	(gtk-cell-renderer-pixbuf-new))
	     (else	renderer))))
    (if updater
	(case renderer
	  ((text)	(gtk-signal-connect r 'edited updater))
	  ((toggle)	(gtk-signal-connect r 'toggled updater))
	  (else		(error "Cannot use #:updater with this renderer"
			       renderer))))
    (gtk-tree-view-column-set-title x title)
    (gtk-tree-view-column-pack-start x r #t)
    (if (symbol? renderer)
	(gtk-tree-view-column-add-attribute x r
					    (case renderer
					      ((text)	"text")
					      ((toggle)	"active")
					      ((pixbuf)	"pixbuf")
					      (else (error "Unsupported renderer symbol"
							   renderer)))
					    column-id))
    (if editable-column
	(gtk-tree-view-column-add-attribute x r
					    (case renderer
					      ((text)	"editable")
					      ((toggle)	"activatable")
					      (else (error "Unsupported renderer symbol"
							   renderer)))
					    editable-column))
    (g+:configure x g+args)
    x))
