;;@title "Glib/GObject 2.0 binding"
(declare
 (usual-integrations)
 (foreign-declare #<<EOF

#include <glib-object.h>

#include "gobject-c.c"

static guint* CG_signal_ids;

EOF
))

;;@ <synopsis>(require 'gobject)</synopsis>
;; The gobject extension module provides a wrapping for a subset of
;; the features offered by GLib version 2.0. Currently it exposes a
;; partial API for manipulating GType, GBoxed, GEnum, GFlags, GValue,
;; GClosure, GObject and GSignal types and values.

(require 'srfi-1)
(require 'srfi-13)
(require 'lolevel)

(require-for-syntax 'srfi-13)

;---------------------------------------------------------------------------;
;;@section "Initialization and glib miscellany"

((foreign-lambda* void ()
		  "g_type_init();"))

(define-macro (gtype:init-types-from-file filename)
  `((foreign-lambda* void ()
		     ,@(map (lambda (name)
			      (string-append name "();\n"))
			    (with-input-from-file
				filename
			      read-lines)))))

(gtype:init-types-from-file "glib-types")

;;@ Delegates to the C function <function>g_warning</function> to
;;produce a warning message using the GLib logging facility.
(define (g-warning . args)
  ((foreign-lambda void "g_warning" c-string c-string)
   "%s"
   (with-output-to-string
     (lambda ()
       (for-each display args)))))

;---------------------------------------------------------------------------;
;;@section "GType"
;;
;; GType is the GLib Runtime type identification and management
;; system. Most of the datatypes used in GLib (and GDK/GTK+ etc) are
;; registered with the GType system.
;;
;; A certain amount of introspection over the GType system is
;; possible. GType itself does not provide information about methods
;; on objects, but does allow enumeration of object properties,
;; signals, superclasses and subclasses, and also provides information
;; on the allowable values of enumerations (GEnum) and flags (GFlags).
;;
;; The combination of the following procedures and variables ought to
;; allow access to much of the available metainformation:
;; <programlisting>
;;   (gtype-name t)                     ; query type name
;;   gtype:fundamental-types            ; list of root types
;;   (gtype-parent t)                   ; retrieve supertype
;;   (gtype-children t)                 ; retrieve subtypes
;;   (gobject-type o)                   ; extract GType from GObject
;;   (gobject-class-properties t)       ; list properties of class
;;   (gobject:methods-on-class t)       ; list methods on a class
;;   (gobject:methods-in-gf gfname)     ; list methods in a generic function
;;   (gsignal-list t)                   ; list signals in a class
;;   (gsignal-list-complete t)          ; list signals in a class and parents
;; </programlisting>
;;
;; Since GType does not collect the information returned by
;; <function>gobject:methods-on-class</function> or
;; <function>gobject:methods-in-gf</function> itself, explicit calls
;; to <function>gobject:register-method!</function> are required to
;; fill in the associated datastructures.
;;
;; Much of the introspection API is described in the sections devoted
;; to each major grouping of GType instances.

;;@ Represents a GType instance - a representation of a type known to
;;the GLib system. The <structfield>number</structfield> is the
;;unsigned-long GType value as used in C.
(define-record gtype number)
(define-foreign-type _gtype unsigned-long)

(define-record-printer (gtype t out)
  (for-each (lambda (x) (display x out))
	    (list "#<gtype "(gtype-name t)">")))

;;@ Given a <structname>gtype</structname> record, returns the name
;;associated with the GType as a string.
(define (gtype-name t)
  ((foreign-lambda c-string "g_type_name" _gtype)
   (gtype-number t)))

;;@ Wraps a GType number in a <structname>gtype</structname>
;;record. If <parameter>num</parameter> is zero (the invalid GType),
;;<literal>#f</literal> is returned.
(define (wrap-gtype num)
  (if (zero? num)
      #f
      (make-gtype num)))

;;@ Looks up a GType by name, wrapping it in a
;;<structname>gtype</structname> record. Returns <literal>#f</literal>
;;if the type name is not found.
(define (gtype-from-name name)
  (let ((result (wrap-gtype
		 ((foreign-lambda _gtype "g_type_from_name" c-string)
		  name))))
    (if (not result)
	(g-warning "Unknown gtype in gtype-from-name: " name))
    result))

;;@function (raw-gtype->fundamental num)
;; Given a GType number (not a record!), returns the GType number of
;; its ultimate parent type - the root of the inheritance tree for the
;; passed-in GType.
(define raw-gtype->fundamental (foreign-lambda _gtype "G_TYPE_FUNDAMENTAL" _gtype))

;;@ As for <function>raw-gtype-&gt;fundamental</function>, but takes
;;and returns a <structname>gtype</structname> record instead of a raw
;;GType number.
(define (gtype->fundamental t)
  (wrap-gtype (raw-gtype->fundamental (gtype-number t))))

;;@ Produces a <structname>gtype</structname> record from a
;;fundamental type number, using the C macro
;;<function>G_TYPE_MAKE_FUNDAMENTAL</function>.
(define (wrap-gtype-fundamental num)
  (wrap-gtype ((foreign-lambda _gtype "G_TYPE_MAKE_FUNDAMENTAL" int) num)))

;;@function (raw-unmake-gtype-fundamental num)
;; Converts a GType number to its raw fundamental-GType number by
;; shifting right by <symbol>G_TYPE_FUNDAMENTAL_SHIFT</symbol>.
(define raw-unmake-gtype-fundamental
  (foreign-lambda* int ((_gtype t))
		   "return(t >> G_TYPE_FUNDAMENTAL_SHIFT);"))

;;@ As for <function>raw-unmake-gtype-fundamental</function>, but
;;takes a record instead of a GType number.
(define (unwrap-gtype-fundamental t)
  (raw-unmake-gtype-fundamental (gtype-number t)))

;;@variable gtype:...
;; The <varname>gtype:...</varname> variables correspond to the
;; fundamental types defined in <filename>gtype.h</filename> as
;; <symbol>G_TYPE_...</symbol>.
;; <programlisting>
;;     G_TYPE_INVALID       gtype:invalid           
;;     G_TYPE_NONE          gtype:none              
;;     G_TYPE_INTERFACE     gtype:interface         
;;     G_TYPE_CHAR          gtype:char              
;;     G_TYPE_UCHAR         gtype:uchar             
;;     G_TYPE_BOOLEAN       gtype:boolean           
;;     G_TYPE_INT           gtype:int               
;;     G_TYPE_UINT          gtype:uint              
;;     G_TYPE_LONG          gtype:long              
;;     G_TYPE_ULONG         gtype:ulong             
;;     G_TYPE_INT64         gtype:int64             
;;     G_TYPE_UINT64        gtype:uint64            
;;     G_TYPE_ENUM          gtype:enum              
;;     G_TYPE_FLAGS         gtype:flags             
;;     G_TYPE_FLOAT         gtype:float             
;;     G_TYPE_DOUBLE        gtype:double            
;;     G_TYPE_STRING        gtype:string            
;;     G_TYPE_POINTER       gtype:pointer           
;;     G_TYPE_BOXED         gtype:boxed             
;;     G_TYPE_PARAM         gtype:param             
;;     G_TYPE_OBJECT        gtype:object            
;; </programlisting>

(define gtype:invalid		(make-gtype 0))
(define gtype:none		(wrap-gtype-fundamental 1))
(define gtype:interface		(wrap-gtype-fundamental 2))
(define gtype:char		(wrap-gtype-fundamental 3))
(define gtype:uchar		(wrap-gtype-fundamental 4))
(define gtype:boolean		(wrap-gtype-fundamental 5))
(define gtype:int		(wrap-gtype-fundamental 6))
(define gtype:uint		(wrap-gtype-fundamental 7))
(define gtype:long		(wrap-gtype-fundamental 8))
(define gtype:ulong		(wrap-gtype-fundamental 9))
(define gtype:int64		(wrap-gtype-fundamental 10))
(define gtype:uint64		(wrap-gtype-fundamental 11))
(define gtype:enum		(wrap-gtype-fundamental 12))
(define gtype:flags		(wrap-gtype-fundamental 13))
(define gtype:float		(wrap-gtype-fundamental 14))
(define gtype:double		(wrap-gtype-fundamental 15))
(define gtype:string		(wrap-gtype-fundamental 16))
(define gtype:pointer		(wrap-gtype-fundamental 17))
(define gtype:boxed		(wrap-gtype-fundamental 18))
(define gtype:param		(wrap-gtype-fundamental 19))
(define gtype:object		(wrap-gtype-fundamental 20))

;;@ Collects all the fundamental (root) types together in a list.
(define gtype:fundamental-types
  (list gtype:none
	gtype:interface
	gtype:char
	gtype:uchar
	gtype:boolean
	gtype:int
	gtype:uint
	gtype:long
	gtype:ulong
	gtype:int64
	gtype:uint64
	gtype:enum
	gtype:flags
	gtype:float
	gtype:double
	gtype:string
	gtype:pointer
	gtype:boxed
	gtype:param
	gtype:object))

;;@function (gtype-...? t)
;; Predicates for examining attributes of GType records.
;; <programlisting>
;;     gtype-fundamental?       G_TYPE_IS_FUNDAMENTAL
;;     gtype-derived?           G_TYPE_IS_DERIVED
;;     gtype-interface?         G_TYPE_IS_INTERFACE
;;     gtype-classed?           G_TYPE_IS_CLASSED
;;     gtype-instantiatable?    G_TYPE_IS_INSTANTIATABLE
;;     gtype-derivable?         G_TYPE_IS_DERIVABLE
;;     gtype-deep-derivable?    G_TYPE_IS_DEEP_DERIVABLE
;;     gtype-abstract?          G_TYPE_IS_ABSTRACT
;;     gtype-value-abstract?    G_TYPE_IS_VALUE_ABSTRACT
;;     gtype-has-value-table?   G_TYPE_IS_HAS_VALUE_TABLE
;; </programlisting>

(let-macro ((defpred (lambda (s)
		       (let* ((ss (symbol->string s))
			      (cs (string-append "G_TYPE_IS_"
						 (string-map (lambda (ch)
							       (case ch
								 ((#\-) #\_)
								 (else ch)))
							     (string-upcase ss))))
			      (pn (string->symbol (string-append "gtype-" ss "?"))))
			 `(define (,pn t)
			    ((foreign-lambda bool ,cs _gtype)
			     (gtype-number t)))))))
	   (defpred fundamental)
	   (defpred derived)
	   (defpred interface)
	   (defpred classed)
	   (defpred instantiatable) ; -- of course, this should read "gtype-instantiable?"...
	   (defpred derivable)
	   (defpred deep-derivable)
	   (defpred abstract)
	   (defpred value-abstract)
	   (defpred has-value-table))

;;@ Returns the parent type of the passed-in
;;<structname>gtype</structname> record.
(define (gtype-parent t)
  (wrap-gtype
   ((foreign-lambda _gtype "g_type_parent" _gtype)
    (gtype-number t))))

;;@ Returns the depth in the inheritance tree of the passed-in
;;<structname>gtype</structname> record. A fundamental (root) type has
;;depth 1, its child types have depth 2, and so forth.
(define (gtype-depth t)
  ((foreign-lambda unsigned-integer "g_type_depth" _gtype)
   (gtype-number t)))

;;@ Given a <parameter>leaf-t</parameter> and a
;;<parameter>root-t</parameter> which is contained in its ancestry,
;;return the type that <parameter>root-t</parameter> is the immediate
;;parent of. In other words, this function determines the type that is
;;derived directly from <parameter>root-t</parameter> which is also a
;;base class of <parameter>leaf-t</parameter>. Given a root type and a
;;leaf type, this function can be used to determine the types and
;;order in which the leaf type is descended from the root
;;type<footnote><para>Documentation nicked outright from the GLib
;;GType documentation.</para></footnote>.
(define (gtype-next-base leaf-t root-t)
  (wrap-gtype
   ((foreign-lambda _gtype "g_type_next_base" _gtype _gtype)
    (gtype-number leaf-t)
    (gtype-number root-t))))

;;@ Returns <literal>#t</literal> if <parameter>t</parameter> is equal
;;to, or a subtype of, <parameter>is-a-t</parameter>; otherwise
;;returns <literal>#f</literal>.
(define (gtype-isa? t is-a-t)
  ((foreign-lambda bool "g_type_is_a" _gtype _gtype)
   (gtype-number t)
   (gtype-number is-a-t)))

;;@ Returns a list of child types of the passed-in
;;<structname>gtype</structname> record.
(define (gtype-children t)
  (let ((kids ((##core#primitive "CG_gtype_children") (gtype-number t))))
    (and kids (map wrap-gtype kids))))

;;@ Returns a list of the interfaces supported by the passed-in
;;<structname>gtype</structname> record.
(define (gtype-interfaces t)
  (let ((interfaces ((##core#primitive "CG_gtype_interfaces") (gtype-number t))))
    (and interfaces (map wrap-gtype interfaces))))

;---------------------------------------------------------------------------;
;;@section "GBoxed"
;;
;; Boxed types are non-reference-counted, explicitly allocated, copied
;; and freed structures. Each boxed type has a pair of associated copy
;; and free routines, which are called automatically when pointers to
;; GBoxed instances are put under the control of
;; <function>wrap-gboxed</function>.

;;@ Represents a wrapped instance of a GBoxed
;;type. <structfield>type</structfield> is the
;;<structname>gtype</structname> record that is the type of the boxed
;;value; <structfield>pointer</structfield> is the C pointer pointing
;;to the boxed value. Do not call <function>make-gboxed</function>
;;directly - usually, <function>wrap-gboxed</function> is more
;;appropriate (as it arranges for reference-counting/finalization
;;where <function>make-gboxed</function> does not).
(define-record gboxed type pointer)

(define-record-printer (gboxed b out)
  (for-each (lambda (x) (display x out))
	    (list "#<gboxed "(gtype-name (gboxed-type b))">")))

;;@function (gboxed-copy-hook (#:optional new-value))
;; Gets (or sets, if the optional argument is supplied) the current
;; value of the hook function called when a GBoxed instance is to be
;; copied. The default hook is the C function
;; <function>g_boxed_copy</function>. The hook function should take an
;; unsigned long (GType) and a <literal>c-pointer</literal>, and
;; should return a <literal>c-pointer</literal>.
(define gboxed-copy-hook (make-parameter
			  (foreign-lambda c-pointer
					  "g_boxed_copy"
					  _gtype
					  c-pointer)))

;;@function (gboxed-finalizer-hook (#:optional new-value))
;; Gets (or sets, if the optional argument is supplied) the current
;; value of the hook function called when a GBoxed instance is to be
;; destroyed. The default hook does nothing. The hook function should
;; accept an unsigned long (GType) and a <literal>c-pointer</literal>.
(define gboxed-finalizer-hook (make-parameter (lambda (t p) 'nothing)))

;;@function (wrap-gboxed type ptr (#:optional copy?))
;; If <parameter>ptr</parameter> is non-<literal>#f</literal> and
;; non-<symbol>NULL</symbol>, calls <function>g_boxed_copy</function>
;; on it, wraps it in a <structname>gboxed</structname> record, and
;; arranges for <function>g_boxed_free</function> to be called on the
;; copied pointer when the <structname>gboxed</structname> record is
;; garbage collected. <parameter>type</parameter> is required to
;; decide which copying/freeing procedures to use.
;;
;; The optional <parameter>copy?</parameter> parameter defaults to
;; <literal>#t</literal>: it controls whether the pointer is to be
;; copied before being wrapped. If <literal>#f</literal>, the
;; passed-in pointer is wrapped without being copied first. Use this
;; only if you know what you are doing, otherwise you can introduce
;; <quote>double-free</quote> problems to your program.
;;
;; <parameter>copy?</parameter> does not control finalization: all
;; records returned by <function>wrap-gboxed</function> are finalized
;; with <function>g_boxed_free</function> when they are garbage
;; collected, whether they were copied originally or not.
(define wrap-gboxed
  (let* ((gboxed-free (foreign-lambda void "g_boxed_free" _gtype c-pointer))
	 (finalizer (lambda (b)
		      (let ((n (gtype-number (gboxed-type b)))
			    (p (gboxed-pointer b)))
			((gboxed-finalizer-hook) n p)
			(gboxed-free n p)))))
    (lambda (t p . and-copy)
      (and p
	   (not (null-pointer? p))
	   (let ((b (make-gboxed t (if (or (null? and-copy)
					   (and (pair? and-copy)
						(car and-copy)))
				       ((gboxed-copy-hook) (gtype-number t) p)
				       p))))
	     (set-finalizer! b finalizer)
	     b)))))

;;@ Returns the GBoxed equivalent of the null pointer.
(define (null-gboxed)
  (make-gboxed gtype:none #f))

;---------------------------------------------------------------------------;
;;@section "GEnum and GFlags"
;;
;; The GType API provides information about enumeration and flags
;; types registered with the system. The associated wrappers provide
;; convenience functions for introspection and translation between
;; enumeration/flag nicknames and numbers.

;;@ Retrieves a list of information about the values in the
;;enumeration GType record passed in.
(define (genum-info t)
  ((##core#primitive "CG_genum_info") (gtype-number t)))

;;@ Returns a procedure that when given a number returns the
;;associated nickname from the enumeration GType record passed in.
;; <programlisting>
;; ((make-genum-number-&gt;nick (gtype-from-name "GtkJustification"))
;;  3)
;; ==&gt; fill
;; </programlisting>
(define (make-genum-number->nick t)
  (let ((info (genum-info t)))
    (lambda (_)
      (cond
       ((find (lambda (x) (= (third x) _)) info) => first)
       (else (error "Unknown genum number"
		    t
		    _))))))

;;@ Returns a procedure that when given a symbol returns the
;;associated number from the enumeration GType record passed in.
;; <programlisting>
;; ((make-genum-nick-&gt;number (gtype-from-name "GtkJustification"))
;;  'fill)
;; ==&gt; 3
;; </programlisting>
(define (make-genum-nick->number t)
  (let ((info (genum-info t)))
    (lambda (_)
      (cond
       ((assq _ info) => third)
       (else (error "Unknown genum nick"
		    t
		    _))))))

;;@ Retrieves a list of information about the available values in the
;;flags GType record passed in.
(define (gflags-info t)
  ((##core#primitive "CG_gflags_info") (gtype-number t)))

;;@ Returns a procedure that when given a list of symbols returns the
;;bitwise or of the associated numbers from the flags GType record
;;passed in.
;; <programlisting>
;; ((make-gflags-&gt;number (gtype-from-name "GdkWindowState"))
;;  '(iconified sticky))
;; ==&gt; 10
;; </programlisting>
(define (make-gflags->number t)
  (let ((info (gflags-info t)))
    (lambda (_)
      (fold (lambda (x acc)
	      (cond
	       ((assq x info) =>
		(lambda (entry)
		  (bitwise-ior (third entry) acc)))
	       (else (error "Unknown gflags nick"
			    t
			    x
			    _))))
	    0
	    _))))

;;@ Returns a procedure that when given a number returns the list of
;;symbols making up that number, from the flags GType record passed
;;in.
;; <programlisting>
;; ((make-number-&gt;gflags (gtype-from-name "GdkWindowState"))
;;  10)
;; ==&gt; (sticky iconified)
;; </programlisting>
(define (make-number->gflags t)
  (let ((info (gflags-info t)))
    (lambda (_)
      (fold (lambda (entry acc)
	      (if (not (zero? (bitwise-and _ (third entry))))
		  (cons (first entry) acc)
		  acc))
	    '()
	    info))))

;---------------------------------------------------------------------------;
;;@section "GValue"
;;
;; GValue is a subtype of GBoxed which is a polymorphic value cell -
;; it can hold any of the fundamental types and their subclasses. The
;; wrapper provides conversion routines between Scheme objects and
;; GValue instances.

;;@function (raw-gvalue-type gvalue-pointer)
;; Given a C pointer to a GValue object, returns the GType number
;; associated with the GValue.
(define raw-gvalue-type (foreign-lambda _gtype "G_VALUE_TYPE" c-pointer))

;;@ Given a properly boxed GValue, returns the
;;<structname>gtype</structname> record associated with the GValue.
(define (gvalue-type gv)
  (wrap-gtype (raw-gvalue-type (gboxed-pointer gv))))

;;@ Extracts a Scheme object from the passed-in boxed GValue. (Also
;;accepts a raw pointer to a GValue object, instead of a properly
;;boxed GValue, for internal implementation use.)
(define gvalue->object
  (let ()
    (define gvalue-object (foreign-lambda c-pointer "g_value_get_object" c-pointer))

    (define gvalue-char (foreign-lambda char "g_value_get_char" c-pointer))
    (define gvalue-uchar (foreign-lambda unsigned-char "g_value_get_uchar" c-pointer))

    (define gvalue-boolean (foreign-lambda bool "g_value_get_boolean" c-pointer))

    (define gvalue-int (foreign-lambda integer "g_value_get_int" c-pointer))
    (define gvalue-uint (foreign-lambda unsigned-integer "g_value_get_uint" c-pointer))
    (define gvalue-long (foreign-lambda long "g_value_get_long" c-pointer))
    (define gvalue-ulong (foreign-lambda unsigned-long "g_value_get_ulong" c-pointer))

    (define gvalue-int64 (foreign-lambda double "g_value_get_int64" c-pointer))
    (define gvalue-uint64 (foreign-lambda double "g_value_get_uint64" c-pointer))

    (define (gvalue-enum vt v)
      (string->symbol
       ((foreign-lambda* c-string ((_gtype vt)
				   (c-pointer v))
			 "GEnumClass *eclass = G_ENUM_CLASS(g_type_class_ref(vt));"
			 "GEnumValue *val = g_enum_get_value(eclass, g_value_get_enum(v));"
			 "if (val == NULL) {"
			 "  return(NULL);"
			 "} else {"
			 "  return(val->value_nick);"
			 "}")
	vt
	v)))

    (define gvalue-flags (foreign-lambda unsigned-integer "g_value_get_flags" c-pointer))
    (define gvalue-float (foreign-lambda float "g_value_get_float" c-pointer))
    (define gvalue-double (foreign-lambda double "g_value_get_double" c-pointer))
    (define gvalue-string (foreign-lambda c-string "g_value_get_string" c-pointer))
    (define gvalue-pointer (foreign-lambda c-pointer "g_value_get_pointer" c-pointer))
    (define gvalue-boxed (foreign-lambda c-pointer "g_value_get_boxed" c-pointer))

    (lambda (gv)
      (let* ((v (if (gboxed? gv)
		    (gboxed-pointer gv)
		    gv))
	     (raw-value-type (raw-gvalue-type v))
	     (value-type (wrap-gtype raw-value-type))
	     (fundamental-type (raw-gtype->fundamental raw-value-type)))
	(case (raw-unmake-gtype-fundamental fundamental-type)
	  ((2 20) ; G_TYPE_INTERFACE and G_TYPE_OBJECT
	   (wrap-gobject (gvalue-object v)))
	  ((3) ; G_TYPE_CHAR
	   (gvalue-char v))
	  ((4) ; G_TYPE_UCHAR
	   (gvalue-uchar v))
	  ((5) ; G_TYPE_BOOLEAN
	   (gvalue-boolean v))
	  ((6) ; G_TYPE_INT
	   (gvalue-int v))
	  ((7) ; G_TYPE_UINT
	   (gvalue-uint v))
	  ((8) ; G_TYPE_LONG
	   (gvalue-long v))
	  ((9) ; G_TYPE_ULONG
	   (gvalue-ulong v))
	  ((10) ; G_TYPE_INT64
	   (gvalue-int64 v))
	  ((11) ; G_TYPE_UINT64
	   (gvalue-uint64 v))
	  ((12) ; G_TYPE_ENUM
	   (gvalue-enum raw-value-type v))
	  ((13) ; G_TYPE_FLAGS
	   (gvalue-flags v))
	  ((14) ; G_TYPE_FLOAT
	   (gvalue-float v))
	  ((15) ; G_TYPE_DOUBLE
	   (gvalue-double v))
	  ((16) ; G_TYPE_STRING
	   (gvalue-string v))
	  ((17) ; G_TYPE_POINTER
	   (gvalue-pointer v))
	  ((18) ; G_TYPE_BOXED
	   (wrap-gboxed value-type (gvalue-boxed v)))
	  (else (error "Unsupported fundamental type in gvalue->object"
		       value-type
		       v)))))))

;;@ Empties a boxed GValue, without altering the type associated with
;;it.
(define (gvalue-empty! gv)
  ((foreign-lambda void "g_value_unset" c-pointer)
   (gboxed-pointer gv)))

;;@function (make-gvalue (#:optional gtype-record))
;; Returns a newly-allocated, boxed GValue, with its type set to the
;; passed in GType record. If <parameter>gtype-record</parameter> is
;; omitted, returns a completely blank GValue object, ready for
;; filling in with any type (by, for instance,
;; <function>gtk-tree-model-get-value</function>).
(define make-gvalue
  (let ((alloc (foreign-lambda* c-pointer ((_gtype t))
				"GValue *v = g_new0(GValue, 1);"
				"v->g_type = t;"
				"return(v);"))
	(gvalue_t (gtype-from-name "GValue")))
    (lambda maybe-t
      (wrap-gboxed gvalue_t
		   (alloc (if (pair? maybe-t)
			      (gtype-number (car maybe-t))
			      0))
		   #f))))

;;@function (raw-gvalue-fill! gvalue-ptr scheme-object)
;; Fills a pointer to a GValue object with a value taken from the
;; passed-in Scheme object. If the type of
;; <parameter>scheme-object</parameter> is not compatible with the
;; type of <parameter>gvalue-ptr</parameter>, returns
;; <literal>#f</literal>; if the fill operation was otherwise
;; successful, returns <literal>#t</literal>.
(define raw-gvalue-fill! (##core#primitive "CG_fill_gvalue"))

;;@ Fills a properly boxed GValue object with the value of the
;;passed-in scheme object, as for
;;<function>raw-gvalue-fill!</function>.
(define (gvalue-fill! gv o)
  (raw-gvalue-fill! (gboxed-pointer gv) o))

;;@ Allocates a new boxed GValue of type <parameter>t</parameter>
;;using <function>make-gvalue</function>, fills it using
;;<function>gvalue-fill!</function>, and returns it. If the fill
;;operation failed, an <function>error</function> is signalled.
(define (object->gvalue t o)
  (let ((v (make-gvalue t)))
    (if (gvalue-fill! v o)
	v
	(error "object->gvalue: incompatible types"
	       t
	       o))))

;---------------------------------------------------------------------------;
;;@section "GClosure"
;;
;; Only basic support for GClosures is implemented, using a custom
;; marshalling function
;; (<function>cg_gclosure_marshaller</function>). Scheme functions
;; wrapped in GClosure instances are properly collected - when the
;; GClosure object is destroyed, a finalizer function
;; (<function>cg_gclosure_finalizer</function>) causes the handle on
;; the scheme function to be released.
;;
;; GClosures are not usually manipulated explicitly in Scheme
;; code. Usually a function like
;; <function>gsignal-connect</function>
;; (<acronym>a.k.a.</acronym> <function>gtk-signal-connect</function>)
;; is used, which transparently manages GClosure instances.

(define gobject:callbacks (make-hash-table =))
(define gobject:freelist '())
(define gobject:nextnum 1)

(define gobject:gpointer-to-uint (foreign-lambda unsigned-integer "GPOINTER_TO_UINT" c-pointer))
(define gobject:extract-param-gvalue (foreign-lambda* c-pointer ((c-pointer param_values)
								 (unsigned-integer index))
						      "return(&((GValue *)param_values)[index]);"))
(define gobject:closure-data (foreign-lambda* unsigned-integer ((c-pointer closure))
					      "return(GPOINTER_TO_UINT("
					      "  ((GClosure *)closure)->data));"))

(define-external (cg_gclosure_finalizer (c-pointer data)
					(c-pointer closure))
  void
  (let ((n (gobject:gpointer-to-uint data)))
    (hash-table-remove! gobject:callbacks n)
    (set! gobject:freelist (cons n gobject:freelist))))

(define-external (cg_gclosure_marshaller (c-pointer closure)
					 (c-pointer return-value)
					 (unsigned-integer n-param-values)
					 (c-pointer param-values)
					 (c-pointer invocation-hint)
					 (c-pointer marshal-data))
  void
  (let ((fn (hash-table-ref gobject:callbacks (gobject:closure-data closure))))
    (if (not fn)
	(g-warning "No scheme closure found for index " n)
	(let ((result (apply fn (list-tabulate
				 n-param-values
				 (lambda (n)
				   (gvalue->object
				    (gobject:extract-param-gvalue param-values n)))))))
	  (unless (or (not return-value)
		      (null-pointer? return-value)
		      (raw-gvalue-fill! return-value result))
	    (error "gvalue-fill! returned false for result of closure"
		   (gvalue-type return-value)
		   result
		   n))))))

;;@ Wrap a scheme function in a GClosure, and return a C pointer to
;;the new GClosure structure. See
;;<function>gsignal-connect</function>.
(define make-gclosure
  (let ((finalizer cg_gclosure_finalizer)
	(marshaller cg_gclosure_marshaller))
    (lambda (fn)
      (let* ((n (if (null? gobject:freelist)
		    (let ((n gobject:nextnum))
		      (set! gobject:nextnum (+ n 1))
		      n)
		    (let ((n gobject:freelist))
		      (set! gobject:freelist (cdr n))
		      (car n))))
	     (p ((foreign-lambda* c-pointer ((unsigned-integer n))
				  "GClosure *c = g_closure_new_simple(sizeof(GClosure),"
				  "                                   GUINT_TO_POINTER(n));"
				  "g_closure_add_finalize_notifier(c, GUINT_TO_POINTER(n),"
				  "                                (GClosureNotify)"
				  "                                cg_gclosure_finalizer);"
				  "g_closure_set_marshal(c, (GClosureMarshal)"
				  "                         cg_gclosure_marshaller);"
				  "return(c);")
		 n)))
	(hash-table-set! gobject:callbacks n fn)
	p))))

;---------------------------------------------------------------------------;
;;@section "GObject"
;;
;; GObject is the base type for all reference-counted objects in the
;; GType hierarchy.

;;@ Represents a GObject instance. <structfield>pointer</structfield>
;;is the C pointer to the GObject instance. Do not call
;;<function>make-gobject</function> directly - use
;;<function>wrap-gobject</function> instead.
(define-record gobject pointer)

(define-record-printer (gobject o out)
  (for-each (lambda (x) (display x out))
	    (list "#<gobject "(gtype-name (gobject-type o))">")))

;;@ Returns the <structname>gtype</structname> record representing the
;;type of the passed-in GObject.
(define (gobject-type o)
  (let ((p (gobject-pointer o)))
    (if p
	(wrap-gtype
	 ((foreign-lambda _gtype "G_OBJECT_TYPE" c-pointer)
	  (gobject-pointer o)))
	gtype:none)))

;;@function (gobject-ref-hook (#:optional new-value))
;; Gets (or sets, if the optional argument is supplied) the current
;; value of the hook function called when a GObject instance is to be
;; referenced. The default hook is the C function
;; <function>g_object_ref</function>. The hook function should take a
;; <literal>c-pointer</literal> and return a
;; <literal>c-pointer</literal>.
(define gobject-ref-hook (make-parameter
			  (foreign-lambda c-pointer "g_object_ref" c-pointer)))

;;@function (gobject-finalizer-hook (#:optional new-value))
;; Gets (or sets, if the optional argument is supplied) the current
;; value of the hook function called when a GObject instance is to be
;; unreferenced. The default hook does nothing. The hook function
;; should accept a <literal>c-pointer</literal>.
(define gobject-finalizer-hook (make-parameter (lambda (p) 'nothing)))

;;@ Given a C pointer to a GObject instance, calls
;;<function>g_object_ref</function> on it, constructs a
;;<structname>gobject</structname> record for it, and registers
;;<function>g_object_unref</function> as the finalizer for the new
;;record. If <parameter>p</parameter> is <literal>#f</literal> or the
;;null pointer, <literal>#f</literal> is returned; otherwise the
;;newly-allocated <structname>gobject</structname> record is returned.
(define wrap-gobject
  (let* ((gobject-unref (foreign-lambda void "g_object_unref" c-pointer))
	 (finalizer (lambda (o)
		      (let ((p (gobject-pointer o)))
			((gobject-finalizer-hook) p)
			(gobject-unref p)))))
    (lambda (p)
      (and p
	   (not (null-pointer? p))
	   (let ((o (make-gobject ((gobject-ref-hook) p))))
	     (set-finalizer! o finalizer)
	     o)))))

;;@ Returns the GObject equivalent of the null pointer. Useful with
;;functions like <function>gtk-scrolled-window-new</function>.
(define (null-gobject)
  (make-gobject #f))

(define (decode-gobject-property-description entry)
  (list (first entry)
	(wrap-gtype (second entry))
	(case (third entry)
	  ((0) '(				))
	  ((1) '(			read	))
	  ((2) '(		write		))
	  ((3) '(		write	read	))
	  ((4) '(construct-only			))
	  ((5) '(construct-only		read	))
	  ((6) '(construct-only	write		))
	  ((7) '(construct-only	write	read	))
	  (else (error "decode-gobject-property-description: flags value out of bounds"
		       7
		       (third entry))))))

;;@ Returns a list of the properties supported by instances of the
;;GObject GType record passed in.
(define (gobject-class-properties t)
  (let ((plist ((##core#primitive "CG_object_list_properties")
		(gtype-number t))))
    (and plist (map decode-gobject-property-description plist))))

;;@ Returns a property specification for the named property on
;;instances of the GObject GType record passed in, or
;;<literal>#f</literal> if no property by that name is found on that
;;class.
(define (gobject-class-find-property t pname)
  (assert (string? pname))
  (let ((p ((##core#primitive "CG_object_find_property")
	    (gtype-number t)
	    pname)))
    (and p (decode-gobject-property-description p))))

;;@ Produces a getter function for the passed-in GType and property
;;name.
(define (make-gobject-property-getter t pname-symbol-or-string)
  (let* ((pname (if (symbol? pname-symbol-or-string)
		    (symbol->string pname-symbol-or-string)
		    pname-symbol-or-string))
	 (info (gobject-class-find-property t pname)))
    (if (and info
	     (member 'read (third info)))
	(lambda (o)
	  (let ((v (make-gvalue (second info))))
	    ((foreign-lambda void "g_object_get_property" c-pointer c-string c-pointer)
	     (gobject-pointer o)
	     pname
	     (gboxed-pointer v))
	    (gvalue->object v)))
	(error "make-gobject-property-getter: property unreadable"
	       t
	       pname
	       info))))

;;@ Retrieves the value of the named property on the GObject instance
;;passed in.
(define (gobject-get-property o pname)
  ((make-gobject-property-getter (gobject-type o) pname) o))

;;@ Produces a setter function for the passed-in GType and property
;;name.
(define (make-gobject-property-setter t pname-symbol-or-string)
  (let* ((pname (if (symbol? pname-symbol-or-string)
		    (symbol->string pname-symbol-or-string)
		    pname-symbol-or-string))
	 (info (gobject-class-find-property t pname)))
    (if (and info
	     (member 'write (third info)))
	(lambda (o newval)
	  (let ((v (make-gvalue (second info))))
	    (if (gvalue-fill! v newval)
		((foreign-lambda void "g_object_set_property" c-pointer c-string c-pointer)
		 (gobject-pointer o)
		 pname
		 (gboxed-pointer v))
		(error "make-gobject-property-setter: gvalue-fill! failed"
		       t
		       info
		       newval))))
	(error "make-gobject-property-setter: property unwritable"
	       t
	       pname
	       info))))

;;@ Updates the value of the named property on the GObject instance
;;passed in.
(define (gobject-set-property! o pname newval)
  ((make-gobject-property-setter (gobject-type o) pname) o newval))

(define gobject:methods (make-hash-table equal?))
(define gobject:gfs (make-hash-table eq?))

;;@ Represents a method associated with a GObject
;;class. <structfield>name</structfield> is the name of the method;
;;<structfield>gf</structfield> is the name of the generic function;
;;<structfield>class</structfield> is the GType record for the class;
;;and <structfield>function</structfield> is the method function
;;itself.
(define-record gobject-method name gf class function)

(define-record-printer (gobject-method m out)
  (for-each (lambda (x) (display x out))
	    (list "#<gobject-method " (gtype-name (gobject-method-class m))
		  ":" (gobject-method-gf m) ">")))

;;@ Retrieve a list of all methods supported by the GObject GType
;;passed in.
(define (gobject:methods-on-class g)
  (hash-table-ref gobject:methods g '()))

;;@ Retrieve a list of all methods in the named generic function.
(define (gobject:methods-in-gf gfname)
  (hash-table-ref gobject:gfs gfname '()))

;;@ Registers a method on a particular class with the system. This
;;procedure is called by the generated code for the GTK+ wrapper.
(define (gobject:register-method! classname gfname methodname function)
  (let* ((class (gtype-from-name classname))
	 (method (make-gobject-method methodname
				      gfname
				      class
				      function)))
    (let ((methods (gobject:methods-on-class class)))
      (hash-table-set! gobject:methods
		       class
		       (lset-adjoin (lambda (x y) (eq? (gobject-method-name x)
						       (gobject-method-name y)))
				    methods
				    method)))
    (let ((methods (gobject:methods-in-gf gfname)))
      (hash-table-set! gobject:gfs
		       gfname
		       (lset-adjoin (lambda (x y) (equal? (gobject-method-class x)
							  (gobject-method-class y)))
				    methods
				    method)))))

;---------------------------------------------------------------------------;
;;@section "GSignal"
;;
;; Only a partial interface to the GSignal system is supported. In
;; particular, there is no support for signal emission.

;;@function (gsignal-connect o sigdetail fn (#:optional after))
;; (also known as <function>gtk-signal-connect</function> within the
;; gtk module) Connects <parameter>fn</parameter> to the signal
;; (string or symbol) <parameter>sigdetail</parameter> on GObject
;; instance <parameter>o</parameter>. When the signal is emitted,
;; <parameter>fn</parameter> will be called with an argument list
;; appropriate to the particular signal. Returns a number representing
;; the connection which can then be passed into
;; <function>gsignal-handler-disconnect</function>.
(define (gsignal-connect o sigdetail fn . after)
  ((foreign-lambda unsigned-long "g_signal_connect_closure"
		   c-pointer
		   c-string
		   c-pointer
		   bool)
   (gobject-pointer o)
   (if (symbol? sigdetail)
       (symbol->string sigdetail)
       sigdetail)
   (make-gclosure fn)
   (and (pair? after)
	(car after))))

;;@ Given an object and a handler connection number as returned by
;;<function>gsignal-connect</function>, disconnects the
;;handler so it will no longer fire when the signal is emitted.
(define (gsignal-disconnect o handlerid)
  ((foreign-lambda void "g_signal_handler_disconnect"
		   c-pointer
		   unsigned-long)
   (gobject-pointer o)
   handlerid))

;;@ Look up a signal in a class by name; returns zero if the signal is
;;not found for some reason.
(define (gsignal-lookup name t)
  ((foreign-lambda unsigned-integer "g_signal_lookup" c-string _gtype)
   name
   (gtype-number t)))

;;@ Returns a list containing information about the signal identified
;;by the signal identifier number passed in.
(define gsignal-query
  (let* ((signal-name (foreign-lambda* c-string ((byte-vector q))
				       "return(((GSignalQuery*)q)->signal_name);"))
	 (signal-itype (foreign-lambda* _gtype ((byte-vector q))
					"return(((GSignalQuery*)q)->itype);"))
	 (signal-flags (foreign-lambda* unsigned-integer ((byte-vector q))
					"return(((GSignalQuery*)q)->signal_flags);"))
	 (signal-return-type (foreign-lambda* _gtype ((byte-vector q))
					      "return(((GSignalQuery*)q)->return_type);"))
	 (signal-n-params (foreign-lambda* unsigned-integer ((byte-vector q))
					   "return(((GSignalQuery*)q)->n_params);"))
	 (signal-param (foreign-lambda* _gtype ((byte-vector q) (unsigned-integer index))
					"return(((GSignalQuery*)q)->param_types[index]);"))
	 (signal-type-gtype (lambda (n)
			      (wrap-gtype
			       ((foreign-lambda* _gtype ((_gtype in))
						 "return(in & ~G_SIGNAL_TYPE_STATIC_SCOPE);")
				n))))
	 (signal-type-static-scope? (foreign-lambda* bool ((_gtype in))
						     "return(in & G_SIGNAL_TYPE_STATIC_SCOPE);"))
	 (signal-type-decode (lambda (t)
			       (list (signal-type-gtype t)
				     (signal-type-static-scope? t))))
	 (sizeof-signal-query ((foreign-lambda* unsigned-integer ()
						"return(sizeof(GSignalQuery));"))))
    (lambda (sigid)
      (let ((q (make-byte-vector sizeof-signal-query)))
	((foreign-lambda* void ((unsigned-integer sigid)
				(byte-vector q))
			  "g_signal_query(sigid, (GSignalQuery*)q);")
	 sigid
	 q)
	(list (signal-name q)
	      (wrap-gtype (signal-itype q))
	      (signal-flags q)
	      (signal-type-decode (signal-return-type q))
	      (list-tabulate (signal-n-params q)
			     (lambda (x) (signal-type-decode (signal-param q x)))))))))

;;@ Returns a list of information about the signals that can be
;;emitted by objects of the passed-in GType record, but not signals
;;that can be emitted by its supertypes.
(define (gsignal-list t)
  (assert (gtype? t))
  (let* ((n ((foreign-lambda* unsigned-integer ((unsigned-integer t))
			      "guint n = 0;"
			      "CG_signal_ids = g_signal_list_ids(t, &n);"
			      "return(n);")
	     (gtype-number t)))
	 (get-id (foreign-lambda* unsigned-integer ((unsigned-integer index))
				  "return(CG_signal_ids[index]);"))
	 (release (foreign-lambda* void ()
				   "g_free(CG_signal_ids);"
				   "CG_signal_ids = NULL;"))
	 (signal-ids (list-tabulate n get-id)))
    (release)
    (map gsignal-query signal-ids)))

;;@ Returns a list of information about the signals that can be
;;emitted by objects of the passed-in GType record, including the
;;signals that can be emitted by its supertypes.
(define (gsignal-list-complete t)
  (apply append
	 (unfold not gsignal-list gtype-parent t)))
