(define *wrap-version* "2.5")

(require 'extras)	;; To get string-map to allow it to be redefined by srfi-13

(require 'srfi-1	;; List library
	 'srfi-13	;; String library
	 'srfi-18)	;; Raise - exceptions

(require 'script-utils)

(require-for-syntax 'moremacros)
;(require-for-syntax 'srfi-26)		; I'm using cut/cute below
					; - they happen to be builtin
					; in chicken.

(require 'gobject)
(require-for-syntax 'gobject)

(gtype:init-types-from-file "gdk-types")
(gtype:init-types-from-file "gtk-types")

(case-sensitive #t)

(define (eprint . x)
  (with-output-to-port (current-error-port)
    (lambda ()
      (apply print x))))

(define *include-path* (list "."))
(define *output-dir* #f)
(define *header-string* "")

;---------------------------------------------------------------------------;

(define (symbol-append . args)
  (string->symbol
   (apply string-append
	  (map (lambda (x)
		 (if (symbol? x)
		     (symbol->string x)
		     x))
	       args))))

(define (schemeify name)
  (string->symbol
   (string-map (lambda (ch)
		 (case ch
		   ((#\_) #\-)
		   (else (char-downcase ch))))
	       (if (string? name)
		   name
		   (symbol->string name)))))

(define (cify name)
  (string->symbol
   (string-append (string-map (lambda (ch)
				(case ch
				  ((#\-) #\_)
				  (else ch)))
			      (if (string? name)
				  name
				  (symbol->string name)))
		  "_")))

(define (basetype-for type)
  (let* ((typestr (if (symbol? type)
		      (symbol->string type)
		      type))
	 (nostars (string-trim-right typestr #\*))
	 (noconst (if (string-prefix? "const-" nostars)
		      (substring nostars 6 (string-length nostars))
		      nostars)))
    (string->symbol noconst)))

(define (basic-type? type)
  (eq? (basetype-for type) type))

(define (unsupported-type type . reason)
  (raise (cons* 'unsupported-type type reason)))

(define (translate-type type is-return-type)
  (assert (symbol? type))
  (cond
   ((or (string-suffix? "**" (symbol->string type))
	(string-suffix? "[]" (symbol->string type)))
    (unsupported-type type 'too-much-indirection))
   (else
    (case type
      ((none)				(values 'void 'simple))
      ((gboolean bool)			(values 'bool 'simple))
      ((gchar char gint8)		(values 'char 'simple))
      ((guchar guint8)			(values 'unsigned-char 'simple))
      ((gint gint32 int)		(values 'integer 'simple))
      ((guint guint32 uint)		(values 'unsigned-integer 'simple))
      ((guint16 uint16)			(values 'unsigned-short 'simple))
      ((gulong ulong)			(values 'unsigned-long 'simple))
      ((gdouble double gfloat float)	(values 'double 'simple))
      ((const-gchar*
	consg-gchar*	;; there's a typo in one of the .defs files
	const-char*
	gchar*
	char*
	string)				(values 'c-string 'simple))
      ((const-guchar*
	guchar*)			(if is-return-type
					    (unsupported-type type
							      'unsupported-as-result-type)
					    (values 'byte-vector 'simple)))

      ((GType
	GtkType)			(values 'unsigned-long 'enum-or-flags))
      ((GValue*
	GObjectClass*)			(values 'c-pointer 'boxed))

      (else
       (let* ((basetype (basetype-for type))
	      (g (fluid-let ((g-warning (lambda _ _)))
		   (gtype-from-name (symbol->string basetype))))
	      (f (and g (gtype->fundamental g))))
	 (cond
	  ((not f)				(unsupported-type type
								  'not-found-by-name basetype))
	  ((member f (list gtype:flags
			   gtype:enum))		(values 'unsigned-integer 'enum-or-flags))
	  ((member f (list gtype:boxed
			   gtype:interface
			   gtype:object))	(values 'c-pointer 'boxed))
	  (else					(unsupported-type type g f)))))))))

(define-record func-arg name cname type ctype style)

(define (chew-parameters parameters)
  (map (lambda (param)
	 (let ((type (string->symbol (first param)))
	       (name (string->symbol (second param))))
	   (receive (ctype style)
	       (translate-type type #f)
	     (make-func-arg name
			    (cify name)
			    type
			    ctype
			    style))))
       (map second parameters)))

(define (emit-function name c-name return-type parameters)
  (handle-exceptions
   exn
   (cond
    ((and (pair? exn)
	  (eq? (car exn) 'unsupported-type))
     (let ((emsg (list ";;; Skipped - "c-name" - unsupported type - "(cdr exn))))
       (apply print emsg)
       (apply eprint emsg))
     #f)
    (else (raise exn)))

   (let ((args (chew-parameters parameters)))
     (receive (chicken-rettype rettype-style) (translate-type (string->symbol return-type) #t)
       (let* ((inner-body
	       `((foreign-lambda
		  ;; return type
		  ,chicken-rettype

		  ;; c-name
		  ,c-name

		  ;; argument list
		  ,@(map func-arg-ctype args)

		  )

		 ;; argument values
		 ,@(map (lambda (a)
			  (case (func-arg-style a)
			    ((simple) (func-arg-name a))
			    ((enum-or-flags)
			     (if (not (basic-type? (func-arg-type a)))
				 (unsupported-type (func-arg-type a)))
			     `(,(symbol-append (func-arg-type a) '->number)
			       ,(func-arg-name a)))
			    ((boxed) `(,(symbol-append 'g:unbox- (basetype-for (func-arg-type a)))
				       ,(func-arg-name a)))))
			args)))

	      (outer-body
	       (case rettype-style
		 ((simple enum-or-flags) inner-body)
		 ((boxed) `(,(symbol-append 'g:box- (basetype-for return-type)) ,inner-body)))))

	 (pretty-print
	  `(define (,(schemeify c-name) ,@(map func-arg-name args))
	     ,outer-body))))
     #t)))

;---------------------------------------------------------------------------;

(define-record classdef name class-methods methods)

(define *functions* (make-hash-table))
(define *classes* (make-hash-table string=?))

(define (register-function! name body)
  (cond
   ((assq 'is-constructor-of body) (register-method! name body))
   ((hash-table-ref *functions* name)
    (eprint ";; Warning! attempt to redefine function "name))
   (else (hash-table-set! *functions* name body))))

(define (lookup-classdef classname)
  (hash-table-ref *classes* classname))

(define (register-method! name body)
  (receive (classname is-class-method)
      (cond
       ((assq 'of-object body) => (compose (cut values <> #f) cadr))
       ((assq 'is-constructor-of body) => (compose (cut values <> #t) cadr))
       (else (eprint ";; Warning! Method "name" registered without class name!")
	     (values #f #f)))
    (if classname
	(let* ((classname (if (symbol? classname)
			      (symbol->string classname)
			      classname))
	       (classdef (or (lookup-classdef classname)
			     (let ((cd (make-classdef classname '() '())))
			       (hash-table-set! *classes* classname cd)
			       cd))))
	  (if is-class-method
	      (classdef-class-methods-set! classdef
					   (cons (list name body)
						 (classdef-class-methods classdef)))
	      (classdef-methods-set! classdef
				     (cons (list name body)
					   (classdef-methods classdef))))))))

;---------------------------------------------------------------------------;

(define (generate-file filename thunk)
  (let ((path (make-pathname *output-dir* filename "scm")))
    (eprint "; Generating file " path)
    (with-output-to-file path
      (lambda ()
	(print ";;; -*- scheme -*-")
	(print ";;; Auto-generated from file "filename" by wrap.scm")
	(print ";;; $Id: wrap.scm,v 1.5 2002/10/12 23:54:40 tonyg Exp $")
	(print *header-string*)
	(newline)
	(thunk)))))

;---------------------------------------------------------------------------;

(define class-units '())

(define (emit-methods-for classname)
  (let ((classdef (lookup-classdef classname)))
    (when classdef
      ;; Class methods.
      ;;
      (for-each (lambda (function)
		  (let ((name (first function))
			(body (second function)))
		    (emit-function name
				   (cadr (assq 'c-name body))
				   (cond
				    ((assq 'return-type body) => cadr)
				    (else "none"))
				   (cond
				    ((assq 'parameters body) => cdr)
				    (else '())))))
		(classdef-class-methods classdef))

      ;; Methods.
      ;;
      (for-each (lambda (method)
		  (let ((gfname (first method))
			(body (second method)))
		    (let ((c-name (cadr (assq 'c-name body)))
			  (of-object (cadr (assq 'of-object body))))
		      (if (emit-function c-name
					 c-name
					 (cond
					  ((assq 'return-type body) => cadr)
					  (else "none"))
					 (cons `(quote (,(string-append of-object "*")
							"self__param"))
					       (cond
						((assq 'parameters body) => cdr)
						(else '()))))
			  (begin
			    (newline)
			    (pretty-print
			     `(gobject:register-method! ,of-object
							',(schemeify gfname)
							',(schemeify c-name)
							,(schemeify c-name)))
			    (newline))))))
		(classdef-methods classdef)))))

(define (emit-classdef classname filebase)
  (generate-file
   filebase
   (lambda ()
     (set! class-units (cons filebase class-units))
     (for-each pretty-print
	       (list

		;; We're a unit, because we'll all be assembled
		;; into a giant extension later.
		;;
		`(declare (unit ,(string->symbol filebase)))

		;; Predicate.
		;;
		`(define ,(symbol-append classname '?)
		   (let ((t (gtype-from-name ,classname)))
		     (lambda (x)
		       (and (gobject? x)
			    (gtype-isa? (gobject-type x) t)))))

		;; Wrapper.
		;;
		`(define ,(symbol-append 'g:box- classname) wrap-gobject)

		;; Unwrapper.
		;;
		`(define ,(symbol-append 'g:unbox- classname) gobject-pointer)))
     (newline)

     (emit-methods-for classname)
     (newline))))

(define (for-each-type fn root-type)
  (let walk ((t root-type))
    (let ((kids (gtype-children t)))
      (fn t)
      (for-each walk kids))))

(define (for-each-child-type fn root-type)
  (for-each (cute for-each-type fn <>) (gtype-children root-type)))

(define (emit-classes root-type prefix)
  (for-each-child-type
   (lambda (class)
     (let* ((classname (gtype-name class))
	    (filebase (string-downcase (string-append prefix"-"classname))))
       (emit-classdef classname filebase)))
   root-type))

(define (emit-objects)
  (emit-classes (gtype-from-name "GObject")
		"wrap-class"))

(define (emit-interfaces)
  (emit-classes (gtype-from-name "GInterface")
		"wrap-interface"))

(define (emit-classes-main)
  (generate-file
   "wrap-classes"
   (lambda ()
     (pretty-print
      `(declare (uses ,@(map string->symbol class-units)))))))

(define (emit-boxeds)
  (generate-file
   "wrap-boxed"
   (lambda ()
     (for-each-child-type
      (lambda (t)
	(let ((c-name (gtype-name t)))
	  (for-each pretty-print
		    (list
		     `(define ,(symbol-append c-name '?)
			(let ((t (gtype-from-name ,c-name)))
			  (lambda (x)
			    (and (gboxed? x)
				 (gtype-isa? (gboxed-type x) t)))))
		     `(define ,(symbol-append 'g:box- c-name)
			(let ((t (gtype-from-name ,c-name)))
			  (lambda (p)
			    (wrap-gboxed t p))))
		     `(define ,(symbol-append 'g:unbox- c-name) gboxed-pointer)))
	  (newline)
	  (emit-methods-for c-name)
	  (newline)))
      (gtype-from-name "GBoxed")))))

(define (emit-enums)
  (generate-file
   "wrap-enum"
   (lambda ()
     (for-each-child-type
      (lambda (t)
	(let ((c-name (gtype-name t)))
	  (for-each pretty-print
		    (list
		     `(define ,(symbol-append c-name '->number)
			(make-genum-nick->number (gtype-from-name ,c-name)))
		     `(define ,(symbol-append 'number-> c-name)
			(make-genum-number->nick (gtype-from-name ,c-name))))))
	(newline))
      (gtype-from-name "GEnum")))))

(define (emit-flags)
  (generate-file
   "wrap-flags"
   (lambda ()
     (for-each-child-type
      (lambda (t)
	(let ((c-name (gtype-name t)))
	  (for-each pretty-print
		    (list
		     `(define ,(symbol-append c-name '->number)
			(make-gflags->number (gtype-from-name ,c-name)))
		     `(define ,(symbol-append 'number-> c-name)
			(make-number->gflags (gtype-from-name ,c-name))))))
	(newline))
      (gtype-from-name "GFlags")))))

(define (emit-functions)
  (generate-file
   "wrap-functions"
   (lambda ()
     (hash-table-for-each
      (lambda (name body)
	(emit-function name
		       (cadr (assq 'c-name body))
		       (cond
			((assq 'return-type body) => cadr)
			(else "none"))
		       (cond
			((assq 'parameters body) => cdr)
			(else '()))))
      *functions*))))

;---------------------------------------------------------------------------;

(define (process-include filename)
  (eprint ";; Including definitions from " filename)
  (input-file filename))

(define (process-definition defsym name body)
  (let* ((defstr (symbol->string defsym))
	 (kind (string->symbol (substring defstr 7 (string-length defstr)))))
    (case kind
      ((method)		(register-method! name body))
      ((function)	(register-function! name body))
      (else		#t)))) ;; Ignore it. Use reflection instead.

(define (with-input-from-file-in-include-path filename thunk)
  (let ((h (any (lambda (dir)
		  (handle-exceptions exn #f
				     (let ((p (string-append dir "/" filename)))
				       (list (open-input-file p)
					     p))))
		*include-path*)))
    (if h
	(let ((port (first h))
	      (path (second h)))
	  (eprint ";; Reading from file "path"...")
	  (handle-exceptions exn
			     (begin
			       (close-input-port port)
			       (raise exn))
			     (with-input-from-port
				 port
			       thunk)))
	(error "Could not open file" filename))))

(define (input-file filename)
  (with-input-from-file-in-include-path filename
    (lambda ()
      (let loop ()
	(let ((d (read)))
	  (unless (eof-object? d)
	    (cond
	     ((not (pair? d))
	      (print "; Skipping non-pair " d))
	     ((eq? (car d) 'include)
	      (process-include (cadr d)))
	     ((string-prefix? "define-" (symbol->string (car d)))
	      (process-definition (car d) (cadr d) (cddr d)))
	     (else
	      (print "; Skipping unhandled key " (car d))))
	    (loop)))))))

(define (help)
  (with-output-to-port (current-error-port)
    (lambda ()
      (display #<<EOF
Usage: wrap -o <directory> [[-I <directory>] [-i <filename>] <filename>] ...

  -h --help		Prints this message.
  -I --include-path     Prepend a directory to the path searched
			for included files.
  -i --include-file	Include a single file.
  -H --header		Specify the header file to include in each
			generated file.
  -o --output-dir	Generate output in the named directory.
			The directory must exist.
			This argument must be supplied.
  -v --version		Print version of wrap to stdout.

EOF
)
      (newline))))

(define (main argv)
  (let ((filenames (args-fold argv
			      (list (option '(#\h "help") #f #f
					    (lambda (o n a x)
					      (help)
					      (exit 0)))
				    (option '(#\I "include-path") #t #f
					    (lambda (o n a x)
					      (set! *include-path*
						    (cons a *include-path*)) x))
				    (option '(#\i "include-file") #t #f
					    (lambda (o n a x)
					      (process-include a) x))
				    (option '(#\H "header") #t #f
					    (lambda (o n a x)
					      (set! *header-string*
						    (string-join
						     (with-input-from-file
							 a
						       read-lines)
						     "\n")) x))
				    (option '(#\o "output-dir") #t #f
					    (lambda (o n a x)
					      (set! *output-dir* a) x))
				    (option '(#\v "version") #f #f
					    (lambda (o n a x)
					      (print "Version " *wrap-version*)
					      (exit 0))))
			      (lambda (o n a x)
				(help)
				(error "Unrecognised option" n))
			      cons
			      '())))
    (if (or (null? filenames)
	    (not *output-dir*))
	(help)
	(begin
	  (for-each input-file filenames)
	  (emit-objects)
	  (emit-interfaces)
	  (emit-classes-main)
	  (emit-boxeds)
	  (emit-enums)
	  (emit-flags)
	  (emit-functions)))))

(main (command-line-arguments))
(exit 0)
