;;; oracle.scm - Simple Oracle glue for Chicken. Use in conjunction
;;; with DB.scm (an implementation of Kiselyov's database interface)
;;; and oracleglue.h/oracleglue.pc.
;
; Copyright (c) 2002 Tony Garnock-Jones
;
; Permission is hereby granted, free of charge, to any person
; obtaining a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction,
; including without limitation the rights to use, copy, modify, merge,
; publish, distribute, sublicense, and/or sell copies of the Software,
; and to permit persons to whom the Software is furnished to do so,
; subject to the following conditions:
; 
; The above copyright notice and this permission notice shall be
; included in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
; BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
; ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
; CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.

;---------------------------------------------------------------------------

(require 'db)

(declare
 (export
  DB:oracle-connect
 )

 (usual-integrations)
 (foreign-declare #<<EOF
#include "oracleglue.h"
EOF
))

(define-foreign-variable column-i integer "C_oracleglue_column.i")
(define-foreign-variable column-f float "C_oracleglue_column.f")
(define-foreign-variable column-s-len int "C_oracleglue_column.s.len")
(define-foreign-variable column-s-ptr c-string "C_oracleglue_column.s.ptr")

(define setup-oracle (foreign-lambda int "C_oracleglue_setup"))
(define oracle-connect (foreign-lambda int "C_oracleglue_connect" c-string))
(define get-sqlcode (foreign-lambda int "C_oracleglue_get_sqlcode"))
(define get-error-string (foreign-lambda c-string "C_oracleglue_get_error_string"))
(define begin-query (foreign-lambda c-string "C_oracleglue_begin_query" c-string))
(define start-fetch (foreign-lambda c-string "C_oracleglue_start_fetch"))
(define fetch-column (foreign-lambda int "C_oracleglue_fetch_column"))
(define end-query (foreign-lambda int "C_oracleglue_end_query"))

; (DB:oracle-connect userpass-string)
(define DB:oracle-connect
  (begin
    (define oracle-initialised? #f)
    (define (maybe-initialise)
      (if (and (not oracle-initialised?)
	       (zero? (setup-oracle)))
	  (set! oracle-initialised? #t))
      oracle-initialised?)

    (define (construct-error message)
      (list #f message (get-sqlcode) (get-error-string)))

    (define end-of-results? #t)

    (define (handler selector . args)
      (case selector
	((close)) ; no-op

	((begin-query)
	 (let ((stmt (car args)))
	   (set! end-of-results? #t)
	   (cond
	    ((begin-query stmt) => construct-error)
	    ((start-fetch) => construct-error)
	    (else
	     (set! end-of-results? #f)
	     (cons #t 'result-set-marker)))))

	((fetch-row)
	 (if end-of-results?
	     #f
	     (let collect-columns ((tail '()))
	       (case (fetch-column)
		 ((-1) (construct-error "oracle:fetch-column-error"))
		 ((0)
		  (set! end-of-results? #t)
		  #f)
		 ((1) (collect-columns (cons '() tail)))
		 ((2) (collect-columns (cons column-i tail)))
		 ((3) (collect-columns (cons column-f tail)))
		 ((4) (collect-columns (cons (substring column-s-ptr 0 column-s-len) tail)))
		 ((5) (cons #t tail))
		 (else (construct-error "oracle:unknown-fetch-column-result"))))))

	((end-query)
	 (if (not end-of-results?)
	     (begin
	       (set! end-of-results? #t)
	       (end-query))
	     #f))

	((imperative-stmt)
	 (let ((stmt (car args)))
	   (set! end-of-results? #t)
	   (cond
	    ((begin-query stmt) => construct-error)
	    (else
	     (end-query)
	     (cons #t '())))))

	(else
	 (error "DB:oracle-connect handler does not understand" selector args))))

    (lambda (userpass)
      (and (maybe-initialise)
	   (zero? (oracle-connect userpass))
	   handler))))
