;;; multiplex.scm - Simple select-and-thread-based multiplexing for Chicken
;
; Copyright (c) 2002 Tony Garnock-Jones
;
; Permission is hereby granted, free of charge, to any person
; obtaining a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction,
; including without limitation the rights to use, copy, modify, merge,
; publish, distribute, sublicense, and/or sell copies of the Software,
; and to permit persons to whom the Software is furnished to do so,
; subject to the following conditions:
; 
; The above copyright notice and this permission notice shall be
; included in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
; BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
; ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
; CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.

;---------------------------------------------------------------------------
; Bugs:
;   - doesn't support timeouts on I/O.

(declare
 (uses srfi-1 srfi-18 posix extras lolevel)

 (export make-empty-mailbox
	 mailbox-name
	 mailbox-get!
	 mailbox-put!

	 current-mailbox

	 make-multiplexor
	 multiplexor-stop!
	 multiplexor-blocker

	 multiplexor:setup)

 (usual-integrations)

 (foreign-declare #<<EOF

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <math.h>

EOF
))

(define-constant poll-interval 0.02)

(define-record mailbox name mutex waiters messages)

(define-record-printer (mailbox box out)
  (fprintf out "#<mailbox: ~a ~a ~a>"
	   (mailbox-name box)
	   (map cdr (queue->list (mailbox-waiters box)))
	   (queue->list (mailbox-messages box))))

(define (make-empty-mailbox . name)
  (make-mailbox (:optional name (gensym "mailbox"))
		(make-mutex)
		(make-queue)
		(make-queue)))

(define (mailbox-put! box msg)
  (let ((mutex (mailbox-mutex box)))
    (mutex-lock! mutex)
    (if (queue-empty? (mailbox-waiters box))
	(queue-add! (mailbox-messages box) msg)
	(let* ((waiter (queue-remove! (mailbox-waiters box))))
	  (set-car! waiter msg)
	  (thread-resume! (cdr waiter))))
    (mutex-unlock! mutex)))

(define (mailbox-get! box . maybe-block-handler)
  (let ((mutex (mailbox-mutex box))
	(block-handler (:optional maybe-block-handler #f)))
    (mutex-lock! mutex)
    (if (queue-empty? (mailbox-messages box))
	(if block-handler
	    (begin
	      (mutex-unlock! mutex)
	      (block-handler box)
	      (mailbox-get! box block-handler))
	    (let* ((ct (current-thread))
		   (waiter (cons #f ct)))
	      (queue-add! (mailbox-waiters box) waiter)
	      ; can't use thread-suspend! because it won't unlock our
	      ; lock before blocking us - and can't unlock first, as
	      ; we get a race (a putter can thread-resume! us before
	      ; we've thread-suspend!ed).
	      (##sys#setslot ct 3 'suspended)
	      (mutex-unlock! mutex) ; mutex-unlock! calls ##sys#schedule.
	      ; when we get here, we've been thread-resume!d
	      (car waiter)))
	(let ((message (queue-remove! (mailbox-messages box))))
	  (mutex-unlock! mutex)
	  message))))

(define current-mailbox-parameter (make-parameter #f))
(define (current-mailbox)
  (let ((box (current-mailbox-parameter)))
    (unless (and box
		 (eq? (mailbox-name box) (current-thread)))
      (set! box (make-empty-mailbox (current-thread)))
      (current-mailbox-parameter box))
    box))

(define (make-multiplexor)
  (let* ((name (gensym "multiplexor"))
	 (mailbox (make-empty-mailbox name))
	 (readers '())
	 (writers '())
	 (maxfd -1)
	 (fdsets ((foreign-lambda* c-pointer ()
				   "return(calloc(4, sizeof(fd_set)));")))
	 (running #t))

    ;;-- Methods

    (define (select)
      ((foreign-lambda* int ((int n) (c-pointer ptr) (double timeout))
			"fd_set *in_r = ((fd_set *) ptr) + 0;"
			"fd_set *in_w = ((fd_set *) ptr) + 1;"
			"fd_set *out_r = ((fd_set *) ptr) + 2;"
			"fd_set *out_w = ((fd_set *) ptr) + 3;"
			"struct timeval tv;"
			"memcpy(out_r, in_r, sizeof(fd_set));"
			"memcpy(out_w, in_w, sizeof(fd_set));"
			"tv.tv_sec = (time_t) timeout;"
			"tv.tv_usec = (long) fmod(timeout * 1000000, 1000000);"
			"return(select(n, out_r, out_w, NULL, &tv));")
       (fx+ maxfd 1) fdsets poll-interval))

    (define (fdset-clear! index)
      ((foreign-lambda* void ((c-pointer ptr) (int index))
			"FD_ZERO(((fd_set *) ptr) + index);")
       fdsets index))

    (define (set-bit! index fd)
      ((foreign-lambda* void ((c-pointer ptr) (int index) (int fd))
			"FD_SET(fd, (((fd_set *) ptr) + index));")
       fdsets index fd))

    (define (clear-bit! index fd)
      ((foreign-lambda* void ((c-pointer ptr) (int index) (int fd))
			"FD_CLR(fd, (((fd_set *) ptr) + index));")
       fdsets index fd))

    (define (is-bit-set? index fd)
      ((foreign-lambda* bool ((c-pointer ptr) (int index) (int fd))
			"return( FD_ISSET(fd, (((fd_set *) ptr) + index)) );")
       fdsets index fd))

    (define (reset-maxfd!)
      (set! maxfd -1))

    (define (bump-maxfd! fd)
      (if (> fd maxfd)
	  (set! maxfd fd))
      maxfd)

    (define (handle-message msg)
      (let ((kind (car msg))
	    (data (cdr msg)))
	(case kind
	  ((command)
	   (let ((command (first data)))
	     (case command
	       ((stop) (set! running #f)))))
	  ((i/o)
	   (let ((action (first data))
		 (mbox (second data))
		 (fd (third data)))
	     (case action
	       ((accepting reading)
		(set! readers (cons (cons fd mbox) readers))
		(set-bit! 0 fd)
		(bump-maxfd! fd))
	       ((connecting writing)
		(set! writers (cons (cons fd mbox) writers))
		(set-bit! 1 fd)
		(bump-maxfd! fd))
	       (else (mailbox-put! mbox (list 'error
					      "multiplex.scm: Unknown reason received from blocker"
					      action
					      fd))))))
	  (else 'ignore-bad-message))))

    (define (awaken event index lst)
      (remove (lambda (entry)
		(let* ((fd (car entry))
		       (mbox (cdr entry))
		       (result (is-bit-set? (+ index 2) fd)))
		  (when result
			(mailbox-put! mbox (list 'ok event fd))
			(clear-bit! index fd))
		  result))
	      lst))

    (define (wait-for-event box)
      (when (and (null? readers)
		 (null? writers))
	    (reset-maxfd!)
	    (fdset-clear! 0)
	    (fdset-clear! 1))
      (case (select)
	((-1) ; error
	 (##sys#update-errno)
	 (if (not (= (errno) errno/intr))
	     (error "multiplex.scm: Select failed" (errno))))
	((0)) ; timeout
	(else ; i/o
	 (set! readers (awaken 'read 0 readers))
	 (set! writers (awaken 'write 1 writers)))))

    ;;-- Initialisation

    (for-each fdset-clear! (iota 4))
    (set-finalizer! fdsets free)

    ;;-- Start workloop thread

    (thread-start!
     (make-thread
      (lambda ()
	(let loop ()
	  (when running
		(handle-message (mailbox-get! mailbox wait-for-event))
		(loop))))
      name))

    ;;-- Return the mailbox
    mailbox))

(define (multiplexor-stop! plex)
  (mailbox-put! plex (list 'command 'stop)))

(define (multiplexor-blocker plex)
  (lambda (action . args)
    (let ((mbox (current-mailbox)))
      (mailbox-put! plex (cons* 'i/o action mbox args))
      (let* ((msg (mailbox-get! mbox))
	     (result (car msg))
	     (data (cdr msg)))
	(case result
	  ((ok) data)
	  ((error) (apply error data))
	  (else (error "multiplex.scm: Unknown response message from multiplexor" msg)))))))

(define (multiplexor:setup style)
  (let ((plex (make-multiplexor)))
    (case style
      ((raw)
       (require 'socket))
      ((plt-net)
       (require 'plt-net)
       (plt-net:set-blocker! (multiplexor-blocker plex)))
      (else
       (error "multiplex.scm: multiplexor:setup: unknown style" style)))
    (set-signal-handler! signal/pipe #f)
    (set-signal-handler! signal/int (lambda (n) (multiplexor-stop! plex)))
    plex))
