;;; plt-net.scm - PLT compatible networking routines for Chicken
;
; Copyright (c) 2002 Tony Garnock-Jones
;
; Permission is hereby granted, free of charge, to any person
; obtaining a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction,
; including without limitation the rights to use, copy, modify, merge,
; publish, distribute, sublicense, and/or sell copies of the Software,
; and to permit persons to whom the Software is furnished to do so,
; subject to the following conditions:
; 
; The above copyright notice and this permission notice shall be
; included in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
; BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
; ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
; CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.

;---------------------------------------------------------------------------
; Specification/documentation lifted verbatim from MzScheme networking
; documentation at:
;
;   http://download.plt-scheme.org/doc/200alpha12/html/mzscheme/mzscheme-Z-H-10.html#%_sec_10.3
;
; The documentation as it stands refers to some features of MzScheme
; that are not implemented by Chicken. Basically these routines don't
; aim to duplicate the entire MzScheme environment; they only provide
; limited compatibility.

(declare
 (export tcp-listener?
	 tcp-listen
	 tcp-connect
	 tcp-connect/enable-break	; not implemented
	 tcp-accept
	 tcp-accept-ready?
	 tcp-accept/enable-break	; not implemented
	 tcp-close
	 tcp-addresses			; not implemented

	 plt-net:get-blocker
	 plt-net:set-blocker!
 )
)

(require 'socket)

(define plt-net:multiplexor-blocker #f)
(define (plt-net:get-blocker)
  plt-net:multiplexor-blocker)
(define (plt-net:set-blocker! x)
  (set! plt-net:multiplexor-blocker x))

; Helpers.

(define-record tcp-listener
  fd            ; file descriptor
  cache         ; either #f or a waiting connection
)

(define-record-printer (tcp-listener t out)
  (display "#<tcp-listener:" out)
  (display (tcp-listener-fd t) out)
  (display ">" out))

(define (PLT-NET:create-tcp-listener fd)
  (make-tcp-listener fd #f))

(define (tcp-listener-pop-cache! tcpl)
  (let ((result (tcp-listener-cache tcpl)))
    (tcp-listener-cache-set! tcpl #f)
    result))

(define (tcp-listener-cache-ready? tcpl)
  (not (eq? #f (tcp-listener-cache tcpl))))

(define (tcp-listener-closed? tcpl)
  (not (tcp-listener-fd tcpl)))

; (tcp-listen port-k [max-allow-wait-k reuse?]) creates a listening
; server on the local machine at the specified port number (where
; port-k is an exact integer between 1 and 65535). The
; max-allow-wait-k argument determines the maximum number of client
; connections that can be waiting for acceptance. (When
; max-allow-wait-k clients are waiting acceptance, no new client
; connections can be made.) The default value for max-allow-wait-k
; argument is 4.
;
; If the reuse? argument is true, then tcp-listen will create a
; listener even if the port is involved in a TIME_WAIT state. Such a
; use of reuse? defeats certain guarantees of the TCP protocol; see
; Stevens's book for details. The default for reuse? is #f.
;
; The return value of tcp-listen is a TCP listener value. This value
; can be used in future calls to tcp-accept, tcp-accept-ready?, and
; tcp-close. Each new TCP listener value is placed into the management
; of the current custodian (see section 7.5).
;
; If the server cannot be started by tcp-listen, the exn:i/o:tcp
; exception is raised.

(define (tcp-listen port . rest)
  (let-optionals rest ((max-allow-wait 4)
		       (reuse? #f))
    (let ((tcpl (PLT-NET:create-tcp-listener
		 (make-server-socket (make-sockaddr_in port #f)
				     #:suspender plt-net:multiplexor-blocker
				     #:reuse? reuse?
				     #:max-allow-wait max-allow-wait))))
      (if (tcp-listener-closed? tcpl)
	  (raise 'exn:i/o:tcp)
	  tcpl))))

; (tcp-connect hostname-string [port-k]) attempts to connect as a
; client to a listening server. The hostname-string argument is the
; server host's internet address name25 (e.g., "cs.rice.edu"), and
; port-k (an exact integer between 1 and 65535) is the port where the
; server is listening.
;
; Two values (see section 2.2) are returned by tcp-connect: an input
; port and an output port. Data can be received from the server
; through the input port and sent to the server through the output
; port. If the server is a MzScheme process, it can obtain ports to
; communicate to the client with tcp-accept. These ports are placed
; into the management of the current custodian (see section 7.5).
;
; If a connection cannot be established by tcp-connect, the
; exn:i/o:tcp exception is raised.

(define (tcp-connect hostname port)
  (let ((ports (open-tcp-socket hostname port plt-net:multiplexor-blocker)))
    (if ports
	(apply values ports)
	(raise 'exn:i/o:tcp))))

; (tcp-connect/enable-break hostname-string [port-k]) is like
; tcp-connect, but breaking is enabled (see section 6.6) while trying
; to connect. If breaking is disabled when tcp-connect/enable-break is
; called, then either ports are returned or exn:break exception is
; raised, but not both.
;
; UNIMPLEMENTED: "breaking" not supported

(define tcp-connect/enable-break tcp-connect)

; (tcp-accept tcp-listener) accepts a client connection for the server
; associated with tcp-listener. The tcp-listener argument is a TCP
; listener value returned by tcp-listen. If no client connection is
; waiting on the listening port, the call to tcp-accept will
; block. (See also tcp-accept-ready?, below.)
;
; Two values (see section 2.2) are returned by tcp-accept: an input
; port and an output port. Data can be received from the client
; through the input port and sent to the client through the output
; port. These ports are placed into the management of the current
; custodian (see section 7.5).
; 
; If a connection cannot be accepted by tcp-accept, or if the listener
; has been closed, the exn:i/o:tcp exception is raised.

(define (tcp-accept tcp-listener)
  (if (or (not (tcp-listener? tcp-listener))
	  (tcp-listener-closed? tcp-listener))
      (raise 'exn:i/o:tcp))
  (let ((result (or (tcp-listener-pop-cache! tcp-listener)
		    (socket:accept-ports (tcp-listener-fd tcp-listener)
					 plt-net:multiplexor-blocker))))
    (if (not result)
	(raise 'exn:i/o:tcp)
	(values (first result)
		(second result)))))

; (tcp-accept-ready? tcp-listener) tests whether an unaccepted client
; has connected to the server associated with tcp-listener. The
; tcp-listener argument is a TCP listener value returned by
; tcp-listen. If a client is waiting, the return value is #t,
; otherwise it is #f. A client is accepted with the tcp-accept
; procedure, which returns ports for communicating with the client and
; removes the client from the list of unaccepted clients.
;
; If the listener has been closed, the exn:i/o:tcp exception is
; raised.

(define (tcp-accept-ready? tcp-listener)
  (if (tcp-listener-closed? tcp-listener)
      (raise 'exn:i/o:tcp))
  (or (tcp-listener-cache-ready? tcp-listener)
      (call-with-current-continuation
       (lambda (escape)
	 (let ((fd (tcp-listener-fd tcp-listener)))
	   (unless plt-net:multiplexor-blocker
	     (socket:set-nonblocking! fd #t)
	     (set! escape
		   (let ((e escape))
		     (lambda (v)
		       (socket-set-nonblocking! fd #f)
		       (e v)))))
	   (let ((result (socket:accept-ports fd
					      (lambda (what fd)
						(escape #f)))))
	     (if (not result)
		 (raise 'exn:i/o:tcp))
	     (tcp-listener-cache-set! tcp-listener result)
	     (escape #t)))))))

; (tcp-accept/enable-break tcp-listener) is like tcp-accept, but
; breaking is enabled (see section 6.6) while trying to accept a
; connection. If breaking is disabled when tcp-accept/enable-break is
; called, then either ports are returned or exn:break exception is
; raised, but not both.
;
; UNIMPLEMENTED: "breaking" not supported

(define tcp-accept/enable-break tcp-accept)

; (tcp-close tcp-listener) shuts down the server associated with
; tcp-listener. The tcp-listener argument is a TCP listener value
; returned by tcp-listen. All unaccepted clients receive an
; end-of-file from the server; connections to accepted clients are
; unaffected.
;
; If the listener has already been closed, the exn:i/o:tcp exception
; is raised.
;
; The listener's port number may not become immediately available for
; new listeners (with the default reuse? argument of tcp-listen). For
; further informtation, see Stevens's explanation of the TIME_WAIT TCP
; state.

(define (tcp-close tcp-listener)
  (if (tcp-listener-closed? tcp-listener)
      (raise 'exn:i/o:tcp))
  (let ((fd (tcp-listener-fd tcp-listener)))
    (tcp-listener-fd-set! tcp-listener #f)
    (if (not (socket:close fd))
	(raise 'exn:i/o:tcp))))

; (tcp-listener? v) returns #t if v is a TCP listener value created by
; tcp-listen, #f otherwise.
;
; IMPLICITLY DEFINED by define-record, above.

; (tcp-addresses tcp-port) returns two strings. The first string is
; the internet address for the local machine a viewed by the given TCP
; port's connection.26 The second string is the internet address for
; the other end of the connection.
;
; If the given port has been closed, the exn:i/o:tcp exception is
; raised.
;
; UNIMPLEMENTED: current socket framework does not allow for this.

(define (tcp-addresses tcp-port)
  (raise 'exn:i/o:tcp))
