#!/usr/local/bin/csi -script

(declare (uses srfi-1 srfi-13 script-utils library posix extras))

(eval-when (load) ; no need to include macros if were running within csi.
  (include "moremacros"))

(define-record goal pattern handler)
(define-record action target satisfied?-thunk thunk subgoal-actions)

(define (action-satisfied? action)
  (let ((r ((action-satisfied?-thunk action))))
    (action-satisfied?-thunk-set! action (if r (lambda () #t) (lambda () #f)))
    r))

(define *build-verbose* (make-parameter 0))
(define *build-depth* 0)
(define *build-db* 'dummy)
(define *build-stack* 'dummy)
(define *build-goals* 'dummy)
(define *build-failed-goals* 'dummy)

(define (*build-comment* level . args)
  (if (>= (*build-verbose*) level)
      (begin
	(printf "chicken-make[~a]: " *build-depth*)
	(if (> level 0) (printf "(~a) " level))
	(apply print args))))

(define (build:check-goal-type goal)
  (cond
   ((symbol? goal) (symbol->string goal))
   ((string? goal) goal)
   (else (error "Goal must be either string or symbol:" goal))))

(define (register-goal! goal)
  (set! *build-db* (cons goal *build-db*)))

(define build:subgoals->actions
  (begin
    (define (subgoal->action goal)
      (fluid-let ((*build-stack* (cons goal *build-stack*)))
	(*build-comment* 3 "build-stack: " (reverse *build-stack*))
	(or (ormap (lambda (rule)
		     (cond
		      ((string-match (goal-pattern rule) goal) => (goal-handler rule)) ; action/#f
		      (else #f)))
		   *build-db*)
	    (begin
	      (set! *build-failed-goals* (cons *build-stack* *build-failed-goals*))
	      #f)))) ; can't satisfy one of our subgoals -> failure
    (lambda (goals)
      (call-with-current-continuation
       (lambda (return)
	 (map (lambda (goal)
		(if (member goal *build-stack*)
		    (error "Cyclic dependency:" (reverse *build-stack*))
		    (or (subgoal->action goal)
			(return #f))))
	      (map build:check-goal-type goals)))))))

(define-macro (define-goal pattern varnames subgoals satisfiable? satisfied? . satisfy!)
  (let ((pattern-sym (gensym))
	(result-sym (gensym)))
    `(let ((,pattern-sym ,pattern))
       (register-goal!
	(make-goal ,pattern-sym
		   (lambda (,result-sym)
		     (if (not (= (length ,result-sym) (length ',varnames)))
			 (error "Wrong number of matched groups in goal:"
				,pattern-sym
				,result-sym
				',varnames))
		     (apply
		      (lambda ,varnames
			(let* ,subgoals
			  (let ((*build-subgoals* (filter (lambda (x) (not (null? x)))
							  (flatten ,@(map car subgoals)))))
			    (let ((*build-actions* (build:subgoals->actions *build-subgoals*)))
			      (and *build-actions*
				   ,satisfiable?
				   (make-action ,(car varnames)
						(lambda () ,satisfied?)
						(lambda () (and ,@satisfy!))
						*build-actions*))))))
		      ,result-sym)))))))

;---------------------------------------------------------------------------

(define (build:file-mtime filename)
  (and (file-exists? filename)
       (file-modification-time filename)))

(define (up-to-date? goal . dependencies)
  (let ((goal-time (build:file-mtime goal))
	(dep-times (map build:file-mtime dependencies)))
    (and goal-time
	 (not (find (lambda (dep-time) (or (not dep-time)
					   (< goal-time dep-time)))
		    dep-times)))))

(define-macro (define-file-goal pattern varnames subgoals . satisfy!)
  `(define-goal ,pattern ,varnames ,subgoals
     #t
     (and (apply up-to-date? ,(car varnames) *build-subgoals*)
	  (andmap action-satisfied? *build-actions*))
     ,@satisfy!))

;---------------------------------------------------------------------------

(define (spawn verbosity command . args)
  (let ((command-line (string-join (map (lambda (elt)
					  (cond
					   ((string? elt) elt)
					   ((symbol? elt) (symbol->string elt))
					   (else
					    (with-output-to-string
					      (lambda () (write elt))))))
					(cons command args))
				   " ")))
    (if (zero? verbosity)
	(print command-line)
	(*build-comment* verbosity command-line))
    (zero? (system command-line))))

;---------------------------------------------------------------------------

(define (build:define-default-goals!)
  (register-goal! (make-goal ".*"
			     (lambda (results)
			       (and (file-exists? (car results))
				    (make-action (car results)
						 (lambda () #t) ; satisfied?
						 (lambda () #t) ; thunk
						 '()))))))

(define (build:achieve-goal! goal)
  (set! *build-failed-goals* '())
  (let ((actions (build:subgoals->actions (list goal))))
    (if (not actions)
	(begin
	  (*build-comment* 0 "Could not achieve goal: " goal)
	  (print "Unattainable goals:")
	  (for-each (lambda (l) (print "\t" l)) (reverse *build-failed-goals*))
	  #f)
	(call-with-current-continuation
	 (lambda (return)
	   (let ((stack '())
		 (top #t))
	     (let execute-action ((action (car actions)))
	       (let ((target (action-target action)))
		 (*build-comment* 3 "Current goal: " target)
		 (if (action-satisfied? action)
		     (*build-comment* (if top 0 1) target " is up to date.")
		     (fluid-let ((stack (cons target stack))
				 (top #f))
		       (for-each execute-action (action-subgoal-actions action))
		       (*build-comment* 1 "Goal stack: " stack)
		       (if (not ((action-thunk action)))
			   (begin
			     (*build-comment* 0
					      "Action for " (action-target action)
					      " failed; supergoal " (action-target (car actions)))
			     (return #f)))))
		 (action-satisfied?-thunk-set! action (lambda () #t)))))
	   #t)))))

(define (build:main . argv)
  (fluid-let ((*build-depth* (+ *build-depth* 1))
	      (*build-db* '())
	      (*build-stack* '())
	      (*build-goals* (make-parameter '()))
	      (*build-failed-goals* '()))
    (let ((filename "Buildfile")
	  (goals '())
	  (new-directory #f)
	  (old-directory #f))
      (receive ignore (args-fold argv
				 (list (option '(#\C "change-dir") #t #f
					       (lambda (opt name arg . seeds)
						 (set! new-directory arg)))
				       (option '(#\f "file") #t #f
					       (lambda (opt name arg . seeds)
						 (set! filename arg)))
				       (option '(#\v "verbose") #f #f
					       (lambda (opt name arg . seeds)
						 (*build-verbose* (+ (*build-verbose*) 1)))))
				 (lambda (opt name arg . seeds)
				   (error "Unrecognised option:" name))
				 (lambda (goal . seeds)
				   (set! goals (cons goal goals))))
	       'dummy)

      (when new-directory
	    (set! old-directory (current-directory))
	    (fluid-let ((*build-depth* (- *build-depth* 1)))
	      (*build-comment* 0 "Entering directory " new-directory))
	    (change-directory new-directory))

      (*build-comment* 2 "Loading buildfile \"" filename "\"...")
      (load filename)

      ; Now *build-db* is in the wrong order (intentionally). Put some
      ; "default" goals on the front, and then reverse the list, to
      ; make the front the back.
      (build:define-default-goals!)
      (set! *build-db* (reverse *build-db*))

      (set! goals (reverse goals))
      (if (null? goals)
	  (set! goals (*build-goals*)))

      (let ((result (andmap build:achieve-goal! goals)))
	(when new-directory
	      (fluid-let ((*build-depth* (- *build-depth* 1)))
		(*build-comment* 0 "Leaving directory " new-directory))
	      (change-directory old-directory))
	result))))

; argv is ("/usr/local/bin/csi" "-script-meta"
; "/path/to/package-builder" ...) when running in the interpreter, or
; ("/path/to/package-builder" ...) when running compiled.
(eval-when (load) (apply build:main (cdr (argv))))
(eval-when (eval) (apply build:main (cdddr (argv))))
