;-*- scheme -*-
;;@title "sxml2xml: pseudo-SXML to XML text"

;;@sxml
;; (para (cmdsynopsis
;;        (command "sxml2xml")
;;        (arg (@ (rep "repeat")
;; 	       (choice "plain"))
;; 	    (replaceable "filename")))
;;       "Reads a file, an S-expression at a time, evaluating each
;; expression. Each expression is expected to yield a pseudo-SXML
;; fragment, which is then converted to plain XML text, and printed on
;; stdout.
;; ")

(require 'srfi-1 'srfi-13 'extras)

;;@section "Internal datatypes and procedures"
;; These subroutines are also available to code loaded as part of the
;; SXML-to-XML conversion process.

;;@ Reads a file, an S-expression at a time, evaluating each
;;expression, and collecting the results into a list of SXML
;;fragments.
(define (include-sxml . filenames)
  (append-map (lambda (filename)
		(with-input-from-file filename
		  (lambda ()
		    (reverse
		     (fold (lambda (x acc) (cons (eval x) acc))
			   '()
			   (read-file))))))
	      filenames))

(let ()
  (define (make-escaping-printer alist)
    (lambda (orig)
      (string-fold
       (lambda (ch acc)
	 (cond
	  ((assv ch alist) => (compose display cdr))
	  (else (display ch))))
       '()
       orig)))

  (define xml-escaping-printer (make-escaping-printer '((#\< . "&lt;")
							(#\> . "&gt;")
							(#\& . "&amp;") 
							(#\" . "&quot;")
							(#\' . "&apos;"))))

  (define (print-attrs alist)
    (for-each (lambda (p)
		(print* " " (car p) "='")
		(xml-escaping-printer (with-output-to-string
					(lambda ()
					  (display (cadr p)))))
		(print* "'"))
	      alist))

  (define (default-printer tag attrs body)
    (print* "<" tag)
    (print-attrs attrs)
    (if (null? body)
	(print* "/>")
	(begin
	  (print* ">")
	  (map convert body)
	  (print* "</" tag ">"))))

  (define (print-pi tag attrs)
    (print* "<?" tag)
    (print-attrs attrs)
    (print "?>"))

  (define (print-external-id x)
    (case (car x)
      ((public) (print* "PUBLIC \"" (cadr x) "\" \"" (caddr x) "\""))
      ((system) (print* "SYSTEM \"" (cadr x) "\""))
      (else (error "Unknown external-id kind" x))))

  (define (print-PEDef def)
    (if (string? def)
	(print* def)
	(print-external-id def)))

  (define (print-entity-def body)
    (if (eq? (car body) '%)
	(begin
	  (print* "<!ENTITY % " (cadr body) " ")
	  (print-PEDef (caddr body)))
	(begin
	  (print* "<!ENTITY " (car body) " ")
	  (print-PEDef (cadr body))))
    (print ">"))

  (define (print-internal-dtd body0)
    (print " [")
    (for-each (lambda (x)
		(assert pair? x)
		(let ((tag (car x))
		      (body (cdr x)))
		  (case tag
		    ((*entity*) (print-entity-def body))
		    ((*literal*) (for-each display body))
		    (else (error "Unsupported internal-dtd clause" x)))))
	      body0)
    (print "]>"))

  (define (print-doctype basetag decltype body)
    (print* "<!DOCTYPE " basetag " ")
    (print-external-id decltype)
    (print-internal-dtd body))

  (define (print-entity-ref tag x)
    (print* tag x ";"))

  (define (convert x)
    (cond
     ((pair? x)
      (let* ((tag (car x))
	     (attrs* (and (pair? (cdr x))
			  (let ((a (cadr x)))
			    (and (pair? a)
				 (eq? (car a) '@)
				 (cdr a)))))
	     (body (if attrs*
		       (cddr x)
		       (cdr x)))
	     (attrs (or attrs* '())))
	(if (not (or (symbol? tag)
		     (string? tag)))
	    (error "Tag must be string or symbol" tag))
	(case tag
	  ((& %)	(print-entity-ref tag (car body)))
	  ((*literal*)	(for-each display (cdr x)))
	  ((*pi*)	(print-pi (car body) (cdr body)))
	  ((*doctype*)	(print-doctype (car body) (cadr body) (cddr body)))
	  (else		(default-printer tag attrs body)))))
     ((string? x)
      (xml-escaping-printer x))
     (else
      (xml-escaping-printer
       (with-output-to-string
	 (lambda ()
	   (display x)))))))

  (define (convert-file filename)
    (with-input-from-file filename
      (lambda ()
	(let loop ((x (read)))
	  (unless (eof-object? x)
	    (convert (eval x))
	    (loop (read))))))
    (newline))

  (for-each convert-file (command-line-arguments)))
